/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

log.info("Default type:" + type("Default"))

import com.simsilica.crig.es.*;
import mythruna.assembly.*;

// 1 block in item scale is 3.125 cm.
standardItemScale = 0.25 * 0.25 * 0.5;

createType("BaseObject").with {

    addGetter("name") { ->
        //log.info("target:" + target);
        //log.info("delegate:" + delegate);
        def on = delegate[ObjectName];
        //log.info("name test:" + on);
        //log.info("delegate test:" + delegate[ObjectName]);
        if( on != null ) {
            return on.getName(entityData);
        }
        def n = delegate[Name];
        if( n != null ) {
            return n.getName();
        }
        return "Entity@" + delegate?.getId();
    }

    addAction("Look") { ->
        def lines = [];

        def name = object.name;
        lines += "Examine: " + name;

        def creator = object[CreatedBy]?.creatorId?.name?:"Unknown";

        def desc = object.type.getTypeVar("description", null);
        if( desc != null ) {
            // Bound to be a better groovy way to do this but so far
            // it only gets more complicated the deeper I drill.
            def s = desc.replaceAll("%creator", creator);
            s = s.replaceAll("%id", id as String);
            s = s.replaceAll("%name", name);
            lines += s;
        } else {
            lines += "ID:" + id;
            if( creator != null ) {
                lines += "Created by:" + creator;
            }
        }

        //lines += "Volume:" + object[ObjectVolume];

        session.showPrompt(target, PromptType.Dialog, lines.join("\n"));
        session.showOptions(target, []);

        return true;
    }

    addAction("move") { loc, facing ->
        //log.info("move:" + loc + ", " + facing + "  by:" + activator);
        // By default, we'll just let everything go wherever
        object << new SpawnPosition(loc, facing);
    }
}

def ignoredBlockTypes = [] as Set;
def blocksArray = BlockTypeIndex.getTypes();
for( int i = 0; i < blocksArray.length; i++ ) {
    BlockType type = blocksArray[i];
    if( type == null ) {
        continue;
    }
    String base = type.name.base;
    if( "magic-white".equals(base) ) {
        ignoredBlockTypes.add(i);
    } else if( "magic-red".equals(base) ) {
        ignoredBlockTypes.add(i);
    } else if( "magic-green".equals(base) ) {
        ignoredBlockTypes.add(i);
    } else if( "magic-blue".equals(base) ) {
        ignoredBlockTypes.add(i);
    } else if( "fire".equals(base) ) {
        ignoredBlockTypes.add(i);
    } else if( "flora".equals(base) ) {
        ignoredBlockTypes.add(i);
    }    
} 


// An object that can defer to the default hand-based manpulation
// drag, drop, activate, etc..
createType("ObjectManipulator").with {
    supertypes("BaseObject");

    // For now, we'll use the creator manipulator type but probably
    // there should be a different one that the creator manipulator extends.
    setTypeVar("manipulator", "Hand");
    
    setTypeVar("moveLimit", 0.75);

    addAction("dragStart") { dragged, loc, facing, click ->
        log.info("dragStart:" + dragged + ", " + loc + ", " + facing + ", " + click);

        env.setVar("draggedObject", dragged);

        // See if we are allowed to move this object
        def objPos = dragged[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canMoveObject() ) {
            env.setVar("objectMoveAllowed", false);
            echo("You do not have permission to move objects here.");
            return;
        }
        env.setVar("objectMoveAllowed", true);

        def offset;
        def shape = dragged[ShapeInfo];
        if( shape != null ) {
            def cells = shape.loadCells();
            if( cells ) {
                def size = cells.size;
                offset = size * 0.5 * shape.scale;
                offset.y = size.y * shape.scale; // to spare us a separate lookup later
            } else {
                offset = new Vec3d();
                offset.y = 1.0;
            }
        } else {
            // FIXME: lookup the alternate shape.
            // We don't use the offset for very much so right now in the the
            // limited use-cases where we don't have a shape (ClaimMarkers)
            // 0 is ok.
            offset = new Vec3d();
            offset.y = 1.0;
        }
        env.setVar("draggedOffset", offset);
    }

    addAction("drag") { dragged, loc, facing, click ->
        //log.info("drag:" + dragged + ", " + loc + ", " + facing + ", " + click);

        if( !env.getVar("objectMoveAllowed", false) ) {
            // Drag all you want, nothing is going to happen
            return;
        }

        // Figure out the proper y-up facing
        def xAxis = facing.mult(new Vec3d(1, 0, 0));
        double rads = Math.atan2(-xAxis.z, xAxis.x);
        facing = new Quatd().fromAngles(0, rads, 0);

        // Figure out where the bottom center is so that
        // we can project down for ground
        def center = loc;
        def offset = env.getVar("draggedOffset", null);
        def yOffset = 1.0; 
        if( offset ) {
            yOffset = offset.y;
            def localOffset = offset.clone();
            localOffset.y = 0;            
            facing.mult(localOffset, localOffset);
            center += localOffset;
        }
        
        // If the center is already in a block then don't let the movement happen
        // or at least try a higher pick offset
        // FIXME: make these physics queries instead
        int cellType = MaskUtils.getType(world.getWorldCell(center));
        if( cellType != 0 ) {
            // Give us a little more space
            yOffset += 1;
        }

        //log.info("loc:" + loc + "  center:" + center + "  yOffset:" + yOffset);
        // Check the ground
        // This probably goes away when we are doing real physics contact queries.
        def ray = new Rayd(center.add(0, yOffset, 0), new Vec3d(0, -1, 0));
        def hits = activator.pick(ray, 1);
        while( hits.hasNext() ) {
            def contact = hits.next();
            log.info("  contact:" + contact);            
            if( ignoredBlockTypes.contains(contact.type) ) {
                continue;
            }
            def hit = contact.point.y;
            log.info("  --> hit:" + hit);
            if( loc.y < hit ) {
                center.y += hit - loc.y;
                loc.y = hit;
                break;
            }
        }

        // FIXME: make these physics queries instead
        //cellType = MaskUtils.getType(world.getWorldCell(center));
        //if( cellType != 0 ) {
        //    return;
        //}
        // The above code seems unncessary and causes us not to be able to drag into
        // half-blocks, etc..  We still need a real physics query to handle object->object
        // collisions and so on.  Or even just proper object->world queries.       
        
        // Now that we know where we'll put it, we need to check to see
        // if it was dragged into an area that isn't allowed.
        if( !getPerms(activator, loc).canMoveObject() ) {
            echo("You do not have permission to move objects into that area.");
            return;
        }

        def objPos = dragged[SpawnPosition]?.location;
        if( objPos ) {
            double moveLimit = dragged.type.getTypeVar("moveLimit", 0.5); 
            // See if we're dragging too far
            if( loc.distance(objPos) > moveLimit ) {
                log.info("Uh oh.  Trying to move too much, from:" + objPos + " to:" + loc
                          + "Max dist:" + moveLimit + " actual:" + loc.distance(objPos));
                return; 
            }
        }
        
        // Base object has a move... so everything should have a move
        // that we will be trying to drag.
        dragged.run(env, "move", loc, facing);
        //dragged << new SpawnPosition(loc, facing);
    }

    addAction("dragEnd") { dragged ->
        log.info("dragEnd:" + dragged);

        // just clear the working env vars
        env.setVar("draggedObject", null);
        env.setVar("draggedOffset", null);
        env.setVar("objectMoveAllowed", null);
    }

    addAction("contextAction") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;

        log.info("contextAction:" + clicked + ", " + pos + " name:" + clicked?.name);

        // Here we want to let the player run some actions on the object

        // Assemble the action list
        def options = [];
        def clickedContext = clicked.context;
        log.info("Checking actions for type:" + clicked.type);        
        for( def action : clicked.type.contextActions ) {
            if( !action.canRun(env, clickedContext) ) {
                log.info("Skipping:" + action);
                continue;
            }
            options.add(new Option(action.getName()));
        }
        session.showPrompt(clicked, PromptType.List, clicked.name);
        session.showOptions(clicked, options); 
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);
    }

    addAction("altPressBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("altPressBlock:" + loc);
    }

    addAction("altClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("altClickBlock:" + loc);
    }

    addAction("mainPressBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("mainPressBlock:" + loc);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("mainClickBlock:" + loc);
    }

    addAction("mainPressObject") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;
        log.info("mainPressObject:" + clicked + "  pos:" + pos);
    }

    addAction("mainClickObject") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;
        log.info("mainClickObject:" + clicked + "  pos:" + pos);
    }

}

// For things that can be in inventory and equipped and stuff
createType("BaseItem").with {

    // Let all equippable items default to the hand manipulator with
    // default actions.  This will automatically let the user continue
    // to perform standard manipulation actions if the object doesn't
    // define its own.
    supertypes("ObjectManipulator");

    // For now, all objects will be equippable if they are in inventory
    addAction("Equip") { ->
        // This should be in the "can run" check
        // ...but we still need to check here as canRun() is speculative and "Equip"
        // may be called directly outside of those normal checks.
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            log.error("Activator:" + activator + " tried to equip foreign object:" + object);
            return;
        }

        // Right now there is only one hand "Holder"
        activator << Holding.create(object, object.type.manipulator?:"Default")
 
        // If it has an onEquip action then run that
        if( canRun("onEquip") ) {
            run("onEquip");
        }
        
    }.onlyIf { ->
        // Conditions for equipping:
        // -item is contained by the thing running the action
        // -item is not already equipped
        
        // Do the faster O(1) check first
        def holding = activator[Holding];
        if( object.isSameEntity(holding?.target) ) {
            return false;        
        }
         
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            return false;
        }
        
        return true;
    }

    addAction("Unequip") { ->
        // This should be in the "can run" check
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            log.error("Activator:" + activator + " tried to unequip foreign object:" + object);
            return;
        }

        // See if the current holder is holding us or not
        def holding = activator[Holding];
        log.info("Unequip holding:" + holding);
        if( !object.isSameEntity(holding?.target) ) {
            log.warn("Activator:" + activator + " trying to unequip item that is not equipped:" + object);
        }

        log.info("Unequpping:" + object);
        activator.remove(Holding);
        
        // If it has an onUnequip action then run that
        if( canRun("onUnequip") ) {
            run("onUnequip");
        }
    }.onlyIf { ->
        // Conditions for unequipping:
        // -item is equipped already
        // ...that's it.
        def holding = activator[Holding];
        return object.isSameEntity(holding?.target);        
    }
}

// For things that can be open and closed
createType("BaseCloseable").with {

    // Let all equippable items default to the hand manipulator with
    // default actions.  This will automatically let the user continue
    // to perform standard manipulation actions if the object doesn't
    // define its own.
    supertypes("BaseObject");

    addAction("Open") { ->
        // See if there is already a morph entity for this
        def filter = Morph.filter(resolveEntityId(object), "Open", entityData);
        def morph = findEntity(filter, Morph.class);
        
        if( morph == null ) {               
            morph = createEntity(Morph.create(resolveEntityId(object), "Open", 1, entityData));
        } else {
            morph << Morph.create(resolveEntityId(object), "Open", 1, entityData);
        }
        
    }.onlyIf { ->
        def shape = getBody(object)?.shape;
        if( !(shape instanceof AssemblyShape) ) {
            return false;
        } 
        return shape.findJoint("Open")?.getMix() == 0;
    }
    
    addAction("Close") { ->        
        // See if there is already a morph entity for this
        def filter = Morph.filter(resolveEntityId(object), "Open", entityData);
        def morph = findEntity(filter, Morph.class);
        
        if( morph == null ) {               
            morph = createEntity(Morph.create(resolveEntityId(object), "Open", 0, entityData));
        } else {
            morph << Morph.create(resolveEntityId(object), "Open", 0, entityData);
        }
    }.onlyIf { ->
        def shape = getBody(object)?.shape;
        if( !(shape instanceof AssemblyShape) ) {
            return false;
        } 
        return shape.findJoint("Open")?.getMix() > 0;
    }
}

// For now, a base type that can do all the things an assembly with joints can do
createType("BaseAssembly").with {

    // Let all equippable items default to the hand manipulator with
    // default actions.  This will automatically let the user continue
    // to perform standard manipulation actions if the object doesn't
    // define its own.
    supertypes("BaseCloseable");
}

type("Default").with {
    // By default, objects will have manipulator actions but not be equippable.
    // This is mostly because we let the empty hand be its own target when
    // it is not holding things... and equipping/unequipping a hand doesn't
    // seem logical.
    supertypes("ObjectManipulator");
}


createType("PlayerCharacter").with {

    // For when the player is bare-handed, "we" are the tool
    supertypes("ObjectManipulator");

    addAction("setSkinColor") { SkinColor skinColor ->
        log.info("setSkinColor:" + skinColor);
        object << skinColor;
    }

    addAction("setHairColor") { HairColor hairColor ->
        log.info("setHairColor:" + hairColor);
        object << hairColor;
    }
    
    addAction("deleteBlueprint") { EntityId blueprint ->
        log.info("Delete blueprint:" + blueprint);
        def bpInfo = blueprint[BlueprintInfo];
        if( bpInfo ) {
            log.info("bpInfo:" + bpInfo);
            // We should be able to "delete" it by setting the parent ID to null and
            // then it just won't appear anyway.  We'll still retain the blueprint
            // info for any object that might exist in the world. 
            blueprint << bpInfo.changeParent(null);
            echo("Deleted blueprint:" + bpInfo.name);
            return; 
        }
        def asmInfo = blueprint[AssemblyBlueprintInfo];
        if( asmInfo ) {
            log.info("asmInfo:" + asmInfo);
            blueprint << asmInfo.changeParent(null);
            echo("Deleted design:" + asmInfo.name);
            return;
        }
        log.warn("Error deleting unknown blueprint type for:" + blueprint);
        echo("Error deleting unknown blueprint type for:" + blueprint);
    } 
}
