/*
 *  Simple API for runtime dungeon generation.
 */

import org.slf4j.*;
import com.simsilica.mathd.*;
import com.simsilica.mblock.*;
import com.simsilica.blocked.*;
import com.google.common.collect.*;

//class GlobalLog {
//    static def log;
//}
//GlobalLog.log = log;

// ---------------------------------------------------------------------------
// Support classes
// ----------------------------------------------------------------------------


class Style {
    int floorType;
    int wallType;
    int roofType;
}

createBasicStyle = { int floorType, int roofType, int wallType ->
    return new Style(floorType:floorType, roofType:roofType, wallType:wallType);
}


class Room {
    static Logger log = LoggerFactory.getLogger(Room.class);

    Vec3i loc;
    Vec3i size;
    int type;
    Style style;

    public Room( Style style ) {
        this.style = style;
    }

    public Vec3i getCenter() {
        Vec3i result = loc.clone();
        result.x += size.x / 2;
        result.z += size.z / 2;
        return result;
    }

    public int getArea() {
        return size.x * size.z;
    }

    public void insert( Vec3i offset, def world, boolean cutawayView ) {

        int xs = 2 + size.x * 2;
        int ys = 2 + size.y;
        int zs = 2 + size.z * 2;

        int xMin = loc.x * 2 - 1;
        int yMin = loc.y - 1;
        int zMin = loc.z * 2 - 1;

        for( int x = 0; x < xs; x++ ) {
            for( int z = 0; z < zs; z++ ) {
                for( int y = 0; y < ys; y++ ) {
                    int block = 0;
                    if( y == 0 ) {
                        block = style.floorType;
                    } else if( y == ys - 1 ) {
                        if( cutawayView ) {
                            block = 0;
                        } else {
                            block = style.roofType;
                        }
                    } else if( x == 0 || z == 0 || x == xs - 1 || z == zs - 1 ) {
                        block = style.wallType;
                        if( cutawayView && y > 1 ) {
                            block = 0;
                        }
                    }
                    world.setWorldCell(new Vec3d(offset.x + x + xMin, offset.y + y + yMin, offset.z + z + zMin), block);
                }
            }
        }
    }

    public boolean intersects( def map ) {
        int x = loc.x - 1;
        int z = loc.z - 1;
        int xs = size.x + 2;
        int zs = size.z + 2;
        if( x + xs > map.length ) {
            xs = map.length - x;
        }
        if( z + zs > map.length ) {
            zs = map.length - z;
        }
        for( int i = 0; i < xs; i++ ) {
            for( int j = 0; j < zs; j++ ) {
                if( map[x + i][z + j] != 0 ) {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean intersectsRooms( def map ) {
        int x = loc.x - 1;
        int z = loc.z - 1;
        int xs = size.x + 2;
        int zs = size.z + 2;
        if( x + xs > map.length ) {
            xs = map.length - x;
        }
        if( z + zs > map.length ) {
            zs = map.length - z;
        }
        for( int i = 0; i < xs; i++ ) {
            for( int j = 0; j < zs; j++ ) {
                if( map[x + i][z + j] > 0 ) {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean intersectsRectangle( int x, int z, int xSize, int zSize ) {
        if( x + xSize <= loc.x ) {
            return false;
        }
        if( loc.x + size.x <= x ) {
            return false;
        }
        if( z + zSize <= loc.z ) {
            return false;
        }
        if( loc.z + size.z <= z ) {
            return false;
        }
        return true;
    }

    public boolean fill( def map, int val ) {
        int x = loc.x;
        int z = loc.z;
        int xs = size.x;// * 2;
        int zs = size.z;// * 2;
        for( int i = 0; i < xs; i++ ) {
            for( int j = 0; j < zs; j++ ) {
                map[x + i][z + j] = val;
            }
        }
        return false;
    }

    public double tunnelDistance( Room room ) {
        // calculate distance based on the ideal tunnel
        // distance... so more of a taxicab distance based on
        // the nearest points, etc..  We may not use these
        // actual connection points when we make the tunnels
        // 'for real' but it's a good starting point for
        // distance checks.
        //
        // If the rooms overlap on some axis, then a straight
        // shot from room to room is the best path.
        int r1xMin = loc.x;
        int r1zMin = loc.z;
        int r1xMax = loc.x + size.x;
        int r1zMax = loc.z + size.z;

        int r2xMin = room.loc.x;
        int r2zMin = room.loc.z;
        int r2xMax = room.loc.x + room.size.x;
        int r2zMax = room.loc.z + room.size.z;

        //log.info("distance r1:" + r1xMin + ", " + r1zMin + " -> " + r1xMax + ", " + r1zMax
        //          + "  r2:" + r2xMin + ", " + r2zMin + " -> " + r2xMax + ", " + r2zMax);

        if( (r1xMax > r2xMin && r1xMax <= r2xMax)
            || (r1xMin >= r2xMin && r1xMin < r2xMax)
            // It's possible for both of r1's min and max to be outside
            // of r2 if it envelopes it, so we have to check both ways.
            || (r2xMax > r1xMin && r2xMax <= r1xMax)
            || (r2xMin >= r1xMin && r2xMin < r1xMax) ) {
            // They overlap on the x-axis
            //log.info("overlap on x-axis");
            if( r1zMin < r2zMin ) {
                // Room one is less than room 2... distance will
                // be between r1 max and r2 min
                return r2zMin - r1zMax;
            } else {
                // Room two is less than room 1
                return r1zMin - r2zMax;
            }
        }
        if( (r1zMax > r2zMin && r1zMax <= r2zMax)
            || (r1zMin >= r2zMin && r1zMin < r2zMax)
            // It's possible for both of r1's min and max to be outside
            // of r2 if it envelopes it, so we have to check both ways.
            || (r2zMax > r1zMin && r2zMax <= r1zMax)
            || (r2zMin >= r1zMin && r2zMin < r1zMax) ) {
            // They overlap on the z-axis
            //log.info("overlap on z-axis");
            if( r1xMin < r2xMin ) {
                // Room one is less than room 2... distance will
                // be between r1 max and r2 min
                return r2xMin - r1xMax;
            } else {
                // Room two is less than room 1
                return r1xMin - r2xMax;
            }
        }

        int x1, z1, x2, z2;

        // Figure out which corners of the rooms we need to
        // connect.
        // Corner of room1
        if( r1xMin < r2xMin ) {
            x1 = r1xMax-1;
            if( r1zMin < r2zMin ) {
                z1 = r1zMax-1;
            } else {
                z1 = r1zMin;
            }
        } else {
            x1 = r1xMin;
            if( r1zMin < r2zMin ) {
                z1 = r1zMax-1;
            } else {
                z1 = r1zMin;
            }
        }

        // Corner of room2
        if( r2xMin < r1xMin ) {
            x2 = r2xMax-1;
            if( r2zMin < r1zMin ) {
                z2 = r2zMax-1;
            } else {
                z2 = r2zMin;
            }
        } else {
            x2 = r2xMin;
            if( r2zMin < r1zMin ) {
                z2 = r2zMax-1;
            } else {
                z2 = r2zMin;
            }
        }

        // They do not overlap at all
        int xDist = Math.abs(x2 - x1);
        int zDist = Math.abs(z2 - z1);

        return xDist + zDist;
    }

    @Override
    public String toString() {
        return "Room[loc:" + loc + ", size:" + size + ", type:" + type + ", style:" + style + "]";
    }
}

createRoom = { style ->
    return new Room(style);
}


class Tunnel {
    static Logger log = LoggerFactory.getLogger(Tunnel.class);

    // Start and end will typically be inside
    // the room.  That way we can determine the
    // direction even for a one-cell tunnel.
    Vec3i start;
    Vec3i end;
    Direction dir;
    int length;
    boolean capped;
    Style style;

    public Tunnel( Style style, Vec3i start, Vec3i end ) {
        this.style = style;
        this.start = start;
        this.end = end;
        // Figure out the direction
        Vec3i v = end.subtract(start);
        // Make a lot of assumptions about the start/end points
        // that we are fed
        if( v.x > 0 ) {
            dir = Direction.East;
            length = v.x;
        } else if( v.x < 0 ) {
            dir = Direction.West;
            length = -v.x;
        } else if( v.z > 0 ) {
            dir = Direction.South;
            length = v.z;
        } else if( v.z < 0 ) {
            dir = Direction.North;
            length = -v.z;
        } else {
            throw new IllegalArgumentException("Can't determine direction from:" + start + " and:" + end);
        }
    }

    public boolean intersects( def map ) {
        Vec3i v = new Vec3i();
        Vec3i d = dir.getVec3i();
        Vec3i right = dir.getRight();
        int stop = length;
        if( capped ) {
            stop++;
        }
        for( int i = 1; i < stop; i++ ) {
            v.set(start.x + i * d.x, start.y, start.z + i * d.z);
            // For now we will let edges cross each other
            // 2023-04-15: I don't know what that comment means because
            // clearly we are not allowing tunnels to intersect tunnels with
            // the below.
            //if( map[v.x][v.z] != 0 ) {
            //    return true;
            //}
            // Instead, we will only check intersection with rooms here
            // and do another check for crossing tunnels.
            //if( map[v.x][v.z] > 0 ) {
            //    return true;
            //}
            // No we will keep tunnels from crossing here because we also
            // check left/right to avoid tunnels that sit right next to each
            // other.  We will check corners specifically against
            // existing edges later.
            if( map[v.x][v.z] != 0 ) {
                return true;
            }
            if( i == length && capped ) {
                continue;
            }
            if( v.x + right.x > 0 && v.z + right.z > 0
                && v.x + right.x < map.length - 1 && v.z + right.z < map[0].length - 1 ) {
                if( map[v.x + right.x][v.z + right.z] != 0 ) {
                    return true;
                }
            }
            if( v.x - right.x > 0 && v.z - right.z > 0
                && v.x - right.x < map.length - 1 && v.z - right.z < map[0].length - 1 ) {
                if( map[v.x - right.x][v.z - right.z] != 0 ) {
                    return true;
                }
            }
        }
        return false;
    }

    public void fill( def map, int val ) {
        Vec3i v = new Vec3i();
        Vec3i d = dir.getVec3i();
        int stop = length;
        if( capped ) {
            stop++;
        }
        for( int i = 1; i < stop; i++ ) {
            v.set(start.x + i * d.x, start.y, start.z + i * d.z);
            if( map[v.x][v.z] == 0 ) {
                map[v.x][v.z] = val
            }
        }
    }

    public void insert( Vec3i offset, def world, boolean cutawayView ) {
        Vec3d v = new Vec3d();
        Vec3i d = dir.getVec3i();
        //if( capped ) {
        //    stop++;
        //}

        int xMin = -1;
        int yMin = -1;
        int zMin = -1;
        int xs = 3;
        int ys = 3;
        int zs = 3;

        // From grid space to world space
        Vec3d base = new Vec3d(start.x * 2, start.y, start.z * 2);
        int stop = length * 2;

        //if( GlobalTypes.cutawayView && GlobalTypes.drawEndpoints ) {
        //    int yTest = base.y + 3;
        //    int testBlock = GlobalTypes.wattle;
        //    while( world.getWorldCell(new Vec3d(base.x, yTest, base.z)) != 0 ) {
        //        yTest++;
        //        testBlock = GlobalTypes.whiteLight;
        //    }
        //    world.setWorldCell(new Vec3d(base.x, yTest, base.z), testBlock);
        //    yTest = end.y + 3;
        //    testBlock = GlobalTypes.cobble;
        //    while( world.getWorldCell(new Vec3d(end.x * 2, yTest, end.z * 2)) != 0 ) {
        //        yTest++;
        //        testBlock = GlobalTypes.redLight;
        //    }
        //    world.setWorldCell(new Vec3d(end.x * 2, yTest++, end.z * 2), testBlock);
        //}

        int startIndex = 0;
        int endIndex = stop;

        if( d.x > 0 || d.z > 0 ) {
            startIndex += 2;
        }
        if( d.x < 0 || d.z < 0 ) {
            startIndex += 1;
            endIndex -= 1;
        }

        for( int i = startIndex; i < endIndex; i++ ) {
            v.set(start.x * 2 + i * d.x, start.y, start.z * 2 + i * d.z);

            //world.setWorldCell(new Vec3d(v.x, v.y, v.z), GlobalTypes.whiteLight);

            if( d.x == 0 ) {
                for( int x = xMin; x < xs; x++ ) {
                    for( int y = yMin; y < ys; y++ ) {
                        Vec3d local = v.add(x, y, 0);

                        int block = 0;
                        if( y == -1 ) {
                            block = style.floorType;
                        } else if( y == 3 - 1 ) {
                            if( !cutawayView ) {
                                block = style.roofType;
                            } else {
                                block = -1;
                            }
                        } else if( x == -1 || x == 3-1 ) {
                            int existing = world.getWorldCell(local.add(offset));
                            if( existing == 0 ) {
                                if( cutawayView && y > 0 ) {
                                    block = 0;
                                } else {
                                    block = style.wallType;
                                }
                            } else {
                                block = -1;
                            }
                        }

                        //cells.setCell(GlobalOffset.xMapOffset + v.x * 2 + x, v.y + y, GlobalOffset.zMapOffset + v.z * 2 + z, block);
                        if( block >= 0 ) {
                            world.setWorldCell(local.add(offset), block);
                        }
                    }
                }
            } else {
                for( int z = zMin; z < zs; z++ ) {
                    for( int y = yMin; y < ys; y++ ) {
                        Vec3d local = v.add(0, y, z);

                        int block = 0;
                        if( y == -1 ) {
                            block = style.floorType;
                        } else if( y == 3 - 1 ) {
                            if( !cutawayView ) {
                                block = style.roofType;
                            } else {
                                block = -1;
                            }
                        } else if( z == -1 || z == 3 - 1 ) {
                            int existing = world.getWorldCell(local.add(offset));
                            if( existing == 0 ) {
                                if( cutawayView && y > 0 ) {
                                    block = 0;
                                } else {
                                    block = style.wallType;
                                }
                            } else {
                                block = -1;
                            }
                        }

                        //cells.setCell(GlobalOffset.xMapOffset + v.x * 2 + x, v.y + y, GlobalOffset.zMapOffset + v.z * 2 + z, block);
                        if( block >= 0 ) {
                            world.setWorldCell(local.add(offset), block);
                        }
                    }
                }
            }
        }

        if( capped ) {

            for( int x = xMin; x < xs; x++ ) {
                for( int z = zMin; z < zs; z++ ) {
                    v.set(end.x * 2 + x, end.y, start.z * 2 + z);

                    world.setWorldCell(v.add(0, -1, 0).add(offset), style.floorType);

                    if( !cutawayView ) {
                        world.setWorldCell(v.add(0, ys - 1, 0).add(offset), style.roofType);
                    }

                    if( x > xMin && x < xs -1 && z > zMin && z < zs - 1 ) {
                        for( int y = 0; y < 2; y++ ) {
                            world.setWorldCell(v.add(0, y, 0).add(offset), 0);
                        }
                    } else {
                        // It's a wall
                        for( int y = 0; y < 2; y++ ) {
                            Vec3d local = v.add(0, y, 0);
                            int existing = world.getWorldCell(local.add(offset));
                            if( existing != 0 ) {
                                int block = style.wallType;
                                if( cutawayView && y > 0 ) {
                                    block = 0;
                                }
                                world.setWorldCell(v.add(0, y, 0).add(offset), block);
                            }
                        }
                    }
                }
            }

            //if( GlobalTypes.lightJunctions ) {
            //    v.set(end.x * 2, end.y + 3 - 1, start.z * 2);
            //    world.setWorldCell(new Vec3d(v.x, v.y, v.z), GlobalTypes.whiteLight);
            //}
        }
    }

    //public void draw( CellArray cells, int type ) {
    public void drawSolid( Vec3i offset, def world, int type, int yOffset, int[] blockTypes ) {
        //int block = type == 0 ? GlobalTypes.stone : GlobalTypes.dirt;
        //if( !GlobalTypes.colorCoded ) {
        //    block = GlobalTypes.wattle;
        //}
        int block = type == 0 ? blockTypes[0] : blockTypes[1];
        if( !GlobalTypes.colorCoded ) {
            block = blockTypes[3];
        }
        Vec3i v = new Vec3i();
        Vec3i d = dir.getVec3i();
        int stop = length;
        if( capped ) {
            stop++;
        }
        for( int i = 1; i < stop; i++ ) {
            v.set(start.x + i * d.x, start.y, start.z + i * d.z);
            for( int x = 0; x < 2; x++ ) {
                for( int z = 0; z < 2; z++ ) {
                    for( int y = 0; y < 2; y++ ) {
                    //for( int y = 0; y < 5; y++ ) {
                        //cells.setCell(GlobalOffset.xMapOffset + v.x * 2 + x, v.y + y, GlobalOffset.zMapOffset + v.z * 2 + z, block);
                        world.setWorldCell(new Vec3d(offset.x + v.x * 2 + x, offset.y + v.y + y + yOffset, offset.z + v.z * 2 + z), block);
                    }
                }
            }
        }

        //if( false ) {
        //    // For now we mark the doors with a single tick
        //    v.set(start.x * 2 + d.x, start.y, start.z * 2 + d.z);
        //    for( int y = 0; y < 5; y++ ) {
        //        cells.setCell(v.x, v.y + y, v.z, GlobalTypes.wood);
        //    }
        //    //v.set(end.x * 2 - d.x, end.y, end.z * 2 - d.z);
        //    if( !capped ) {
        //        v.set(end.x * 2, end.y, end.z * 2);
        //        for( int y = 0; y < 5; y++ ) {
        //            cells.setCell(v.x, v.y + y, v.z, GlobalTypes.wood);
        //        }
        //    }
        //}
    }
}

createTunnel = { Style style, Vec3i start, Vec3i end ->
    return new Tunnel(style, start, end);
}

// Internal room graph tracking
class Edge implements Comparable {
    static Logger log = LoggerFactory.getLogger(Edge.class);

    //static int edgeCounter = 0;

    //def log;
    final int edgeId;// = edgeCounter++;
    final Room r1;
    final Room r2;
    boolean connected;
    boolean hasTunnel;
    int type;
    Double length;

    List<Tunnel> tunnels;

    boolean traversable;

    public Edge( int edgeId, Room r1, Room r2 ) {
        this.edgeId = edgeId;
        this.r1 = r1;
        this.r2 = r2;
    }

    //public void addTunnel( Vec3i start, Vec3i end ) {
    //    addTunnel(new Tunnel(start, end));
    //}

    public void addTunnel( Tunnel tunnel ) {
        if( tunnels == null ) {
            tunnels = new ArrayList<>();
        }
        tunnels.add(tunnel);
        hasTunnel = true;
    }

    public double getLength() {
        //return r1.center.getDistance(r2.center);
        if( length == null ) {
            length = r1.tunnelDistance(r2);
            //log.info(" length:" + length);
            if( length < 0 ) {
                throw new RuntimeException("Bad things happened:" + length);
            }
        }
        return length;
    }

    public Room adjacent( Room r ) {
        return r == r1 ? r2 : r1;
    }

    public int compareTo( Object other ) {
        return getLength() <=> other.length;
    }

    public String toString() {
        return "(" + edgeId + ")" + r1 + "<->" + r2 + ":" + connected + " len:" + getLength();
    }
}
//Edge.edgeCounter = 0;

class Path {
    def room;
    Path parent;
    int length;
    boolean deadEnd;
    int deadEndDepth;

    public Path expand( def room ) {
        return new Path(room:room, parent:this, length:length + 1);
    }
}

class SizeFunction {
    int minSize;
    int maxSize;
    int minHeight;
    int maxHeight;
    
    public Vec3i create( DungeonGenerator generator ) {
        return create(maxSize, maxSize, generator);
    }

    public Vec3i create( int xMax, int zMax, DungeonGenerator generator ) {
        int x = generator.nextInRange(minSize, xMax);
        int y = generator.nextInRange(minHeight, maxHeight);
        int z = generator.nextInRange(minSize, zMax);
        return new Vec3i(x, y, z);
    }
}
class FeatureSizeFunction extends SizeFunction {
    
    public Vec3i create( int xMax, int zMax, DungeonGenerator generator ) {
        Vec3i result = super.create(xMax, zMax, generator);

        // Keep the rooms from being really weird shapes or too small.
        if( result.x == 2 && result.z == 2 ) {
            switch( generator.nextInRange(0, 2) ) {
                case 0:
                    result.x++;
                    break;
                case 1:
                    result.z++;
                    break;
                default:
                    result.x++;
                    result.z++;
                    break;
            }
        } else if( result.x == 2 && result.z > 4 ) {
            result.x++;
        } else if( result.z == 2 && result.x > 4 ) {
            result.z++;
        }
        return result;
    }
}
createSizeFunction = { int minSize, int maxSize, int minHeight, int maxHeight ->
    return new SizeFunction(minSize:minSize, maxSize:maxSize, minHeight:minHeight, maxHeight:maxHeight);
}
createFeatureSizeFunction = { int minSize, int maxSize, int minHeight, int maxHeight ->
    return new FeatureSizeFunction(minSize:minSize, maxSize:maxSize, minHeight:minHeight, maxHeight:maxHeight);
}


// ---------------------------------------------------------------------------
// Actual dungeon generator code
// ----------------------------------------------------------------------------

class DungeonGenerator {
    static Logger log = LoggerFactory.getLogger(DungeonGenerator.class);

    private long seed;
    private Random rand;
    private int xMapSize = 80;
    private int yMapSize = 40;
    private int zMapSize = 80;

    private SizeFunction featureSize;
    private SizeFunction infillSize;
    private int maxFeatureCount = 20;
    private int maxInfillCount = 20;

    private Style featureStyle;
    private Style infillStyle;
    private Style tunnelStyle;

    private List<Room> entrances = new ArrayList<>();
    private List<Room> rooms = new ArrayList<>();
    private ListMultimap<Room, Edge> graph = MultimapBuilder.hashKeys().arrayListValues().build();
    private List<Edge> edges = new ArrayList<>();

    // Keep track of the pairs of rooms that are only two hops
    // apart by main links
    private Multimap<Room, Room> nearPairs = MultimapBuilder.hashKeys().hashSetValues().build();

    // On a 2x2 grid, map size is half of build area
    private int xMap = xMapSize / 2;
    private int zMap = zMapSize / 2;
    private int[][] map = new int[xMap][zMap];

    // Paths from any room to the nearest entrance room
    private Map<Room, Path> pathIndex = new HashMap<>();
    private Path maxFeaturePath = null;
    private Path maxInfillPath = null;

    public DungeonGenerator( long seed ) {
        this.seed = seed;
        this.rand = new Random(seed);
    }

    public DungeonGenerator copySetup( long seed ) {
        DungeonGenerator result = new DungeonGenerator(seed);
        result.featureStyle = featureStyle;
        result.infillStyle = infillStyle;
        result.tunnelStyle = tunnelStyle;
        result.featureSize = featureSize;
        result.infillSize = infillSize;
        result.maxFeatureCount = maxFeatureCount;
        result.maxInfillCount = maxInfillCount;
        result.xMapSize = xMapSize; 
        result.yMapSize = yMapSize;
        result.zMapSize = zMapSize;
        result.xMap = xMap;
        result.zMap = zMap;
        result.map = new int[xMap][zMap];
        return result;
    }
    
    /**
     *  Returns the next random int between min and max inclusive.
     */
    public int nextInRange( int min, int max ) {
        return rand.nextInt(max - min + 1) + min;
    }

    public void addEntrance( Room room ) {
        entrances.add(room);
        rooms.add(room);
    }    
    
    protected void buildGraph() {
        int edgeCount = 0;
        for( int i = 0; i < rooms.size(); i++ ) {
            Room r1 = rooms.get(i);
            for( int j = i + 1; j < rooms.size(); j++ ) {
                Room r2 = rooms.get(j);
                Edge edge = new Edge(edgeCount++, r1, r2);
                if( edge.edgeId != edges.size() ) {
                    throw new RuntimeException("Mismatch in edge table, edgeId:" + edge.edgeId + "  should be:" + edges.size());
                }
                edges.add(edge);
                graph.put(r1, edge);
                graph.put(r2, edge);
            }
        }
    }

    public void generateFeatureRooms( int count, Style initialStyle ) {
        log.info("Feature room count:" + count);        
        for( int i = 0; i < count; i++ ) {     
            def room = new Room(initialStyle);
            room.size = featureSize.create(this);
            rooms.add(room);
        }
    }

    public void generateFeatureRooms( Style initialStyle ) {
        int maxRooms = nextInRange(11, maxFeatureCount);
        int count = maxRooms - rooms.size();
        generateFeatureRooms(count, initialStyle);
    }

    public void generateFeatureRooms() {
        generateFeatureRooms(featureStyle);
    }

    /**
     *  Takes all of the existing rooms and plots them randomly around
     *  the map, ensuring no overlap and nice distribution.
     */
    protected void placeRooms() {

        // Sort the rooms by area so that we place the largest ones
        // first
        rooms.sort { r1, r2 -> -(r1.area <=> r2.area) }

        // Places random rooms in a grid that is sized to contain
        // the maximum sized room possible.  The rooms are then ranomly placed
        // within those grid cells.

        // Find the max room sizes for a grid
        int xGrid = 0;
        int zGrid = 0;
        rooms.each { room ->
            xGrid = Math.max(room.size.x, xGrid);
            zGrid = Math.max(room.size.z, zGrid);
        }

        // Calculate a grid size that should contain all of the rooms.
        // Usuing the ceiling of the sqrt means that the grid will be
        // big enough to hold the 'next largest square' count of rooms.
        // ie: anything from 17-24 would get an idealSize of 5 which is
        // enough to hold 25 rooms.  It might further get clamped to "map size"
        // later but at least this would have held all of the rooms.
        // ...and we end up not even using it depending on how the 'compact'
        // flag is set.
        int idealSize = Math.ceil(Math.sqrt(rooms.size()));

        // In this case, we don't do a border check for placement as we
        // assume our grid cells are empty, so we need extra space.
        //xGrid += 2;
        //zGrid += 2;
        // But one extra cell should be enough because we only need
        // one block between rooms.
        xGrid++;
        zGrid++;

        // We could be smarter about how we set this up to be a tighter
        // fit +some random fudge... but for now we'll just base it on the
        // maximum map size.
        int xs = xMap / xGrid;
        int zs = zMap / zGrid;
        // Today I'm not sure what the above comment means exactly.
        // It seems like in some cases we could be spreading the map out
        // unnecessarily but that might be desirable.  It's easy to imagine
        // other cases where we have to fit more rooms than our map would
        // normally allow... maybe that's what it's referring to but I don't
        // know how we could be 'tighter' in that case.
        // If we were not constained by map size, we could use idealSize
        // directly and always have enough room and never 'spread out'.

        // In practice, one seems no better than the other as far as
        // sprawling dungeons.  A real compact grid would try to get as close
        // as possible to perfect fit.  sqrt() can add a lot of extra rooms (11 -> 16, 17->25, etc.)
        boolean compact = false;
        if( compact ) {
            int xTemp = Math.min(xs, idealSize);
            int zTemp = Math.min(zs, idealSize);
            if( xTemp * zTemp >= rooms.size() ) {
                xs = xTemp;
                zs = zTemp;
                //log.info("Compact grid:" + xs + ", " + zs);
            }
        }

        // 2023-04-08 - Final analysis: today grid size is purely determined by
        // map size and grid cell size.  This means that we might leave rooms
        // unplaced.  As such, if there are required rooms then they should be
        // placed early in the list.
        //
        // The comments above (the old ones... and maybe even my new ones) make
        // a lot of claims.  Either approach is going to have lots of extra cells
        // for many cases.  The only 'tight fits' are when things are already
        // closely aligned.  If xs * zs is bigger than rooms.size() then there
        // will be extra cells... and potentially extra spread.
        //
        // If it ever comes up, we could use the idealSize to calculate a more
        // compact grid size.
        // xs = idealSize;
        // zs = Math.ceil(rooms.size() / xs);
        // ...of flip xs,zs to have taller instead of wider.
        // So in the 17 rooms case, instead of a 25 cell grid we'd have:
        // xs = 5
        // zs = 4   ...20 cells

        // Generate a grab-bag of all of the available cells so we can select
        // them randomly and not repeat cells.
        def gridCells = [];
        for( int x = 0; x < xs; x++ ) {
            for( int z = 0; z < zs; z++ ) {
                gridCells.add(new Vec3i(x, 0, z));
            }
        }

//log.info("Available cells before:" + gridCells.size());

        // See if any of the already placed rooms intersect the cells
        rooms.each { room ->
//log.info("Checking room:" + room);
            if( room.loc == null ) {
                // waiting for a location
                log.info("Room waiting for a location:" + room);
                return;
            }
            log.info("Room already has a location:" + room);
            // Else it already has a location and we need to block the
            // map out for it
            for( Iterator it = gridCells.iterator(); it.hasNext(); ) {
                Vec3i cell = it.next();
                if( room.intersectsRectangle(cell.x * xGrid, cell.z * zGrid, xGrid, zGrid) ) {
                    log.info("Removing cell from consideration:" + cell + "  because of room:" + room);
                    it.remove();
                }
            }
        }

//log.info("Available cells after:" + gridCells.size());

        // Randomly assign a cell to each room
        def missed = [] as Set;
        rooms.each { room ->
            if( room.loc == null ) {
                if( gridCells.isEmpty() ) {
                    //log.info("Ran out of cells for:" + room);
                    // This happens when xs * xs < rooms.size() which can happen
                    // if the room count is high with respect to map size as it
                    // relates to max room size.
                    missed.add(room);
                    return;
                }
                //log.info("Finding location for:" + room);
                int index = rand.nextInt(gridCells.size());
                def v = gridCells.remove(index);

                // Randomly place the room within that cell
                int xd = xGrid - room.size.x;
                int zd = zGrid - room.size.z;
                int xOffset = xd > 0 ? rand.nextInt(xd) : 0;
                int zOffset = zd > 0 ? rand.nextInt(zd) : 0;

                room.loc = new Vec3i(v.x * xGrid + xOffset, 0, v.z * zGrid + zOffset);
            }

            // Mark that area of the map as used.
            room.fill(map, 1);
        }

        log.info("Failed to place " + missed.size() + " rooms out of " + rooms.size());
        rooms.removeAll(missed);
    }

    protected void generateInfillRooms() {
        generateInfillRooms(infillStyle);
    }
    
    protected void generateInfillRooms( Style initialStyle ) {

        // Probably we want to bound this grid by the out bounds of
        // the rooms generated in the first pass above... maybe with a
        // little extra buffer space.  If we are using grid-based placement
        // then it shouldn't be too different, though, in most cases.

        // Intersect tests will check against neighbors so we
        // can tighten the grid for infill, ie: no 1 unit buffer
        int xGrid = infillSize.maxSize; //maxInfillSize;
        int zGrid = infillSize.maxSize; //maxInfillSize;
        int minGrid = infillSize.minSize; //minInfillSize;
        //int xs = xMap / xGrid;
        //int zs = zMap / zGrid;
        //
        //int xCenter = xs / 2;
        //int zCenter = zs / 2;

        //log.info("Grid size:" + xGrid + ", " + zGrid);

        //int maxInfillCount = 20;
        double chanceFactor = 1.0;
        double chanceDelta = 1.0/maxInfillCount;

        //for( int xg = minGrid; xg < xGrid; xg++ ) {
        //    for( int zg = minGrid; zg < zGrid; zg++ ) {

        for( int xg = xGrid; xg >= minGrid; xg-- ) {
            for( int zg = zGrid; zg >= minGrid; zg-- ) {

                //log.info("pass:" + xg + ", " + zg);

                int xs = xMap / xg;
                int zs = zMap / zg;

                int xCenter = xs / 2;
                int zCenter = zs / 2;

                for( int i = 0; i < xs; i++ ) {
                    for( int j = 0; j < zs; j++ ) {
                        // See if the grid cell is empty or not
                        //def room = new Room(log:log, loc:new Vec3i(i * xg, 0, j * zg), size:new Vec3i(xg, 1, zg));
                        def room = new Room(initialStyle);
                        room.loc = new Vec3i(i * xg, 0, j * zg);
                        room.size = new Vec3i(xg, 1, zg);
                        //log.info("Checking cell:" + i + ", " + j + "  loc:" + room.loc + "  size:" + room.size);
                        if( room.intersects(map) ) {
                            continue;
                        }
                        double roll = rand.nextDouble();

                        // Higher probability near the center
                        double xDist = Math.abs(xCenter - i);
                        double zDist = Math.abs(zCenter - i);
                        double xChance = (xDist * 0.5) / xCenter;
                        double zChance = (zDist * 0.5) / zCenter;
                        //log.info("xChance:" + xChance + "  zChance:" + zChance + "  chanceFactor:" + chanceFactor + "  roll:" + roll);
                        roll *= chanceFactor;
                        if( roll < xChance + zChance ) {
                            continue;
                        }
                        Vec3i size = infillSize.create(xg, zg, this);

                        //log.info("xg:" + xg + "  x:" + x+ " minInfill:" + minInfillSize);

                        int xOffset = (xg-size.x) > 0 ? rand.nextInt(xg - size.x) : 0;
                        int zOffset = (zg-size.z) > 0 ? rand.nextInt(zg - size.z) : 0;

                        room.loc.x += xOffset;
                        room.loc.z += zOffset;
                        room.size = size;
                        room.type = 1;
                        rooms.add(room);

                        room.fill(map, 2);

                        chanceFactor -= chanceDelta;
                    }
                    if( chanceFactor <= 0 ) {
                        break;
                    }
                }
                if( chanceFactor <= 0 ) {
                    break;
                }
            }
            if( chanceFactor <= 0 ) {
                break;
            }
        }
    }

    /**
     *  Make sure all rooms in the graph are connected to all of the others.
     */
    protected double fullyConnect() {

        def visited = [] as Set;
        def pending = new PriorityQueue();
        def current = rooms[0];
        visited.add(current);
        //current.size.y = 10;
        int count = 1;

        double longestEdge = 0;

        // While there are still disconnected rooms
        while( visited.size() < rooms.size() ) {
            //log.info("Checking room:" + current + "  pending edges:" + pending.size());

            // Add all of the pending edges leaving this new pending node
            pending.addAll(graph.get(current));

            // Clean the pending queue out up to the shortest 'unconnected' edge
            Edge shortest = null;
            while( !pending.isEmpty() ) {
                Edge e = pending.poll();
                def r1 = visited.contains(e.r1);
                def r2 = visited.contains(e.r2);
                //log.info("purging:" + e + "  r1:" + r1 + " r2:" + r2);
                if( e.connected ) {
                    continue;
                }
                if( r1 && r2 ) {
                    // This edge is redundant
                    continue;
                }
                // One end is guaranteed to be connected or we wouldn't even have added
                // it to the pending queue

                shortest = e;
                break;
            }

            shortest.connected = true;
            shortest.type = 0;
            if( shortest.length > longestEdge ) {
                longestEdge = shortest.length;
            }

            // Just because we started on a particular room does not mean
            // that it's part of the edge we picked.  Have to check both ends.
            if( visited.add(shortest.r1) ) {
                current = shortest.r1;
                //log.info("Newly connected:" + current);
            }
            if( visited.add(shortest.r2) ) {
                current = shortest.r2;
                //log.info("Newly connected:" + current);
            }
            visited.add(current);
            //current.size.y = 10;
            int test = 0;
            if( count <= test ) {
                current.size.y = 10;
            }
            if( count == test ) {
                log.info("-------------------------------------");
            }
            count++;
        }

        return longestEdge;
    }

    /**
     *  Randomly add additional connections between nearby rooms.
     */
    protected void randomCrossConnect( double longestEdge ) {

        // For every room, connect every adjacent room (through real links)
        rooms.each { room ->
            def adjacent = [];
            graph.get(room).each { edge ->
                if( edge.connected ) {
                    adjacent.add(edge.adjacent(room));
                }
            }
            //log.info("Room:" + room);
            //adjacent.each {
            //    log.info("   adj:" + it);
            //}
            // n * n because we actually want both directions anyway
            for( Room r1 : adjacent ) {
                for( Room r2 : adjacent ) {
//                    log.info("[" + r1 + "][" + r2 + "]");
                    nearPairs.put(r1, r2);
                }
            }
        }

        //pending.clear();
        def pending = new PriorityQueue();

        //log.info("longest:" + longestEdge);

        boolean redundantEdges = true;
        if( redundantEdges ) {
            edges.each { edge ->
                if( !edge.connected ) {
                    pending.add(edge);
                }
            }
            while( !pending.isEmpty() ) {
                Edge edge = pending.poll();
            //log.info("Random check:" + edge);
                if( edge.length > longestEdge ) {
                    break;
                }
                // Random chance for connecting them
                int roll = rand.nextInt(8);
            //log.info("roll:" + roll);
                if( roll <= 4 ) {
                    edge.connected = true;
                    edge.type = 1;
                }
            }
        }
    }

    protected void generateTunnels() {
        generateTunnels(tunnelStyle);
    }

    protected void generateTunnels( Style initialStyle ) {
        double longestEdge = fullyConnect();
        randomCrossConnect(longestEdge);
    
        generateTunnelsForEdgeType(0, initialStyle);
        generateTunnelsForEdgeType(1, initialStyle);
    }

    protected void generateTunnelsForEdgeType( int edgeType, Style initialStyle ) {
        // Plot the actual tunnels
        // The main tunnels we will have to draw for sure and so we may
        // have to force rendering when we'd rather give up.  The alternate
        // tunnels can be skipped if we can't plot them.  So we'll do this in
        // two passes.
        boolean keepGoing = true;
        edges.each { edge ->
            if( !keepGoing ) {
                return;
            }
            if( edge.hasTunnel || !edge.connected ) {
                return;
            }
            if( edge.type != edgeType ) { //> 0 ) {
                // Not a main tunnel
                return;
            }

            // Figure out what kind of tunnel we need...
            // Note: we might have been able to keep track of this information
            // in the length calculation if we moved it to edge instead of room.

            if( edges.get(edge.edgeId) != edge ) {
                throw new RuntimeException("Hmmm?");
            }
            int edgeId = -(edge.edgeId + 1);
            if( edges.get(-edgeId - 1) != edge ) {
                throw new RuntimeException("What the what?");
            }

            int r1xMin = edge.r1.loc.x;
            int r1zMin = edge.r1.loc.z;
            int r1xMax = edge.r1.loc.x + edge.r1.size.x;
            int r1zMax = edge.r1.loc.z + edge.r1.size.z;

            int r2xMin = edge.r2.loc.x;
            int r2zMin = edge.r2.loc.z;
            int r2xMax = edge.r2.loc.x + edge.r2.size.x;
            int r2zMax = edge.r2.loc.z + edge.r2.size.z;

            Tunnel tunnel = null;
            if( (r1xMax > r2xMin && r1xMax <= r2xMax)
                || (r1xMin >= r2xMin && r1xMin < r2xMax)
                // It's possible for both of r1's min and max to be outside
                // of r2 if it envelopes it, so we have to check both ways.
                || (r2xMax > r1xMin && r2xMax <= r1xMax)
                || (r2xMin >= r1xMin && r2xMin < r1xMax) ) {
                // They overlap on the x-axis
                // Figure out the best spot
                int xMin = Math.max(r1xMin, r2xMin);
                int xMax = Math.min(r1xMax, r2xMax);
                int x = (xMin + xMax) / 2;
                //log.info("overlap on x-axis");
                if( r1zMin < r2zMin ) {
                    // Draw from r1zMax to r2zMin
                    //drawTunnelZ(x, r1zMax, r2zMin, edgeId);
                    tunnel = new Tunnel(initialStyle, new Vec3i(x, 0, r1zMax-1), new Vec3i(x, 0, r2zMin));
                    //tunnel = createTunnel(initialStyle, new Vec3i(x, 0, r1zMax-1), new Vec3i(x, 0, r2zMin));
                } else {
                    // Draw from r2zMax to r1zMin
                    //drawTunnelZ(x, r2zMax, r1zMin, edgeId);
                    tunnel = new Tunnel(initialStyle, new Vec3i(x, 0, r2zMax-1), new Vec3i(x, 0, r1zMin));
                    //tunnel = createTunnel(initialStyle, new Vec3i(x, 0, r2zMax-1), new Vec3i(x, 0, r1zMin));
                }
            }
            if( (r1zMax > r2zMin && r1zMax <= r2zMax)
                || (r1zMin >= r2zMin && r1zMin < r2zMax)
                // It's possible for both of r1's min and max to be outside
                // of r2 if it envelopes it, so we have to check both ways.
                || (r2zMax > r1zMin && r2zMax <= r1zMax)
                || (r2zMin >= r1zMin && r2zMin < r1zMax) ) {
                // They overlap on the z-axis
                //log.info("overlap on z-axis");
                // Figure out the best spot
                int zMin = Math.max(r1zMin, r2zMin);
                int zMax = Math.min(r1zMax, r2zMax);
                int z = (zMin + zMax) / 2;
                if( r1xMin < r2xMin ) {
                    // Draw from r1xMax to r2xMin
                    //drawTunnelX(z, r1xMax, r2xMin, edgeId);
                    tunnel = new Tunnel(initialStyle, new Vec3i(r1xMax-1, 0, z), new Vec3i(r2xMin, 0, z));
                    //tunnel = createTunnel(initialStyle, new Vec3i(r1xMax-1, 0, z), new Vec3i(r2xMin, 0, z));
                } else {
                    // Draw from r2xMax to r1xMin
                    //drawTunnelX(z, r2xMax, r1xMin, edgeId);
                    tunnel = new Tunnel(initialStyle, new Vec3i(r2xMax-1, 0, z), new Vec3i(r1xMin, 0, z));
                    //tunnel = createTunnel(initialStyle, new Vec3i(r2xMax-1, 0, z), new Vec3i(r1xMin, 0, z));
                }
            }
            if( tunnel != null ) {
                if( !tunnel.intersects(map) ) {
                    edge.addTunnel(tunnel);
                    tunnel.fill(map, edgeId);
                    return;
                } else {
//                    log.info("Blocked:" + edge + "  type:" + edgeType);
                    if( edgeType == 1 ) {
                        // Might be redundant anyway
                        if( nearPairs.get(edge.r1).contains(edge.r2) ) {
//                            log.info("Disconnecting redundant edge");
                            edge.connected = false;
                        }
                        return;
                    }
                    //edge.connected = false;
                }
                // I think for type 0 edges then we don't want to give up at this point.
                //return;
            }

            // They do not overlap at all
            //log.info("no overlap");

            //    log.info("tunnel r1:" + r1xMin + ", " + r1zMin + " -> " + r1xMax + ", " + r1zMax
            //              + "  r2:" + r2xMin + ", " + r2zMin + " -> " + r2xMax + ", " + r2zMax);

            // There are multiple tunnels we might try before
            // giving up and using pathfinding.  Two different
            // forms of center-side to center-side as well as
            // doors near the corner, etc..
            def r1xDoors = [];
            if( r1xMin < r2xMin ) {
                // Doors on the east side
                // Just a center door for now
                r1xDoors.add(new Vec3i(r1xMax-1, 0, edge.r1.loc.z + (int)(edge.r1.size.z / 2)));
            } else {
                // Doors on the west side
                // Just a center door for now
                r1xDoors.add(new Vec3i(r1xMin, 0, edge.r1.loc.z + (int)(edge.r1.size.z / 2)));
            }

            def r1zDoors = [];
            if( r1zMin < r2zMin ) {
                // Doors on the north side
                r1zDoors.add(new Vec3i(edge.r1.loc.x + (int)(edge.r1.size.x / 2), 0, r1zMax - 1));
            } else {
                // Doors on the south side
                r1zDoors.add(new Vec3i(edge.r1.loc.x + (int)(edge.r1.size.x / 2), 0, r1zMin));
            }

            def r2xDoors = [];
            if( r2xMin < r1xMin ) {
                // Doors on the east side
                // Just a center door for now
                r2xDoors.add(new Vec3i(r2xMax-1, 0, edge.r2.loc.z + (int)(edge.r2.size.z / 2)));
            } else {
                // Doors on the west side
                // Just a center door for now
                r2xDoors.add(new Vec3i(r2xMin, 0, edge.r2.loc.z + (int)(edge.r2.size.z / 2)));
            }

            def r2zDoors = [];
            if( r2zMin < r1zMin ) {
                // Doors on the north side
                r2zDoors.add(new Vec3i(edge.r2.loc.x + (int)(edge.r2.size.x / 2), 0, r2zMax - 1));
            } else {
                // Doors on the south side
                r2zDoors.add(new Vec3i(edge.r2.loc.x + (int)(edge.r2.size.x / 2), 0, r2zMin));
            }

            boolean found = false;

            // Try r1x to r2z first
            for( Vec3i v1 : r1xDoors ) {
                for( Vec3i v2 : r2zDoors ) {
                    // The x-axis tunnel
                    Tunnel t1 = new Tunnel(initialStyle, v1, new Vec3i(v2.x, 0, v1.z));
                    //Tunnel t1 = createTunnel(initialStyle, v1, new Vec3i(v2.x, 0, v1.z));
                    t1.capped = true;
                    if( t1.intersects(map) ) {
                        continue;
                    }
                    // The z-axis tunnel
                    Tunnel t2 = new Tunnel(initialStyle, new Vec3i(v2.x, 0, v1.z), v2);
                    //Tunnel t2 = createTunnel(initialStyle, new Vec3i(v2.x, 0, v1.z), v2);
                    if( t2.intersects(map) ) {
                        continue;
                    }
                    // Need to also check the junction.  For that, we'll use
                    // a fake 1x1 room.
                    // Without this, it's possible to get corners that sit right
                    // next to an unrelated room and actuall overlap its eventual
                    // wall.  It's tempting to add a real room here but we don't
                    // want to interfere with alternate tunnel routing which normally
                    // can overlap existing tunnels.
                    //def junction = new Room(loc:t1.end, size:new Vec3i(1, 2, 1));
                    def junction = new Room(null);
                    junction.loc = t1.end;
                    junction.size = new Vec3i(1, 2, 1);
                    if( junction.intersectsRooms(map) ) {
                        continue;
                    }

                    edge.addTunnel(t1);
                    edge.addTunnel(t2);
                    // Don't fill the corner tunnels because we will go back later
                    // and check for intersection with existing straight tunnels.
                    // If we fill here, we will never have 'T' intersections because
                    // we'll detect the crossing.
                    //t1.fill(map, edgeId);
                    //t2.fill(map, edgeId);
                    found = true;
                    break;
                }
                if( found ) {
                    break;
                }
            }
            if( found ) {
                return;
            }

            // Now try r1z to r2x
            for( Vec3i v1 : r1zDoors ) {
                for( Vec3i v2 : r2xDoors ) {
                    // The x-axis tunnel
                    Tunnel t1 = new Tunnel(initialStyle, v2, new Vec3i(v1.x, 0, v2.z));
                    //Tunnel t1 = createTunnel(initialStyle, v2, new Vec3i(v1.x, 0, v2.z));
                    t1.capped = true;
                    if( t1.intersects(map) ) {
                        continue;
                    }
                    // The z-axis tunnel
                    Tunnel t2 = new Tunnel(initialStyle, new Vec3i(v1.x, 0, v2.z), v1);
                    //Tunnel t2 = createTunnel(initialStyle, new Vec3i(v1.x, 0, v2.z), v1);
                    if( t2.intersects(map) ) {
                        continue;
                    }

                    // Need to also check the junction.  For that, we'll use
                    // a fake 1x1 room.
                    //def junction = new Room(loc:t1.end, size:new Vec3i(1, 2, 1));
                    def junction = new Room(null);
                    junction.loc = t1.end;
                    junction.size = new Vec3i(1, 2, 1);
                    if( junction.intersectsRooms(map) ) {
                        continue;
                    }

                    edge.addTunnel(t1);
                    edge.addTunnel(t2);
                    // Don't fill the corner tunnels because we will go back later
                    // and check for intersection with existing straight tunnels.
                    // If we fill here, we will never have 'T' intersections because
                    // we'll detect the crossing.
                    //t1.fill(map, edgeId);
                    //t2.fill(map, edgeId);
                    found = true;
                    break;
                }
                if( found ) {
                    break;
                }
            }
            if( found ) {
                return;
            }

            if( edgeType == 1 ) {
                // Might be redundant anyway
                if( nearPairs.get(edge.r1).contains(edge.r2) ) {
//                    log.info("Disconnecting redundant edge");
                    edge.connected = false;
                    return;
                }
                // These are the really interesting edges
            }
        }
    }
    
    public void generateLevel() {
        // Randomly generate the 'feature' rooms
        generateFeatureRooms();  
        log.info("Generated " + rooms.size() + " rooms");

        // Sort those rooms based on area
        placeRooms();

        // See about creating random 'in fill' rooms
        generateInfillRooms();

        // Build a graph that connects all rooms to all other rooms
        buildGraph();

        generateTunnels();
    }

    public void pruneDisconnectedEdges() {
        for( Iterator<Edge> it = edges.iterator(); it.hasNext(); ) {
            Edge edge = it.next();
            if( !edge.connected ) {
                it.remove();
                graph.remove(edge.r1, edge);
                graph.remove(edge.r2, edge);
            }
        }
    }

    public void calculatePaths() {
        if( !pathIndex.isEmpty() ) {
            return;
        }

        // Make sure we don't have cruft laying around
        pruneDisconnectedEdges();

        List<Path> pending = new LinkedList<>();

        // Calculate the different 'shortest paths' from the start to the various
        // rooms.
        for( Room room : entrances ) {
            Path root = new Path(room:room);
            pending.add(root);
        }

        log.info("Starting with " + pending.size() + " roots");

        maxFeaturePath = null;
        maxInfillPath = null;

        while( !pending.isEmpty() ) {
            Path path = pending.removeFirst();
            Path existing = pathIndex.get(path.room);
            if( existing != null ) {
                // We've already visited here and either we tied or the old
                // path is shorter
                continue;
            }

            // Else this is the first time we've visited this room
            pathIndex.put(path.room, path);

            if( path.room.type == 0 ) {
                if( maxFeaturePath == null || path.length > maxFeaturePath.length) {
                    maxFeaturePath = path;
                }
            } else {
                if( maxInfillPath == null || path.length > maxInfillPath.length ) {
                    maxInfillPath = path;
                }
            }

            // Extend into adjacent rooms
            graph.get(path.room).each { edge ->
                Room adj = edge.adjacent(path.room);
                // See if we've already been there or not
                if( pathIndex.containsKey(adj) ) {
                    return;
                }
                pending.add(path.expand(adj));
            }
        }

        // Mark the dead ends because that's useful
        for( Map.Entry<Room, Path> e : pathIndex.entrySet() ) {
            Room room = e.getKey();
            if( graph.get(room).size() != 1 ) {
                continue;
            }
            // This room is a dead end... and potentially some number of
            // hops along the way
            Path deadEnd = e.getValue();
            deadEnd.deadEnd = true;

            int count = 1;
            for( Path p = deadEnd.parent; p != null; p = p.parent ) {
                // The first time through we already know the degree will
                // be 1.  Any subsequent rooms with degree 2 will also be
                // part of this dead end because of where we came from and
                // where we're going.  Any additional branching and we're done
                // with this loop.
                if( graph.get(p.room).size() > 2 ) {
                    break;
                }
                // If we reached the starter room then it doesn't count
                if( entrances.contains(p.room) ) {
                    break;
                }
                p.deadEnd = true;
                count++;
            }
            deadEnd.deadEndDepth = count;
        }
    }

    public void insert( Vec3i offset, def world, boolean cutawayView ) {
        for( Room room : rooms ) {
            //room.loc.y++;
            room.insert(offset, world, cutawayView);
        }

        for( Edge edge : edges ) {
            if( !edge.connected ) {
                //log.info("Not drawing edge:" + edge);
                continue;
            }
            if( edge.tunnels ) {
                edge.tunnels.each { tunnel ->
                    //tunnel.start.y++;
                    //tunnel.end.y++;
                    tunnel.insert(offset, world, cutawayView);
                }

                // Already drawn
                continue;
            }
        }
    }

    public CellArray createBlockMap( int background, int[] types ) {

        // We never filled in the edges for the corners
        for( Edge edge : edges ) {
            if( !edge.connected ) {
                continue;
            }
            if( edge.tunnels ) {
                int edgeId = -(edge.edgeId + 1);
                edge.tunnels.each { tunnel ->
                    tunnel.fill(map, edgeId);
                }
            }
        }

        // Refill the map with the room IDs because they probably moved around
        int roomId = 1;
        for( Room room : rooms ) {
            room.fill(map, roomId++);
        }

        // Figure out how big we need to be.
        Vec3i min = new Vec3i(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE);
        Vec3i max = new Vec3i(0, 0, 0);
        for( int x = 0; x < xMap; x++ ) {
            for( int z = 0; z < zMap; z++ ) {
                if( map[x][z] != 0 ) {
                    min.x = Math.min(x, min.x);
                    min.z = Math.min(z, min.z);
                    max.x = Math.max(x, max.x);
                    max.z = Math.max(z, max.z);
                }
            }
        }

        for( Room room : rooms ) {
            min.y = Math.min((int)(room.loc.y / 2), min.y);
            max.y = Math.max((int)((room.loc.y + room.size.y) / 2), max.y);
        }

        int border = 0;
        int xSize = (max.x - min.x + 1);
        int ySize = (max.y - min.y + 1);
        int zSize = (max.z - min.z + 1);

        int yOffset = 0;
        if( background != 0 ) {
            border = 1;
            xSize += border * 2;
            zSize += border * 2;
            ySize++;
            yOffset++;
        }
        CellArray result = new CellArray(xSize, ySize, zSize);

        if( background != 0 ) {
            for( int x = 0; x < xSize; x++ ) {
                for( int z = 0; z < zSize; z++ ) {
                    result.setCell(x, 0, z, background);
                }
            }
        }

        for( int x = 0; x < xMap; x++ ) {
            for( int z = 0; z < zMap; z++ ) {
                int type = map[x][z];
                if( type == 0 ) {
                    continue;
                }
                if( type > 0 ) {
                    Room room = rooms.get(type - 1);
                    int blockType = types[0];
                    if( room.type == 2 && types.length > 2 ) {
                        blockType = types[2];
                    } else if( room.type == 3 && types.length > 3 ) {
                        blockType = types[3];
                    }
                    for( int y = 0; y < (int)(room.size.y / 2); y++ ) {
                        result.setCell(x-min.x + border, y + yOffset, z-min.z + border, blockType);
                    }
                } else {
                    result.setCell(x-min.x + border, yOffset, z-min.z + border, types[1]);
                }
            }
        }

        MaskUtils.calculateSideMasks(result);

        return result;
    }
}

createDungeonGenerator = { long seed ->
    return new DungeonGenerator(seed);
}

findStairs = { gen, boolean debug ->

    def maxFeature = null;
    def maxInfill = null;

    gen.calculatePaths();
    maxFeature = gen.maxFeaturePath;
    maxInfill = gen.maxInfillPath;

    // Possible stair placement logic
    // start with a certain probability
    // go through the best candidates and roll dice
    // keep track of how many 'stairs' we've generated.
    // lower the probability each time
    // ...if we still have no stairs then look for less
    // good candidates.
    // Whatever the case, stairs should have a certain configuration
    // of rooms to get to them: either bosses or puzzle rooms depending
    // on how deep in the dungeon, surroundings, etc..
    // May also influence whether secret doors are created to close off
    // some paths.
    //
    // Best paths:
    // -dead end infill rooms with a feature room nearby  Or just deadend infill rooms.
    // -starting multiplier = 1
    // -chance of making stairs = length/maxlength * multiplier
    //
    // If we still have 0 stairs then we'll look at
    // farthest feature rooms

    def stairs = [];
    def skip = [] as Set;
    def sortedPaths = new ArrayList(gen.pathIndex.values());
    sortedPaths.sort { p1, p2 ->
        int test = -(p1.length <=> p2.length);
        if( test != 0 ) {
            return test;
        }
        test = p1.room.loc.x <=> p2.room.loc.x;
        if( test != 0 ) {
            return test;
        }
        test = p1.room.loc.z <=> p2.room.loc.z;
        if( test != 0 ) {
            return test;
        }
        return test;
    }


    sortedPaths.each {
        log.info(" length:" + it.length + "  dead end:" + it.deadEnd + "  depth:" + it.deadEndDepth);
    }

    double chanceMultiplier = 0.99; // always some chance of failure even for the best one
    double falloff = 0.25; // higher values will create more stairs per level
    for( def p : sortedPaths ) {
        if( skip.contains(p) ) {
            continue;
        }
        //if( p.room.style == GlobalTypes.defaultStyle ) {
        if( p.room.type == 0 ) {
            // Only interested in infill rooms this pass
            continue;
        }
        double chance = ((double)p.length / maxInfill.length) * chanceMultiplier;
        if( !p.deadEnd ) {
            chance = chance * 0.75; // prefer dead-ends
        }
        double roll = gen.rand.nextDouble();
        if( roll < chance ) {
            stairs.add(p);
            chanceMultiplier *= falloff;
            if( p.deadEnd ) {
                for( def r = p.parent; r != null; r = r.parent ) {
                    if( r.deadEnd ) {
                        skip.add(r);
                    } else {
                        break;
                    }
                }
            }
        }
    }
    log.info("Found " + stairs.size() + " stairs");

    if( stairs.isEmpty() ) {
        // Just add the longest path for now
        stairs.add(sortedPaths[0]);
    }

    //if( debug ) {
    //    for( def p : gen.pathIndex.values() ) {
    //        def room = p.room;
    //
    //        Vec3d base = new Vec3d((room.loc.x * 2) + room.size.x, room.loc.y + room.size.y, (room.loc.z * 2) + room.size.z);
    //        for( int y = 0; y <= p.length; y++ ) {
    //            world.setWorldCell(base.add(globalOffset).add(0, y, 0), GlobalTypes.woodPlank);
    //        }
    //        if( p == maxInfill ) {
    //            world.setWorldCell(base.add(globalOffset).add(0, p.length+1, 0), GlobalTypes.redLight);
    //        }
    //        if( p == maxFeature ) {
    //            world.setWorldCell(base.add(globalOffset).add(0, p.length+1, 0), GlobalTypes.whiteLight);
    //        }
    //        if( p.deadEnd ) {
    //            world.setWorldCell(base.add(globalOffset).add(1, 0, 0), GlobalTypes.tile);
    //        }
    //        if( stairs.contains(p) ) {
    //            world.setWorldCell(base.add(globalOffset).add(0, 0, 1), GlobalTypes.tile);
    //        }
    //        for( int y = 0; y < p.deadEndDepth; y++ ) {
    //            world.setWorldCell(base.add(globalOffset).add(-1, y, 0), GlobalTypes.redLight);
    //        }
    //    }
    //}

    return stairs;
}

decorateRoom = { world, offset, room, config ->
    
    int xs = room.size.x * 2;
    int ys = room.size.y;
    int zs = room.size.z * 2;
    
    for( int x = -1; x <= xs; x++ ) {
        for( int z = -1; z <= zs; z++ ) {
            for( int y = -1; y <= ys; y++ ) {
                config(world, offset, x, y, z);
            }
        }  
    }  
    
}

class WallInfo {
    public int perimeterIndex;
    public int wallIndex;
    public int wallLength;
    public int wallHeight;
    public double wallPercent;
    public Vec3i localPos;
    public Vec3i dir;
    public Vec3i normal;
}

decorateRoomPerimeter = { world, offset, room, config ->

    int xMin = room.loc.x * 2;
    int yMin = room.loc.y;
    int zMin = room.loc.z * 2;     
    int xs = room.size.x * 2;
    int ys = room.size.y;
    int zs = room.size.z * 2;
 
    // Each of the four sides starting with 'origin' -> x
    int index = 0;
    def info = new WallInfo();
    info.dir = new Vec3i(1, 0, 0);
    info.normal = new Vec3i(0, 0, 1);
    info.wallHeight = ys;    
    info.wallLength = xs;
    info.localPos = new Vec3i(0, 0, 0);
    Vec3i base = new Vec3i((int)offset.x + xMin, (int)offset.y + yMin, (int)offset.z + zMin);
    for( int x = 0; x < xs; x++ ) {
        info.perimeterIndex = index++;
        info.wallIndex = x;
        info.wallPercent = info.wallIndex / (info.wallLength - 1);
        info.localPos.set(x, 0, 0);
        config(world, base.add(info.localPos), info); 
    }
    info.dir = new Vec3i(0, 0, 1);
    info.normal = new Vec3i(-1, 0, 0);    
    info.wallLength = zs;
    for( int z = 0; z < zs; z++ ) {
        info.perimeterIndex = index++;
        info.wallIndex = z;
        info.wallPercent = info.wallIndex / (info.wallLength - 1);
        info.localPos.set(xs - 1, 0, z);
        config(world, base.add(info.localPos), info); 
    }
    info.dir = new Vec3i(-1, 0, 0);
    info.normal = new Vec3i(0, 0, -1);    
    info.wallLength = xs;
    for( int x = xs - 1; x >= 0; x-- ) {
        info.perimeterIndex = index++;
        info.wallIndex = xs - 1 - x;
        info.wallPercent = info.wallIndex / (info.wallLength - 1);
        info.localPos.set(x, 0, zs - 1);
        config(world, base.add(info.localPos), info); 
    }
    info.dir = new Vec3i(0, 0, -1);
    info.normal = new Vec3i(1, 0, 0);    
    info.wallLength = zs;
    for( int z = zs - 1; z >= 0; z-- ) {
        info.perimeterIndex = index++;
        info.wallIndex = zs - 1 - z;
        info.wallPercent = info.wallIndex / (info.wallLength - 1);
        info.localPos.set(0, 0, z);
        config(world, base.add(info.localPos), info); 
    }    
}



