/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

import mythruna.assembly.io.SubassemblyProtocol;

// Could also be a getter on the player object type
getBlueprints = { parent ->
    // Get the blueprints as a tree set just so they are ordered by
    // ID... which is roughly by creation time.
    def id = resolveEntityId(parent);   
    def filter1 = BlueprintInfo.parentFilter(id); 
    def blueprints = findEntities(filter1, BlueprintInfo) as TreeSet;
              
    def filter2 = AssemblyBlueprintInfo.parentFilter(id);
    blueprints.addAll(findEntities(filter2, AssemblyBlueprintInfo));
    
    return blueprints;
}

createType("ObjectTool").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "The Object Tool is used for placing and updating objects.",
               "",
               "Clicking the right mouse button in an open location will",
               "place the current object type centered over that location.",
               "",
               "The mouse wheel will switch the object type.  The current",
               "object type is displayed in the bottom middle of the screen.",
               "",
               "Clicking the right mouse button on an existing object will",
               "open its action menu.",
               "",
               "Press the right mouse button down on an object and drag to",
               "move it.",
               "While dragging an object, the mouse wheel will rotate the",
               "object.",
               "",
               "New object types can be created in the blueprint editor",
               "accessible from the player menu (tab).",
             );

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/etcher.blocks", standardItemScale * 0.5)); 
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    addAction("onEquip") { ->
        log.info("Subclass equip");

        // Need to run the real equip
        run(type("BaseItem"), "Equip");

        def blueprints = getBlueprints(activator);
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }

        // See if the one we have is valid
        def focus = target[ObjectFocus]?.entityId;
        if( focus && blueprints.contains(focus) ) {
            // We're done... the current focus is valid.
        } else {
            // Just use the first item
            focus = blueprints.find();
            target << ObjectFocus.entity(focus);
        }

        env.setVar("selectedBlueprint", focus);

        return true;
    }
    
    addAction("place") { blockHit ->
        Vec3d loc = blockHit.location;
        log.info("place:" + loc);

        if( !getPerms(activator, loc).canAddObject() ) {
            echo("You do not have permission to place objects here.");
            return;
        }

        def bp = env.getVar("selectedBlueprint", null);
        if( bp == null ) {
            // Not our job to set it
            return false;
        }
        log.info("" + activator + " creating object at:" + loc + " with blueprint:" + bp);

        def shape = bp[ShapeInfo];

        def asmInfo = bp[AssemblyBlueprintInfo];
        if( asmInfo == null ) {
            // Assume it's a regular placeable

            // Need to get the size so we can find its base's center.
            // We probably should keep this as part of the object type or even a separate
            // component.
            def cells = shape.loadCells();
            def size = cells.size;
            def offset = size * 0.5 * shape.scale;
            offset.y = 0;
            log.info("cells:" + cells + "  offset:" + offset);

            loc -= offset;

            def pos = new SpawnPosition(loc, new Quatd());
            def name = bp[ObjectName];
            def placed = createEntity(
                    pos, shape, name,
                    new BlueprintSource(bp),
                    new CreatedBy(activator.id)
                );
        } else {
            // Else this is an assembly so make sure to make it at least
            // a static object and to give it the right type 
            def pos = new SpawnPosition(loc, new Quatd());
            def name = bp[ObjectName];
            def placed = createEntity(
                    pos, shape, name,
                    new BlueprintSource(bp),
                    ObjectTypeInfo.create("BaseAssembly", entityData),
                    new CreatedBy(activator.id),
                    new Mass(0)
                );
        }
        return true;
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);

        def blueprints = getBlueprints(activator);
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }

        // See what the current index is
        def selected = env.getVar("selectedBlueprint", null);
        int index = blueprints.findIndexOf { it == selected };
        index = (index + delta) % blueprints.size();
        if( index < 0 ) {
            index += blueprints.size();
        }
        selected = blueprints[index];
        target << ObjectFocus.entity(selected);
        env.setVar("selectedBlueprint", selected);
    }

    addAction("contextAction") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;

        log.info("contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";

        // Here we want to let the player run some actions on the object

        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        def owner = clicked[OwnedBy]?.owner;
        log.info("Player:" + activator.id + " creator:" + creator + " owner:" + owner);
        boolean isOurs = activator.isSameEntity(creator) || activator.isSameEntity(owner);
        boolean canRemove = isOurs || (creator == null && owner == null); 

        // Assemble the action list
        def options = [];
        def clickedContext = clicked.context;
        log.info("Checking actions for type:" + clicked.type);        
        for( def action : clicked.type.contextActions ) {
            if( !action.canRun(env, clickedContext) ) {
                log.info("Skipping:" + action);
                continue;
            }
            if( "Take".equals(action.getName()) ) {
                // We'll assume that if it can be 'taken' then deleting it
                // would be destructive.
                canRemove = false;
            }
            options.add(new Option(action.getName()));
        }

        // Can't copy the blueprint unless it has a shape
        def shape = clicked[ShapeInfo];
        boolean hasShape = shape != null;
        if( hasShape ) {
            def shapeName = ShapeName.parse(shape.getShapeName());
            if( "assembly" == shapeName.type || "asm" == shapeName.type ) {
                options.add(new Option(target, "Copy Design", "copyAssembly", clicked));
            } else {
                // For the blueprint related actions, we are the thing upon
                // which the action is run and the object is the parameter.
                options.add(new Option(target, "copyBlueprint", clicked));
            }
        }

        // Instead of hasShape... we should probably base this on
        // the actions that the object has.  That way we can have non-removable
        // objects that have shapes.  And then move "deleteObject" to the objects
        // that support delete... the others can echo a message that they can't
        // be removed.
        def objPos = clicked[SpawnPosition]?.location;
        if( canRemove && hasShape && getPerms(activator, objPos).canRemoveObject() ) {
            // Options to call us back as the manipulator
            options.add(new Option(target, "deleteObject", clicked));
        }

        // If it's our own object then allow us to make it solid or not
        // or to make it a physics object.
        if( isOurs && canRemove ) { //activator.isSameEntity(creator) ) {
            Mass mass = clicked[Mass];
            if( mass == null ) {
                options.add(new Option(target, "makeSolid", clicked));
                options.add(new Option(target, "makePhysical", clicked));
            } else if( mass.getMass() == 0 ) {
                options.add(new Option(target, "makeIllusion", clicked));
                options.add(new Option(target, "makePhysical", clicked));
            } else {
                options.add(new Option(target, "makeIllusion", clicked));
                options.add(new Option(target, "makeStatic", clicked));
            }
        }

        session.showPrompt(clicked, PromptType.List, name);
        session.showOptions(clicked, options);
    }

    // Targetted context operations
    //-----------------------------------------------
    addAction("copyBlueprint") { clicked ->
        log.info("copyBlueprint:" + clicked);

        def shape = clicked[ShapeInfo];
        def bpSource = clicked[BlueprintSource];
        def name = clicked.name;
        if( name == null ) {
            if( bpSource ) {
                name = bpSource.blueprintId.name?:"Unknown";
            } else {
                name = "Unknown";
            }
        }
        
        def cells = shape.loadCells();

        // Load the cells just so we can get a proper offset
        def offset = cells.size * 0.5;
        offset.x = 5 - offset.x;
        offset.y = 0;
        offset.z = 5 - offset.z;
        offset = offset.floor(); // back to vec3i

        log.info("name:" + name + "  shape:" + shape + "  offset:" + offset);

        if( bpSource ) {
            createEntity(
                        new BlueprintInfo(activator.id, name + " Copy", offset),
                        ObjectName.create(name),
                        shape,
                        // We'll make this player the creator but also
                        // set the blueprint source so we can still track the
                        // original author.  This player is likely to make changes
                        // the blueprint and we won't change the creator when that
                        // happens... so we need to change it now.
                        new CreatedBy(activator.id),
                        bpSource
                        );
        } else {
            // Assume we have to create the blueprint manually
            createBlueprintFromCells(activator.id, name + " Copy", name, cells, false, shape.scale);
        }
    }

    addAction("copyAssembly") { clicked ->
        log.info("copyAssembly:" + clicked);

        def shape = clicked[ShapeInfo];
        def bpSource = clicked[BlueprintSource];
        def shapeName = ShapeName.parse(shape.getShapeName());
 
        def originalCreator = null;
        if( bpSource ) {
            originalCreator = bpSource.blueprintId[CreatedBy]?.creatorId;
        }
        //echo("Original creator:" + originalCreator + "  us:" + activator.id);
        log.info("Original creator:" + originalCreator + "  us:" + activator.id);
 
        def blueprints = getBlueprints(activator); 
        def index = [:];
        blueprints.each { bp ->
            index.put(bp[ShapeInfo].getShapeName(), bp);
        }
        
        // If this player already has this design then tell them
        if( index.get(shapeName.toCompositeString()) ) {
            echo("You already have this design.");
            return;
        }
        
        def assembly = loadSubassembly(shapeName);
        // Clone it just in case
        assembly = SubassemblyProtocol.fromBytes(SubassemblyProtocol.toBytes(assembly));
        
        boolean assemblyChanged = false; 
        log.info("Found:" + assembly);
        def children = assembly.findAllShapeAssemblies(); 
        children.each { child -> 
            // See if we already have a blueprint for this shape            
            def existing = index.get(child.shapeName.toCompositeString());
            if( existing ) {
                // Then we're good... we already have it
                return;
            }   
 
            log.info("Copying child blueprint for:" + child.name + " " + child.shapeName.fullName + "  existing:" + existing);
            //echo("Copying child blueprint for:" + child.name);
                        
            // See if there are already existing blueprints for it somewhere.
            // Note: objects can exist in the world using old blueprints.  If the author has
            // edited the blueprint in some way then this loop will no longer find it.
            // That's ok... we'll make do.
            def best = null; 
            findEntities(ShapeInfo.shapeNameFilter(child.shapeName.toCompositeString()), ShapeInfo.class, BlueprintInfo.class).each { 
                log.info("Existing blueprint:" + it);
                //echo("Existing blueprint:" + it);
                if( best == null ) {
                    // Take at least the first of whatever we find
                    best = it;
                }                    
                // But we would prefer the blueprint by the same creator as the assembly
                if( originalCreator == it[BlueprintInfo].parent ) {
                    log.info("Same creator:" + originalCreator);
                    //echo("Same creator:" + originalCreator);
                } 
            }
            log.info("Best:" + best);
            //echo("Best:" + best);
            
            // Here we have one of the following conditions:
            // 1) we found the original blueprint from the same author
            // 2) we found 'some' matching blueprint from another author
            // 3) we found no blueprint because the original blueprint has been edited.
            // 4) we found no blueprint because this is a resource-based object or
            //      some other type of object that was not a blueprint.
            if( best != null ) {
                // case (1) or (2)
                def on = best[ObjectName];
                if( on == null ) {
                    if( child.name ) {
                        on = ObjectName.create(child.name);
                    } else {
                        on = ObjectName.create(best[BlueprintInfo].name); 
                    }
                }
                def bpInfo = best[BlueprintInfo].changeParent(activator.id);
                if( !bpInfo.name.endsWith(" Copy") ) {
                    bpInfo = bpInfo.changeName(bpInfo.name + " Copy") 
                }
                
                createEntity(
                        bpInfo,
                        on,
                        best[ShapeInfo],
                        // We'll make this player the creator but also
                        // set the blueprint source so we can still track the
                        // original author.  This player is likely to make changes
                        // the blueprint and we won't change the creator when that
                        // happens... so we need to change it now.
                        new CreatedBy(activator.id),
                        new BlueprintSource(best)
                        );
                echo("Added blueprint:" + bpInfo.name);                        
            } else { 
                // Have to manufacture a blueprint from raw data, basically
                def cells = loadCellsForName(child.shapeName.toCompositeString());
                def childName = child.name;
                if( !childName ) {
                    childName = child.shapeName.name;
                    int split = childName.lastIndexOf("/");
                    if( split >= 0 ) {
                        childName = childName.substring(split + 1);
                    }
                }
                boolean carved = child.shapeName.name.startsWith("c_") || child.shapeName.name.endsWith(".carved");
                //log.info("child name:" + childName + "  cells:" + cells + " carved:" + carved);
                def newBp = createBlueprintFromCells(activator.id, childName + " Copy", childName, 
                                                     cells, carved, 0.25);
                echo("Created new blueprint:" + childName + " Copy"); 
                                                     
                def newShapeName = ShapeName.parse(newBp[ShapeInfo].shapeName);
                //log.info("new shape name:" + newShapeName);
                
                child.shapeName = newShapeName;
                assemblyChanged = true;
            }
        }

        def name = clicked.name;
        if( name == null ) {
            if( bpSource ) {
                name = bpSource.blueprintId.name?:"Unknown";
            } else {
                name = "Unknown";
            }
        }

        if( assemblyChanged || shapeName.type != "asm" ) {
            // Then we need to write the assembly out and get its new ID
            def subId = worldManager.getSubassemblyStorage().store(assembly);
            log.info("Stored changed assembly as:" + subId);
            shape = ShapeInfo.create(subId.toFileName(), 0.25);
        }

        // Now that we've copy the children we can copy the main assembly
        if( bpSource ) {
            createEntity(
                    new AssemblyBlueprintInfo(activator.id, name + " Copy"),
                    ObjectName.create(name),
                    shape,
                    // We'll make this player the creator but also
                    // set the blueprint source so we can still track the
                    // original author.  This player is likely to make changes
                    // the blueprint and we won't change the creator when that
                    // happens... so we need to change it now.
                    new CreatedBy(activator.id),
                    bpSource
                    );
        } else {
            createEntity(
                    new AssemblyBlueprintInfo(activator.id, name + " Copy"),
                    ObjectName.create(name),
                    shape,
                    // We'll make this player the creator but also
                    // set the blueprint source so we can still track the
                    // original author.  This player is likely to make changes
                    // the blueprint and we won't change the creator when that
                    // happens... so we need to change it now.
                    new CreatedBy(activator.id)
                    );
        }                        
        echo("Added design:" + name + " Copy"); 
    }

    addAction("deleteObject") { clicked ->
        log.info("deleteObject:" + clicked);

        def objPos = clicked[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canRemoveObject() ) {
            echo("You do not have permission to remove objects here.");
            return;
        }
        // We should probably check to see if we can in case we hacked
        // in around the context menu
        removeEntity(clicked);
    }

    addAction("makeSolid") { clicked ->
        clicked << new Mass(0);
    }

    addAction("makePhysical") { clicked ->
        // Calculate the mass of the object

        // Set the mass
        clicked << new Mass(50);
    }

    addAction("makeIllusion") { clicked ->
        clicked.remove(Mass);
    }

    addAction("makeStatic") { clicked ->
        clicked << new Mass(0);
    }

}


