/*
 * $Id$
 *
 * Copyright (c) 2023, Simsilica, LLC
 * All rights reserved.
 */

/**
 *  Convenience APIs for querying things about the world.
 */

import mythruna.world.tile.*;
import mythruna.world.town.*;

sedectileManager = worldManager.sedectileManager;

getPointOfInterest = { EntityId poiEntity ->
    def featureId = new FeatureId(PointOfInterest.class, poiEntity.getId());
    return sedectileManager.getFeatures().getFeature(featureId, PointOfInterest.class);
}

findPointsOfInterest = { location, radius ->
    //log.info("findPointsOfInterest(" + location + ", " + radius + ")");
    //log.info("-------------------------------------------");
    def results = [];
    int tileRadius = (int)Math.ceil(radius/TileId.SIZE);
    //log.info("tileRadius:" + tileRadius);
    TileId centerTileId = TileId.fromWorld(location);
    def visited = [] as Set;
    centerTileId.visitNeighbors(tileRadius) { tileId ->
        //log.info("tile:" + tileId);
        def sed = sedectileManager.getSedectile(tileId.getSedectileId());
        //log.info("sed:" + sed);
        for( FeatureId<PointOfInterest> id : sed.getFeatureIds(tileId, PointOfInterest.class) ) {
            if( !visited.add(id) ) {
                continue;
            }
            //log.info("    " + id);
            PointOfInterest poi = sedectileManager.getFeatures().getFeature(id, PointOfInterest.class);
            //log.info("     poi:" + poi.type + "  loc:" + poi.location + "  size:" + poi.radius + "  id:" + id);

            double distSq = location.distanceSq(poi.location);
            double r2 = poi.radius + radius;
            r2 *= r2;
            //log.info("   distanceSq:" + distSq + "  r2:" + r2);
            if( distSq <= r2 ) {
                //log.info("   hit");
                results.add(poi);
            }
        }
    }
    return results;
}

findCurrentPoi = { location ->
    def bestPoi = null;
    def biggestRadius = 0;
    def inside = findPointsOfInterest(location, 0);
    inside.each { poi ->
        //log.info("inside:" + poi);
        if( poi.radius > biggestRadius ) {
            biggestRadius = poi.radius;
            bestPoi = poi;
        }
    }
    return bestPoi;
}

findEmptyCell = { loc, maxRadius ->
    for( int radius = 1; radius <= maxRadius; radius++ ) {
        //log.info("  trying radius:" + radius);
        for( int x = -radius; x <= radius; x++ ) {
            boolean xEdge = x == -radius || x == radius;
            for( int y = -radius; y <= radius; y++ ) {
                boolean yEdge = y == -radius || y == radius;
                for( int z = -radius; z <= radius; z++ ) {
                    boolean zEdge = z == -radius || z == radius;
                    if( xEdge || yEdge || zEdge ) {
                        // Then we'll check
                        def local = loc.add(x, y, z);
                        int value = MaskUtils.getType(world.getWorldCell(local));
                        if( value == 0 ) {
                            return local;
                        }
                    }
                }
            }
        }
    }
    return null;
}

findRandomLocationInStructure = { structure ->
    log.info("findRandomLocationInStructure(" + structure + ")");
    def loc = structure[SpawnPosition]?.location;
    log.info("  base pos:" + loc);
    if( loc == null ) {
        return null;
    }
    def info = structure[TriggerInfo];
    log.info("  trigger:" + info);
    if( info ) {
        def xOffset = Math.random() * info.xSize;
        def zOffset = Math.random() * info.zSize;
        loc = loc.add(xOffset, 0, zOffset);
    }
    //log.info("starting location:" + loc);
    int value = MaskUtils.getType(world.getWorldCell(loc));
//    log.info(" cell:" + value);
    if( value != 0 ) {
        // Hunt for an empty cell working our way out
        loc = findEmptyCell(loc, 5);
    }
    return loc;
}

findInsertInfoEntity = { EntityId parent, InsertInfo target ->

    log.info("findInsertInfoEntity(" + target + ")");
    def children = findEntities(StructureInfo.parentFilter(parent), StructureInfo);
    for( EntityId child : children ) {
        def childLocation = child[SpawnPosition].location;
        log.info("   " + child[StructureInfo].getFactory(entityData) + " " + childLocation);
        if( target.contains(childLocation) ) {
            return child;
        }
    }
    return null;
}

