/*
 * Decompiled with CFR 0.152.
 */
package com.simsilica.thread;

import com.google.common.base.Throwables;
import com.simsilica.thread.Job;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkerPool {
    static Logger log = LoggerFactory.getLogger(WorkerPool.class);
    public static final int DEFAULT_PRIORITY = Integer.MAX_VALUE;
    private int poolSize;
    private ThreadPoolExecutor workers;
    private ConcurrentLinkedQueue<JobRunner> toFinish = new ConcurrentLinkedQueue();
    private ConcurrentHashMap<Job, Job> queuedJobs = new ConcurrentHashMap();
    private ConcurrentHashMap<Job, JobRunner> runnerIndex = new ConcurrentHashMap();
    private AtomicLong jobSequence = new AtomicLong(0L);
    private AtomicInteger activeCount = new AtomicInteger(0);
    private AtomicLong errorCount = new AtomicLong(0L);
    private boolean shuttingDown = false;

    public WorkerPool() {
        this(4);
    }

    public WorkerPool(int poolSize) {
        this.poolSize = poolSize;
        this.workers = new ThreadPoolExecutor(poolSize, poolSize, 0L, TimeUnit.MILLISECONDS, new PriorityBlockingQueue<Runnable>());
    }

    public int getPoolSize() {
        return this.poolSize;
    }

    public void execute(Job job) {
        this.execute(job, Integer.MAX_VALUE);
    }

    public void execute(Job job, int priority) {
        Job old = this.queuedJobs.putIfAbsent(job, job);
        if (old == null) {
            if (log.isTraceEnabled()) {
                log.trace("Queuing:" + job + "  at:" + priority);
            }
            JobRunner runner = new JobRunner(job, priority);
            this.runnerIndex.put(job, runner);
            this.workers.execute(runner);
        }
    }

    public boolean cancel(Job job) {
        JobRunner runner = this.runnerIndex.get(job);
        if (runner == null) {
            if (log.isTraceEnabled()) {
                log.trace("Unknown job:" + job);
            }
            return false;
        }
        if (log.isTraceEnabled()) {
            log.trace("Attempting to cancel:" + job);
        }
        if (this.workers.getQueue().remove(runner)) {
            this.queuedJobs.remove(job);
            this.runnerIndex.remove(job);
            if (log.isTraceEnabled()) {
                log.trace("Job canceled:" + job);
            }
            return true;
        }
        if (log.isTraceEnabled()) {
            log.trace("Job no longer in queue:" + job);
        }
        return false;
    }

    public boolean isQueued(Job job) {
        return this.queuedJobs.containsKey(job);
    }

    public int getQueuedJobCount() {
        return this.queuedJobs.size();
    }

    public int getActiveJobCount() {
        return this.activeCount.get();
    }

    public boolean isBusy() {
        return this.getActiveJobCount() + this.getQueuedJobCount() > 0;
    }

    public void shutdownNow(boolean awaitTermination) {
        this.shuttingDown = true;
        this.workers.shutdownNow();
        if (awaitTermination) {
            log.info("Waiting for thread pool shutdown");
            try {
                this.workers.awaitTermination(10000L, TimeUnit.DAYS);
            }
            catch (InterruptedException e) {
                throw new RuntimeException("Interrupted waiting for shutdown", e);
            }
        }
    }

    public boolean isRunning() {
        return !this.workers.isShutdown();
    }

    public double update(double maxWork) {
        JobRunner job = null;
        double totalWork = 0.0;
        while ((job = this.toFinish.poll()) != null) {
            if (log.isTraceEnabled()) {
                log.trace("Finishing job:" + job.job + " at priority:" + job.priority);
            }
            try {
                double work = job.job.runOnUpdate();
                totalWork += work;
                if (!(maxWork >= 0.0) || !(totalWork >= maxWork)) continue;
                break;
            }
            catch (RuntimeException e) {
                this.errorCount.incrementAndGet();
                throw e;
            }
            finally {
                this.activeCount.decrementAndGet();
            }
        }
        return totalWork;
    }

    private class JobRunner
    implements Runnable,
    Comparable<JobRunner> {
        private Job job;
        private int priority;
        private long jobId;

        public JobRunner(Job job, int priority) {
            this.jobId = WorkerPool.this.jobSequence.getAndIncrement();
            this.job = job;
            this.priority = priority;
        }

        @Override
        public int compareTo(JobRunner other) {
            if (this.priority < other.priority) {
                return -1;
            }
            if (this.priority > other.priority) {
                return 1;
            }
            if (this.jobId < other.jobId) {
                return -1;
            }
            if (this.jobId > other.jobId) {
                return 1;
            }
            return 0;
        }

        @Override
        public void run() {
            WorkerPool.this.activeCount.incrementAndGet();
            WorkerPool.this.queuedJobs.remove(this.job);
            WorkerPool.this.runnerIndex.remove(this.job);
            if (log.isTraceEnabled()) {
                log.trace("Running background job:" + this.job + " at priority:" + this.priority);
            }
            try {
                this.job.runOnWorker();
            }
            catch (Exception e) {
                if (WorkerPool.this.shuttingDown && Throwables.getRootCause((Throwable)e) instanceof InterruptedException) {
                    log.info("Thread interrupted successfully");
                } else {
                    log.error("Error running job:" + this.job, (Throwable)e);
                    WorkerPool.this.activeCount.decrementAndGet();
                    WorkerPool.this.errorCount.incrementAndGet();
                }
                return;
            }
            if (log.isTraceEnabled()) {
                log.trace("Job runOnWorker() done:" + this.job);
            }
            WorkerPool.this.toFinish.add(this);
        }
    }
}

