/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

// Some essentially temporary test objects for testing inventory, item
// holding, item manipulation, and other test things.


import com.simsilica.mworld.tile.Resolution;
import com.simsilica.mworld.tile.tree.*;

findTree = { Vec3i block ->
    def tileId = TileId.fromWorld(block);
    def treeTile = world.getTrees(tileId, null);
    def tileLoc = block - tileId.getWorld(null);
    //treeTile.getTrees(block).each {
    //}
    return treeTile.getTrees(block).find { tree -> tree.x == tileLoc.x && tree.z == tileLoc.z }
}

createType("Axe").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Swing");

    setTypeVar("description",
        "This was axe was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(6, 0, 5));

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/axe1.blocks", standardItemScale));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    addAction("mainPressBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("start chopping:" + block);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("chop:" + loc);

        int cell = world.getWorldCell(block as Vec3d);
        int type = MaskUtils.getType(cell);
        int sides = MaskUtils.getSideMask(cell);

        def genType = CellGenType.getType(cell);

        echo("Chop:" + genType);

        def found = findTree(block);
        echo("Tree:" + found);

        // The below would be nearly 100% accurate but would require
        // straddling columns to get the full tree and requires shuffling
        // lots of additional leaf data around just to pull out a single
        // tree.  Instead, we will let TreeType guess if block type is
        // part of it or not.
        //def colId = ColumnId.fromWorld(block);
        //def colData = new ColumnData(colId, GameConstants.MAX_BUILD_HEIGHT / 32);
        //
        //def tileId = TileId.fromWorld(block);
        //def treeTile = world.getTrees(tileId, null);
        //
        //def localTrees = treeTile.getTrees(colId);
        //if( localTrees.isEmpty() ) {
        //    return false;
        //}
        //
        //Random rand = new Random(columnId.getId()); // consistent random per column
        //def cells = new TileColumnCells(colData);
        //for( Tree tree : localTrees ) {
        //    //log.info("  insert:" + tree + " into:" + col.getColumnId());
        //    if( tree == found ) {
        //        // Clear everything so far
        //        colData.leafs.each { leaf ->
        //            leaf.rawCells.clear();
        //        }
        //    }
        //    tree.type.insertTree(tree, cells, rand);
        //    if( tree == found ) {
        //        break;
        //    }
        //}

        // Guess which blocks are part of this tree
        def tileId = TileId.fromWorld(block);
        def tileLoc = block - tileId.getWorld(null);

        int size = 1 + found.radius * 2;
        int xBase = found.x - found.radius;
        int yBase = found.y;
        int zBase = found.z - found.radius;

        CellArray cells = new CellArray(size, found.height, size);

        Vec3d base = new Vec3d(xBase, yBase, zBase);
        //log.info("base:" + base);
        base = base + tileId.getWorld(null);
        //log.info("world base:"+ base);

        for( int y = 0; y < found.height; y++ ) {
            for( int z = 0; z < size; z++ ) {
                def sb = new StringBuilder();
                for( int x = 0; x < size; x++ ) {
                    cell = world.getWorldCell(base.add(x, y, z));
                    genType = CellGenType.getType(cell);
                    if( genType == CellGenType.Flora ) {
                        type = MaskUtils.getType(cell);
                        if( found.type.isBlock(found, x - found.radius, y, z - found.radius, type) ) {
                            //sb.append("[1]");
                            cells.setCell(x, y, z, type);
                        } else {
                            //sb.append("[0]");
                        }
                    } else {
                        //sb.append("[0]");
                    }
                }
                //log.info("[" + y + "][" + z + "] = " + sb);
            }
        }

        // Make sure the cells have accurate masks
        MaskUtils.calculateSideMasks(cells);

        //createBlueprintFromCells(activator.id, "Tree:" + block.x + "x" + block.z, "Tree", cells, false, 0.25);
        createBlueprintFromCells(activator.id, "Tree:" + block.x + "x" + block.z, "Tree", cells, false, 1);
    }


    addAction("altClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        echo("Loc:" + loc + " block:" + block);

        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);

        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;

        def tree = blueprints.find { it.name.startsWith("Tree") };

        echo("Tree blueprint:" + tree);

        def bp = tree;

        def shape = bp[ShapeInfo];

        // Need to get the size so we can find its base's center.
        // We probably should keep this as part of the object type or even a separate
        // component.
        def cells = shape.loadCells();

        def worldEdits = new WorldEdits();

        for( int x = 0; x < cells.sizeX; x++ ) {
            for( int z = 0; z < cells.sizeZ; z++ ) {
                for( int y = 0; y < cells.sizeY; y++ ) {
                    int val = cells.getCell(x, y, z);
                    world.setWorldCell(loc.add(x, y, z), val, worldEdits);
                }
            }
        }

        world.deliverEvents(worldEdits);

        echo("World implemention:" + world.getClass() + "  thread:" + Thread.currentThread());

        return true;

    }

    addAction("startSwing") { facing ->
        log.info("startSwing:" + facing);
    }

    addAction("swing") { facing ->
        log.info("swing:" + facing);
    }

    addAction("stopSwing") { facing ->
        log.info("stopSwing:" + facing);
    }

    addAction("onRemove") { ->
        log.info("onRemove()");

        def on = self[ObjectName];
        log.info("removing:" + on + " " + on.name + " self:" + self.name);
        if( "Test Axe" == on.name ) {
            log.info("calling findEntities()");
            int count = findEntities(Filters.fieldEquals(ObjectName.class, "nameId", on.nameId)).size();
            log.info("test axe count:" + (count - 1));
            echo("" + (count - 1) + " test axes remaining.");
        }

        log.info("calling superRun()");
        superRun();
        log.info("done calling superRun()");
    }
}

createType("Sword").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Swing");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/sword1.blocks", standardItemScale));
    // The sword requires some custom sizing to make sense in inventory
    def size = getTypeVar("shape").loadCells().size.toVec3d();
    size.x = 3;
    size *= standardItemScale;
    setTypeVar("volume", new ObjectVolume(size));

    setTypeVar("description",
        "This %name was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(3, 1, 4));
    setTypeVar("handleRotation", new Quatd().fromAngles(0, 0, -Math.PI * 0.5));

    addAction("startSwing") { facing ->
        log.info("startSwing:" + facing);
    }

    addAction("swing") { facing ->
        log.info("swing:" + facing);
    }

    addAction("stopSwing") { facing ->
        log.info("stopSwing:" + facing);
    }
}

createType("Book").with {
    supertypes("BaseItem");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/book.blocks", standardItemScale));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    setTypeVar("description",
        "This was book was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(0, 4, 1));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, Math.PI, 0));
}


createType("TestObject").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
        "This was object was created by %creator with ID %id.");

    addAction("Test") { ->
        //log.info("" + this + "." + getName() + "(" + Arrays.asList(args) + ")");
        log.info("" + this + "." + "Test" + "(" + ")");
        log.info("owner:" + owner + " delegate:" + delegate);
        log.info("Activator:" + activator);
        log.info("Activator:" + getActivator());
        //log.info("variables:" + getVariables()); // from 'delegate'
        log.info("activator.vars:" + activator.vars); // from 'delegate'
        log.info("target:" + getTarget()); // from 'this'

        // This goes to a global binding
        testProperty = 123;

        //log.info("variables:" + getVariables()); // from 'delegate'

        //activator.target = "Test Target";
        //log.info("activator.vars:" + activator.vars); // from 'delegate'
        //log.info("target:" + getTarget()); // from 'this'?  Yep.

        //log.info("vars:" + activator.vars);
        //log.info("test:" + activator.vars.get("selectedType"));
        //log.info("test1:" + activator.getProperty("selectedType"));
        //log.info("test2:" + activator.selectedType);
        //activator.foo = 123;
        //log.info("test3:" + activator.foo);
        return true;
    }

    addAction("Test2") { String name ->
        log.info("Test2:" + name);
        return true;
    }

    addAction("Test3") { ->
        log.info("Calling delegate version of run()");
        // This one calls it directly on ActionContext which is our 'this'
        //object.run(env, "Test2", "This is a test")
        object.run("Test2", "This is a test")
        log.info("Calling non-delegate version of run()");
        run("Test2", "This is a different test")
        //run("BaseTest")

        def manipulator = object.getType().getTypeVar("manipulator", "Default");

        log.info("Manipulator:" + manipulator);
        log.info("test:" + (object.manipulator?:"Default"));

        log.info("holding1:" + Holding.create(object.getTarget(), manipulator, entityData));
        log.info("holding2:" + Holding.create(object.getTarget(), manipulator));
        log.info("holding3:" + Holding.create(object, manipulator));
        log.info("holding4:" + Holding.create(activator, manipulator));

        log.info("contains1:" + activator.contains(object));

        // What about running an action from a different type or one of
        // our super types?
        log.info("Running it the hard way--------------------");
        type("BaseObject").findAction("Look").run(env, context);

        log.info("Running it the easy way--------------------");
        run(type("BaseObject"), "Look");

        return true;
    }

    addGetter("test") { -> //target ->
        log.info("getter-test");
        log.info("  getter-ed:" + system(EntityData));
        log.info("  getter-target:" + target);
        log.info("  getter-activator:" + activator);
        log.info("  owner:" + owner);
        log.info("  delegate:" + delegate);
        log.info("  this:" + this);
        return "foo-123"
    }

    //addGetter("name") { ->
    //    log.info("target:" + target);
    //    log.info("delegate:" + delegate);
    //    def on = delegate[ObjectName];
    //    log.info("name test:" + on);
    //    log.info("delegate test:" + delegate[ObjectName]);
    //    if( on != null ) {
    //        return on.getName(entityData);
    //    }
    //    def n = delegate[Name];
    //    if( n != null ) {
    //        return n.getName();
    //    }
    //    return "Entity@" + target.getId();
    //}

}

createType("TestWand").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "A Test Wand."
             );

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/metal-wand.blocks", standardItemScale * 0.5));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));
    setTypeVar("modes", ["Sound", "Random Human", "Random Human Male", "Random Human Female", "Gaefen", "Town Info"])

    addAction("showDefaultBlockAction")  { blockHit ->
        //log.info("base-objects: showDefaultBlockAction:" + blockHit);
        session.showPrompt(null, PromptType.Flyover, "Climate Info");
    }

    // Note: onEquip is not called when we first load the game and the tool is already
    // equipped.  So we really need some other action that can be invoked in both cases.
    addAction("onEquip") { ->
        log.info("Subclass equip");

        // Need to run any super-class onEquip... but our current version
        // of this will fail if the action is not defined
        //run(type("BaseItem"), "onEquip");

        // Set our focus
        run("rotate", 0);
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);

        def modes = getTypeVar("modes", null);
        int index = env.getVar("testWandFocus", 0);
        index = (index + delta) % modes.size();
        if( index < 0 ) {
            index += modes.size();
        }
        def focus = modes[index];
        target << ObjectFocus.stringValue(focus, entityData);
        env.setVar("testWandFocus", index);
        env.setVar("testWandMode", focus);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        echo("Clicked:" + block)

        //double temperature = worldManager.worldFractal.getTemperature(block.x, block.y, block.z);
        //double weather = worldManager.worldFractal.getWeather(block.x, block.y, block.z);
        //double ecology = worldManager.worldFractal.getEcology(block.x, block.y, block.z);
        //
        //// Adjust temperature for elevation
        ////int seaLevel = 128;
        ////int treeLine = seaLevel + 490;
        ////double height = Math.abs(block.y - seaLevel) / (double)treeLine;
        ////echo("height scale:" + height);
        ////double baselineHeight = 0.2;
        ////double effectiveTemperature = temperature;
        ////if( height > baselineHeight ) {
        ////    double heightEffect = (height - baselineHeight) / (1.0 - baselineHeight);
        ////    double temperatureAdjust = heightEffect * 0.4;
        ////    effectiveTemperature = temperature - temperatureAdjust;
        ////}
        //double effectiveTemperature = worldManager.worldFractal.getAdjustedTemperature(block.x, block.y, block.z);
        //
        //// Adjust precipiation by temperature if we are in particularly
        //// hot climates.  Anything above 40C is desert and will significantly
        //// affect whether precipitation accumulates.
        //// 40C in the range of -20 to 50 should be 0.857
        ////double effectiveWeather = weather;
        ////if( effectiveTemperature > 0.857 ) {
        ////    double weatherAdjust = (effectiveTemperature - 0.857) / (1.0 - 0.857);
        ////    weatherAdjust = 1 - weatherAdjust;
        ////    // Weather adjust should be in 0..1 where 0 is no precip. and
        ////    // 1 is max precip
        ////    effectiveWeather *= weatherAdjust;
        ////}
        //double effectiveWeather = worldManager.worldFractal.getAdjustedWeather(block.x, block.y, block.z, effectiveTemperature);
        //
        ////// Temper ecology by how wet things are on average
        ////double effectiveEcology = ecology;
        ////if( effectiveWeather < 0.25 ) {
        ////    effectiveEcology *= (effectiveWeather / 0.25);
        ////} else if( effectiveWeather > 0.75 ) {
        ////    // Increase ecology, up to double
        ////    effectiveEcology *= 1 + ((effectiveWeather - 0.75)/0.25);
        ////    if( effectiveEcology > 1 ) {
        ////        effectiveEcology = 1;
        ////    }
        ////}
        //double effectiveEcology = worldManager.worldFractal.getAdjustedEcology(block.x, block.y, block.z, effectiveWeather);

        def info = worldManager.worldFractal.getBioInfo(block.x, block.y, block.z, null);

        echo("Temperature:" + info.baseTemperature + "  effective:" + info.temperature);
        echo("Weather:" + info.basePrecipitation + "  effective:" + info.precipitation);
        echo("Climate:" + info.climate + "  Soil Quality:" + info.soilQuality);
        echo("Vegetation:" + info.baseVegetationLevel + "  effective:" + info.vegetationLevel);

        int foliage = (int)Math.round(info.vegetationLevel * 3) & 0x3;
        int wetness = (int)Math.round(info.precipitation * 3) & 0x3;
        echo("foliage level:" + foliage + "/3   wetness level:" + wetness + "/3");

        def tileId = TileId.fromWorld(block.x, block.y, block.z);
        def terrain = world.getTerrainImage(tileId, TerrainImageType.Terrain, null);
        def origin = tileId.getWorld(null);
        int type = terrain.getType((int)(block.x - origin.x), (int)(block.z - origin.z));
        byte light = terrain.getLight((int)(block.x - origin.x), (int)(block.z - origin.z));
        int baseType = TerrainTypes.getBaseType(type);
        echo("Encoded type:" + baseType + " foliage level:" + TerrainTypes.getFoliageLevel(type) + "/3 wetness level:" + TerrainTypes.getWetnessLevel(type) + "/3 frozen:" + TerrainTypes.isFrozen(type));

        double c = (info.temperature * 70) - 20;
        double f = c * 1.8 + 32;
        double p = info.precipitation * 100;
        echo(String.format("%.01f C (%.01f F)  Avg. rainfall: %.02f cm (%.02f inches)", c, f, p*2.5, p));

        // Show what the tree generator would use for type selection
        int TEMPERATURE_ZONES = 7;
        double VEG1 = 0.2;
        double VEG2 = 0.25;
        double VEG3 = 0.4;
        double VEG4 = 0.5;
        double VEG5 = 0.75;
        double VEG6 = 1;

        double PROB1 = 0.001;
        double PROB2 = 0.003;
        double PROB3 = 0.01;
        double PROB4 = 0.1;
        double PROB5 = 0.75;
        double PROB6 = 1.0;
        int treeLine = 490 + GameConstants.SEA_LEVEL;
        def density = "LOW";
        double chance = 0;
        double veg = 0;

        if( baseType == TerrainTypes.TYPE_SAND ) {
            chance = info.vegetationLevel * 0.001;
        } else {
            veg = info.vegetationLevel;

            if( veg < VEG1 ) {
                chance = (veg/VEG1);
                chance = chance * PROB1;
                density = "LOW";
            } else if( veg < VEG2 ) {
                chance = (veg - VEG1)/(VEG2 - VEG1);
                chance = PROB1 + (chance * (PROB2 - PROB1));
                density = "LOW";
            } else if( veg < VEG3 ) {
                chance = (veg - VEG2)/(VEG3 - VEG2);
                chance = PROB2 + (chance * (PROB3 - PROB2));
                density = "MEDIUM";
            } else if( veg < VEG4 ) {
                chance = (veg - VEG3)/(VEG4 - VEG3);
                chance = PROB3 + (chance * (PROB4 - PROB3));
                density = "MEDIUM";
            } else if( veg < VEG5 ) {
                chance = (veg - VEG4)/(VEG5 - VEG4);
                chance = PROB4 + (chance * (PROB5 - PROB4));
                density = "HIGH";
            } else if( veg < VEG6 ) {
                chance = (veg - VEG5)/(VEG6 - VEG5);
                chance = PROB5 + (chance * (PROB6 - PROB5));
                density = "HIGH";
            } else {
                chance = 1;
                density = "HIGH";
            }
        }
        if( block.y > treeLine ) {
            // only the very very smallest chance
            t = 0.0001 * info.soilQuality;
        }
        int zone = (int)(TEMPERATURE_ZONES * info.temperature);

        echo(String.format("Density: %s  Zone: %d  Chance: %.03f  Splat: %d", density, zone, chance, light));
    }

    addAction("place") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("place:" + loc);

        if( !getPerms(activator, loc).canAddObject() ) {
            echo("You do not have permission to place objects here.");
            return false;
        }

        def mode = env.getVar("testWandMode", "Gaefen");

        def stepTime = gameSystems.stepTime;

        def components = [];
        def entity = null;
        switch( mode ) {
            case "Sound":
                entity = type("Speakers").newInstance(
                        ObjectName.create("Speakers"),
                        SoundInfo.create("/Sounds/Effects/TestSounds.ogg", now(), 1, true),
                        new CreatedBy(activator.id),
                        Decay.seconds(32)
                    );
                entity.run("placeInWorld", loc, new Quatd());
                break;
            case "Gaefen":
                entity = type("Animal").newInstance(
                    ShapeInfo.create("animals/gaefen.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Gaefen", entityData),
                    new Mass(10),
                    new CreatedBy(activator.id),
                    AgentType.create("Gaefen", AgentType.LEVEL_HIGH, entityData),
                    Decay.seconds(120)
                );
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());

                //entity = createEntity(
                //    ShapeInfo.create("animals/gaefen.rig", 1),
                //    new SpawnPosition(loc, new Quatd()),
                //    ObjectName.create("Gaefen", entityData),
                //    new Mass(10),
                //    new CreatedBy(activator.id),
                //    AgentType.create("Gaefen", AgentType.LEVEL_HIGH, entityData),
                //    Decay.seconds(120)
                //);
                break;
            case "Random Human":
                entity = createRandomNpc(resolveEntityId(session), loc, new Random());
                break;
            case "Random Human Male":
                entity = createEntity(
                    ShapeInfo.create("human/male/human-male.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Human Male", entityData),
                    ObjectTypeInfo.create("NPC"),
                    Race.create("human-male", entityData),
                    new Mass(60),
                    new CreatedBy(activator.id),
                    AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)//,
                    //Decay.seconds(120)
                );
                //entity.run("resetDefaultAppearance");
                entity.run("randomize", new Random());
                break;
            case "Random Human Female":
                entity = createEntity(
                    ShapeInfo.create("human/female/human-female.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Human Female", entityData),
                    ObjectTypeInfo.create("NPC"),
                    Race.create("human-female", entityData),
                    new Mass(60),
                    new CreatedBy(activator.id),
                    AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)//,
                    //Decay.seconds(120)
                );
                //entity.run("resetDefaultAppearance");
                entity.run("randomize", new Random());
                break;
            //case "Town Info":
            //    testTownInfo.call(session, self, loc);
            //    break;
            default:
                echo("Unknown mode:" + mode);
                return false;
        }
        return true;
    }

    addAction("contextAction") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;

        log.info("test wand: contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";

        // Here we want to let the player run some actions on the object

        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        def owner = clicked[OwnedBy]?.owner;
        log.info("Player:" + activator.id + " creator:" + creator + " owner:" + owner);
        boolean isOurs = activator.isSameEntity(creator) || activator.isSameEntity(owner);
        boolean canRemove = isOurs || (creator == null && owner == null);

        // Assemble the action list
        def options = [];
        clicked.run("listActions") { action ->
            //echo("test:" + action);
            if( "Take".equals(action.getName()) ) {
                // We'll assume that if it can be 'taken' then deleting it
                // would be destructive.
                canRemove = false;
            }
            options.add(new Option(action.getName()));
        }

        // Instead of hasShape... we should probably base this on
        // the actions that the object has.  That way we can have non-removable
        // objects that have shapes.  And then move "deleteObject" to the objects
        // that support delete... the others can echo a message that they can't
        // be removed.
        // FIXME: use the runnable action list when we have it... ie: not all
        // actions, just the ones that canRun().
        def objPos = clicked[SpawnPosition]?.location;
        if( getPerms(activator, objPos).canRemoveObject() ) {
            // Options to call us back as the manipulator
            options.add(new Option(target, "deleteObject", clicked));
        }
        session.showPrompt(clicked, PromptType.List, name);
        session.showOptions(clicked, options);
    }

    addAction("deleteObject") { clicked ->
        log.info("deleteObject:" + clicked);

        def objPos = clicked[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canRemoveObject() ) {
            echo("You do not have permission to remove objects here.");
            return;
        }
        clicked.runIfExists("cleanup");

        // We should probably check to see if we can in case we hacked
        // in around the context menu
        log.info("removing:" + clicked);
        removeEntity(clicked);
    }
}

createType("TestChest").with {
    supertypes("BaseCloseable");
}

createType("TestDoor").with {
    supertypes("BaseCloseable");
}

createType("TestDesk").with {
    supertypes("BaseCloseable");
}


createType("Speakers").with {
    supertypes("BaseObject");

    setTypeVar("shape", ShapeInfo.create("/Models/objects/speakers.blocks", 0.25));

    addAction("Look") { ->
        def lines = [];

        lines += "Examine: " + object.name;

        lines += "Sound:" + object[SoundInfo]

        session.showPrompt(target, PromptType.Dialog, lines.join("\n\n"));
        session.showOptions(target, []);

        return true;
    }
}

npcRandom = new Random(0);
skinToneIndex = SkinToneIndex.create { id ->
    return worldManager.getTextDb().getText(id, null);
}
hairToneIndex = HairToneIndex.create { id ->
    return worldManager.getTextDb().getText(id, null);
}

characterNameGenerator = CharacterNameGenerator.create { id ->
    return worldManager.getTextDb().getText(id, null);
}

createType("QuestAxe").with {
    supertypes("Axe");

    addAction("Take") { ->
        log.info("Activator:" + activator + " picking up object " + object.name);

        //object <<
        // FIXME: hook up activator.id a little more consistently
        def backpack = findItem(activator.id, "Backpack");
        log.info("Putting it into backpack:" + backpack);

        def shape = object[ShapeInfo];

        // Need to find an empty spot or not allow it to be picked up
        object << new ContainedIn(activator.id, backpack, 18, 0);
        object << object.type.getTypeVar("shape", null)
        object << new Mass(2);
        object.remove(SpawnPosition);

        runIfExists("onTaken", backpack);
    }.onlyIf { ->
        // Can only pick up the object if it originally belonged to
        // the player and is not currently contained
        if( object[ContainedIn] ) {
            return false;
        }
        return true;
    }

    // For testing
    addAction("Run onTaken") { ->
        def backpack = findItem(activator.id, "Backpack");
        runIfExists("onTaken", backpack);
    }.onlyIf { ->
        return activator.isSameEntity(self[ContainedIn]?.root);
    }

    addAction("Drop") { ->

        def pos = activator.pos + activator.dir * 0.1;
        pos.y += 0.1;
        //echo("Dropping quest item at:" + pos);

        run("placeInWorld", pos, new Quatd());

        //questItemOnDropped.call(session, self);
        log.info("quest item dropped:" + self.name + " at:" + pos);
        findObjectiveQuests(self, resolveEntityId(session)).each { quest ->
            quest.run("itemDropped", self);
        }
    }.onlyIf { ->
        return activator.isSameEntity(self[ContainedIn]?.root);
    }

    addAction("onTaken") { container ->
        echo("Retrieved " + object.name + ".")
        //questItemOnTaken.call(session, self, container);
        findObjectiveQuests(self, resolveEntityId(session)).each { quest ->
            quest.run("itemTaken", self);
        }
    }
}

createType("Animal").with {
    supertypes("BaseObject");

    addAction("move") { loc, facing ->
        // Overridden to prevent externally moving NPCs.
    }

    addAction("Pet") { ->
        self.brain.signal("Pet", resolveEntityId(session));
    }
}

createType("NPC").with {
    supertypes("BaseObject");

    // For the cases where we really do want a non-contextual name.
    addGetter("characterName") { ->
        return self[CharacterName]?.fullName ?: "Entity@" + self?.getId();
    }

    addAction("randomize") { Random rand ->
        run("randomizeAppearance", rand);
        run("randomizeName", rand);
        run("randomizeClothing", rand);
    }

    addAction("resetHome") { ->
        def existing = findEntity(LocationRelationship.sourceFilter(self), LocationRelationship);
        if( !existing ) {
            existing = entityData.createEntity();
        }
        def pos = self[SpawnPosition];
        existing << LocationRelationship.createLocation(self, pos.location, "home");
        log.info("Set location to:" + existing[LocationRelationship].toString(entityData));
    }

    addAction("randomizeAppearance") { Random rand ->
        def race = self[Race]?.getTypeName(entityData);
        def skin = skinToneIndex.getRandomTone(race, rand);
        def hair = hairToneIndex.getRandomTone(race, rand);
        log.info("Skin:" + skin.name + " - " + skin.description);
        log.info("Hair:" + hair.name + " - " + hair.description);
        self << new SkinColor(skin.getColor());
        self << new HairColor(hair.getColor());
    }

    addAction("randomizeName") { Random rand ->
        def race = self[Race]?.getTypeName(entityData);
        String[] names = characterNameGenerator.generateName(race, rand);
        log.info("name:" + names[0] + " " + names[1]);
        //self << new Name(names[0] + " " + names[1]);
        self << new CharacterName(names[0], names[1]);
    }

    addAction("randomizeClothing") { Random rand ->
        log.info("Randomize clothes for:" + self + "  " + self.name);

        // Remove and delete any existing clothing
        log.info("Removing existing clothes from:" + self.name);
        def wearing = getWornItems(self);
        wearing.each { item ->
            log.info("Removing:" + item.name);
            removeEntity(item);
        }

        def race = self[Race]?.getTypeName(entityData);
        def types = clothingTypeIndex.getRandomClothes(race, rand);
        int slot = 0;
        for( def type : types ) {
            type = type.randomize(similarFabricIndex, rand);
            log.info("Wearing:" + type.getFullName());

            def cellId = worldManager.getCellArrayStorage().store(type.cells);
            def shapeName = new ShapeName("fab", cellId.toIdString());

            def item = createEntity(
                    // Since we often treat designs and items interchangably,
                    // create the design and assign it to the NPC.
                    // Note that assigning it to the world will put it in every
                    // player's design list.
                    new ClothingInfo(self, type.getFullName()),
                    ObjectName.create(type.getName()),
                    ObjectTypeInfo.create("ClothingDesign"),
                    ShapeInfo.create(shapeName.toCompositeString(), 1),
                    new CreatedBy(worldManager.worldEntity),
                    new WornBy(self, slot++)
                );
        }
    }

    addAction("generateQuest") { ->
        //npcGenerateQuest.call(session, self);
        log.info("generateQuest(" + session + ", " + self + ")");

        def existing = findActiveQuest(self);
        log.info("Existing:" + existing);
        if( existing ) {
            log.info("" + self.characterName + " already has an active quest:" + existing.name);
            echo("" + self.characterName + " already has an active quest:" + existing.name);
            return;
        }

        def questTarget = findRandomFetchQuestTarget(self[SpawnPosition].location);
        log.info("Quest Target:" + questTarget.target[StructureInfo] + " loc:" + questTarget.location);
        log.info(" ground block:" + questTarget.block + " " + BlockTypeIndex.get(questTarget.block)?.name);
        if( !questTarget ) {
            echo("Quest target not found.");
            return;
        }

        def relLoc = new RelativeLocation(questTarget.town, questTarget.target, 99, questTarget.block);
        log.info("relLoc:" + relLoc);
        def item = type("QuestAxe").newInstance(
                new SpawnPosition(questTarget.location, new Quatd()),
                new Mass(5),
                ObjectName.create("Axe"),
                new Name("" + self.characterName + "'s Axe")
            );

        def quest = type("FetchQuest").newInstance(
                new Quest(self, worldTime),
                new QuestObjective(null, item),
                relLoc,
                new Name("Find " + item.name)
            );
    }

    addAction("move") { loc, facing ->
        // Overridden to prevent externally moving NPCs.
    }

    addAction("Talk") { ->
        //echo("Want to talk?");
        //npcTalk.call(session, self)

        boolean knows = session.activator.relationship(self)?.level > 0;
        def whoAmI = knows ? session.activator.name : "Stranger";
        def opts = [];

        if( knows ) {
            opts.add(new Option(self, "Nice to see you again, " + self.name + ".", "answer", 1))
        } else {
            opts.add(new Option(self, "Hi, I'm " + session.activator.name + ".", "answer", 1))
        }

        def existing = findActiveQuest(self);

        def message = "Hail and well met.";
        if( existing ) {
            message = "It is too fine a day for the trouble I'm in...";
        }

        if( existing ) {
            def questEntity = findOurActiveQuest(self, resolveEntityId(session.activator));
            if( questEntity ) {
                log.info("Found questEntity:" + questEntity + "  quest:" + questEntity[Quest]);
                message = "Have you found my " + questEntity.questItem[ObjectName].name.toLowerCase() + " yet?";
                if( questEntity[Quest].status == 1 ) {
                    opts.add(new Option(self, "I found your item.", "answer", 4));
                } else {
                    opts.add(new Option(self, "No, I'm still looking.", "wishLuck"));
                }
            } else if( knows ) {
                opts.add(new Option(self, "What kind of trouble?", "answer", 2));
            }
        }
        session.showPrompt(self, PromptType.Dialog,
                            "## Greetings, ${whoAmI}!\n"
                            + message
                        );

        session.showOptions(self, opts, new Option(self, "Nevermind.", "sayGoodbye"));

        def brain = self.brain;
        brain.signal("Talking", resolveEntityId(session));
        brain.say("Greetings, ${whoAmI}!");
    }

    addAction("wishLuck") { ->
        def responses = [
            "I hope you have better luck than I did!",
            "Let me know when you find it.",
            "Thanks for continuing to look.",
            "I guess it will turn up eventually."
        ];
        def response = responses[(int)(Math.random() * responses.size())];
        session.showPrompt(self, PromptType.Dialog, "###" + response);
        self.brain.signal("Done Talking", resolveEntityId(session), response);
    }

    addAction("sayGoodbye") { ->
        self.brain.signal("Done Talking", resolveEntityId(session), "Safe travels.");
    }

    addAction("answer") { int index ->
        if( index == 1 ) {
            def relationship = findEntity(CharacterRelationship.linkFilter(resolveEntityId(session), self),
                                          CharacterRelationship.class);
            boolean knows = relationship && relationship[CharacterRelationship].level > 0;
            if( !knows ) {
                // Have to create the relationship first or self.name will fail to show
                // the character name.
                if( !relationship ) {
                    relationship = createEntity();
                }
                relationship << new CharacterRelationship(resolveEntityId(session), self, 1, 0, worldTime);

                session.showPrompt(self, PromptType.Dialog, "###Hi, I'm " + self.name + ".\nNice to meet you, " + activator.name);
            } else {
                session.showPrompt(self, PromptType.Dialog, "###Nice to see you, too.");
            }
            def opts = [];

            def questEntity = findOurActiveQuest(self, resolveEntityId(session.activator));
            if( questEntity ) {
                if( questEntity[Quest].status == 1 ) {
                    opts.add(new Option(self, "I found your item.", "answer", 4));
                } else {
                    opts.add(new Option(self, "I'm still looking for your " + questEntity.questItem[ObjectName].name.toLowerCase() + ".", "wishLuck"));
                }
            } else {
                def existing = findActiveQuest(self);
                if( existing ) {
                    opts.add(new Option(self, "You said something before about some trouble?", "answer", 2));
                }
            }

            session.showOptions(self, opts, new Option(self, "See you later.", "sayGoodbye"));

            // Still talking to us
            self.brain.signal("Talking", resolveEntityId(session));
            return;
        } else if( index == 2 ) {
            def questEntity = findActiveQuest(self);
            if( !questEntity ) {
                session.showPrompt(self, PromptType.Dialog, "I'm losing my mind.  I don't even know anymore.");
                self.brain.signal("Done Talking", resolveEntityId(session), "I'm losing my mind.  I don't even know anymore.");
                return;
            }
            def questGiver = self;
            def rand = new Random(self.getId() + questEntity.getId());

            def relativeLoc = questEntity[RelativeLocation]
            def objectName = questEntity.questItem[ObjectName].name.toLowerCase();

            // my 'type' axe
            def itemTypes = [
                    "grandfather's",
                    "grandmother's",
                    "father's",
                    "brother's",
                    "favorite",
                    "family's",
                    "specially made",
                    "hand-crafted",
                    "custom",
                    "guild signature-series"
                ]
            def itemType = itemTypes[rand.nextInt(itemTypes.size())];

            def verbs = [
                "walking",
                "exploring",
                "traveling",
                "adventuring",
                "foraging"
                ]
            def verb = verbs[rand.nextInt(verbs.size())];
            def mistakes = [
                "was startled and dropped the ${objectName}",
                "sat down to rest and the ${objectName} must have fallen",
                "tripped and then tossed the ${objectName} in frustration",
                "set the ${objectName} down while I adjusted my pack"
            ];
            def mistake = mistakes[rand.nextInt(mistakes.size())];

            def estimates = [
                "it might have been",
                "maybe",
                "probably",
                "I think",
                ""
            ]
            def estimate = estimates[rand.nextInt(estimates.size())];

            def excuses = [
                "I've been back a few times but can't seem to find exactly where I left it.",
                "Things have been keeping me busy here and I haven't had a chance to look for it.",
                "I don't know when I'll be able to head back that way.",
                "I'm not sure when I'll get a chance to look.",
                "I went back the next day but I couldn't spot it."
            ]
            def excuse = excuses[rand.nextInt(excuses.size())];

            def sourceLoc = relativeLoc.source[SpawnPosition]?.location;
            def targetLoc = relativeLoc.target[SpawnPosition]?.location;

            def targetPoi = relativeLoc.target[StructureInfo];
            def factoryName = targetPoi.getFactory(entityData);
            def poiDesc = worldManager.textDb.getText("poi/test-poi-desciptions.txt#${factoryName}", null);
            poiDesc = poiDesc.readLines()[0];

            def angle = Math.atan2(targetLoc.z - sourceLoc.z, targetLoc.x - sourceLoc.x);
            def distance = targetLoc.distance(sourceLoc);
            angle = Math.toDegrees(angle);
            if( angle < 0 ) {
                angle += 360;
            }
            def angleDelta = 360 / 16; // n, nne, ne, ene, etc.
            def angleIndex = (int)((angle + (angleDelta * 0.5)) / angleDelta);
            def dir = directionNames[angleIndex];

            def description = /### I've lost my ${itemType} ${questEntity.questItem[ObjectName].name.toLowerCase()}!
                I was ${verb} in the area ${dir} of town when I ${mistake}.
                I remember I was at ${poiDesc} and
                ${estimate} ${findDistanceName(distance)} from here.
                ${excuse}
            /

            session.showPrompt(self, PromptType.Dialog, description);
            def opts = [
                new Option(self, "I can try to find it for you.", "answer", 3),
            ]
            session.showOptions(self, opts, new Option(self, "That's really too bad.", "sayGoodbye"));

            // Still talking to us
            self.brain.signal("Talking", resolveEntityId(session));
            return;
        } else if( index == 3 ) {
            def quest = findActiveQuest(self);
            if( !quest ) {
                session.showPrompt(self, PromptType.Dialog, "Wait... what were we talking about?");
                self.brain.signal("Done Talking", resolveEntityId(session), "Wait... what were we talking about?");
                return;
            }
            quest.run("acceptQuest");
            session.showPrompt(self, PromptType.Dialog, "Thanks so much!  Let me know when you've found it.");

            self.brain.signal("Done Talking", resolveEntityId(session), "Thanks so much!  Let me know when you've found it.");
            return;
        } else if( index == 4 ) {
            def quest = findOurActiveQuest(self, resolveEntityId(session));
            quest.run("questCompleted");
            session.showPrompt(self, PromptType.Dialog, "Thanks for bringing it back!");

            self.brain.signal("Done Talking", resolveEntityId(session), "Thanks for bringing it back!");
            return;
        }
        session.showPrompt(self, PromptType.Message, "Unhandled dialog answer type:" + index);
    }

    addAction("Generate Quest") { ->
        //npcGenerateQuest.call(session, self);
        run("generateQuest");
    }

    //addAction("Forget Me") { ->
    //    def relationship = findEntity(CharacterRelationship.linkFilter(resolveEntityId(session), self),
    //                                  CharacterRelationship.class);
    //    if( relationship ) {
    //        removeEntity(relationship);
    //        echo("Done!");
    //    } else {
    //        echo("I never knew you anyway.");
    //    }
    //}

    addAction("Randomize Name and Coloring") { ->
        run("randomizeAppearance", npcRandom);
        run("randomizeName", npcRandom);
    }

    addAction("Randomize Clothing") { ->
        run("randomizeClothing", npcRandom);
    }

    //addAction("Fix Name") { ->
    //    def cn = self[CharacterName];
    //    if( cn == null ) {
    //        return;
    //    }
    //    def first = CharacterNameGenerator.removeDumbStuff(cn.first);
    //    def last = CharacterNameGenerator.removeDumbStuff(cn.last);
    //    self << new CharacterName(first, last);
    //}
    addAction("Fix Location") { ->
        def existing = findEntity(LocationRelationship.sourceFilter(self), LocationRelationship);
        if( !existing ) {
            existing = entityData.createEntity();
        }
        def pos = self[SpawnPosition];
        existing << LocationRelationship.createLocation(self, pos.location, "home");
        echo("Set location to:" + existing[LocationRelationship].toString(entityData));
    }

    //addAction("Test Stuff") { ->
    //    npcTestStuff.call(session, self);
    //}
    //

    addAction("resetDefaultAppearance") { ->
        def race = self[Race]?.getTypeName(entityData);
        def model = self[BodyType]?.getTypeName(entityData);
        log.info("race:" + race + "  model:" + model);
        def bodyConfig = system(BodyConfigIndex).find(race, model);

        log.info("bc:" + bodyConfig);

        self << new SkinColor(bodyConfig.getSkinColor());
        self << new HairColor(bodyConfig.getHairColor());

        log.info("looking up:" + bodyConfig.getClothing());
        def clothing = findClothingDesign(worldManager.worldEntity, bodyConfig.getClothing());
        log.info("found:" + clothing);

        // Because for now the design IS the item, we'll cheat
        def item = copyClothingDesign(clothing, self);
        item << new WornBy(self, 0);
    }

    addAction("cleanup") { ->
        log.info("cleanup:" + self + " " + self.name);
        def wearing = getWornItems(self);
        wearing.each { item ->
            log.info("Removing:" + item.name);
            removeEntity(item);
        }
    }
}
