/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

import org.slf4j.*;
//import org.apache.log4j.Category;
//import org.apache.log4j.NDC;
// I don't know why those fail.
import com.simsilica.action.*;
import com.simsilica.bpos.BodyPosition;
import com.simsilica.mod.*;
import com.simsilica.mworld.World;
import com.simsilica.ext.mblock.*;
import com.simsilica.sim.SimTime;
import mythruna.GameConstants;
import mythruna.sim.Activator;
import mythruna.world.WorldManager;
import mythruna.net.server.GameSessionHostedService.ActivatorImpl;
import mythruna.es.vars.LocalVars;

objectTypes = system(ObjectTypeRegistry.class);

// This lets the "real classes" access the global bindings and stuff.
class GlobalAccess {
    static Object owner;
}
GlobalAccess.owner = this;

class GetterDelegate {
    ObjectType objectType;
    EntityId entity;
    ActionEnvironment env;
    private Logger actionLog;

    public GetterDelegate( ObjectType objectType, EntityId entity, ActionEnvironment env, Logger actionLog ) {
        this.objectType = objectType;
        this.entity = entity;
        this.env = env;
        this.actionLog = actionLog;
    }

    public Logger getLog() {
        return actionLog;
    }

    public EntityId getSelf() {
        return entity;
    }

    public ObjectType getType() {
        return objectType;
    }

    public ActionEnvironment getSession() {
        if( env == null ) {
            throw new IllegalStateException("No action environment available in getter context");
        }
        return env;
    }

    public Object getActivator() {
        return env.getActivator();
    }

}

// Resolve properties in a standard way where the property could be
// a closure.
Object resolveProperty( ObjectType objectType, EntityId entity, String name ) {
    Object val = objectType.getTypeVar(name, null);
    //log.info("  found:" + val);
    if( val == null ) {
        // throw new MissingPropertyException(name, getClass());
        // We don't provide callers any other way to find misses and
        // we can't report our custom types with a real groovy exception
        // so best to wait for some use-cases.
        return null;
    }
    if( val instanceof ClosureGetter ) {
        return val.resolveProperty(objectType, entity, name);
    } else if( val instanceof Closure ) {
        throw new RuntimeException("nothing should be getting here");
        //// rehydrate(Object delegate, Object owner, Object thisObject)
        //ActionEnvironment env = ActionEnvironment.getCurrentEnvironment(false);
        //if( env == null ) {
        //    throw new IllegalStateException("No ActionEnvironment is current for getter call:" + entity + "." + name);
        //}
        //
        //log.warn("We shouldn't be here.");
        //Logger actionLog = LoggerFactory.getLogger("ObjectType.type:" + type.getName());
        //Closure c = val.rehydrate(new GetterDelegate(objectType, entity, env, actionLog), val.owner, val);
        ////MDC.pushByKey("stack", objectType.name + "." + name);
        //def stack = MDC.get("stack");
        //MDC.put("stack", objectType.name + "." + name);
        ////NDC.push(objectType.name + "." + name);
        //try {
        //    return c();
        //} finally {
        //    //MDC.popByKey("stack");
        //    MDC.put("stack", stack);
        //    //NDC.pop();
        //}
    }
    return val;
}

// We try to make ActionContext look like EntityId so that
// they are interchangeable in scripts.  So any meta-method we add
// to EntityId should be added here.
// 2024-01-02: I'm not sure what case ActionContext is actually used today.
// If we find use-cases then we should document them here, until then I'm leaving the
// verbos logging in.
// 2024-04-14: Any code that uses object[] instead of self[] will hit this
// code.  Probably those cases should be found and converted to 'self'.
boolean actionContextVerbose = false;
ActionContext.metaClass {
    getAt() { Class c ->
        //log.info("ActionContext.getAt(" + c + ") delegate:" + delegate);
if( actionContextVerbose ) log.info("ActionContext.getAt(" + c + ")");
        return entityData.getComponent(delegate.getTarget(), c);
    }

    get() { String name ->
if( actionContextVerbose ) log.info("ActionContext.get(" + name + ")");
        //log.info("ActionContext.get(" + name + ") delegate:" + delegate);
        def objectType = delegate.getType();
        return resolveProperty(objectType, delegate.target, name);
    }

    isSameEntity() { Object other ->
if( actionContextVerbose ) log.info("ActionContext.isSameEntity(" + other + ")");
        return resolveEntityId(other)?.id == delegate.target.id;
    }

    leftShift() { EntityComponent component ->
if( actionContextVerbose ) log.info("ActionContext.leftShift(" + component + ")");
        entityData.setComponent(delegate.target, component);
    }

    leftShift() { List components ->
if( actionContextVerbose ) log.info("ActionContext.leftShift(" + components + ")");
        entityData.setComponents(delegate.target, components as EntityComponent[]);
    }

    remove() { Class type ->
if( actionContextVerbose ) log.info("ActionContext.removeComponent(" + delegate.target + ", " + type + ")");
        entityData.removeComponent(delegate.target, type);
    }

    contains() { Object o ->
        if( log.isDebugEnabled() ) {
            log.debug("ActionContext.contains(" + o + ")");
        }
if( actionContextVerbose ) log.info("ActionContext.contains(" + o + ")");
        delegate.target.contains(o);
    }

// These methods all already exist on ActionContext and I don't know why
// I added them here other than cut-paste-itus.
//    run() { ActionEnvironment env, ...vargs ->
//        if( log.isDebugEnabled() ) {
//            log.debug("run:" + vargs);
//        }
//log.info("ActionContext.run:" + vargs);
//        return objectTypes.getContext(delegate.target).run(env, *vargs);
//    }
//
//    canRun() { ActionEnvironment env, ...vargs ->
//        if( log.isDebugEnabled() ) {
//            log.debug("canRun:" + vargs);
//        }
//log.info("ActionContext.canRun:" + vargs);
//        return objectTypes.getContext(delegate.target).canRun(env, *vargs);
//    }
//
//    runIfExists() { ActionEnvironment env, ...vargs ->
//        if( log.isDebugEnabled() ) {
//            log.debug("runIfExists:" + vargs);
//        }
//log.info("ActionContext.runIfExists:" + vargs);
//        return objectTypes.getContext(delegate.target).runIfExists(env, *vargs);
//    }
//
//    hasAction() { ...vargs ->
//        if( log.isDebugEnabled() ) {
//            log.debug("hasAction:" + vargs);
//        }
//log.info("ActionContext.hasAction:" + vargs);
//        return objectTypes.getContext(delegate.target).hasAction(*vargs);
//    }
}

// If you include this one in the closure then it causes an infinite recursion
// but if I don't then I can't even get it to be called, Object or no.
//ActionContext.metaClass.isCase = { EntityId o ->
//    log.info("ActionContext.isCase(" + o + ")");
//    return delegate.target.isCase(o);
//}

raceTypeIndex = optionalSystem(RaceTypeIndex.class);
bodyConfigIndex = optionalSystem(BodyConfigIndex.class);
pronounsIndex = optionalSystem(PronounsIndex.class);

// ActivatorImpl is used in cases where an action refers directly
// to the 'activator' of the action.  Adding these metaClass methods allows
// it to be used just like an entity.  Most common cases in scripts will
// be leftShift and isSameEntity.
ActivatorImpl.metaClass {
    getAt() { Class c ->
        return entityData.getComponent(delegate.getId(), c);
    }

    get() { String name ->
        def objectType = objectTypes.getType(delegate.getId());
        return resolveProperty(objectType, delegate.getId(), name);
    }

    isSameEntity() { Object other ->
        return resolveEntityId(other)?.id == delegate.id.id;
    }

    leftShift() { EntityComponent... components ->
        entityData.setComponents(delegate.id, components as EntityComponent[]);
    }

    remove() { Class type ->
        entityData.removeComponent(delegate.id, type);
    }

    contains() { Object o ->
        if( log.isDebugEnabled() ) {
            log.debug("ActivatorImpl.contains(" + o + ")");
        }
        delegate.id.contains(o);
    }

    getPronouns() { ->
        return delegate.id.getPronouns();
    }

    relationship() { Object other ->
        return delegate.id.relationship(other);
    }

    getBody() { ->
        return getBody(delegate.id);
    }
}

EntityId.metaClass {
    getAt() { Class c ->
        return entityData.getComponent(delegate, c);
    }

    get() { String name ->
        //log.info("EntityId.get(" + name + ") delegate:" + delegate);
        // Look up the object type
        def objectType = objectTypes.getType(delegate);
        return resolveProperty(objectType, delegate, name);
    }

    isSameEntity() { Object other ->
        return resolveEntityId(other)?.id == delegate.id;
    }

    // Var-args version should take care of singles, nulls, etc.
    leftShift() { EntityComponent... components ->
        if( components ) {
            entityData.setComponents(delegate, components as EntityComponent[]);
        }
    }

    remove() { Class type ->
        entityData.removeComponent(delegate, type);
    }

    contains() { Object o ->
        log.info("EntityId.contains(" + o + ")");
        def other = resolveEntityId(o);
        def contained = other[ContainedIn];
        log.info("Other contained in:" + contained);
        if( contained == null ) {
            log.info("No container");
            // It's not contained in anything
            return false;
        }
        // Are we its root?
        if( delegate == contained.root ) {
            log.info("We are the root");
            // We are their root so contain them by definition
            return true;
        }

        // Check the hiearchy starting here
        while( contained != null ) {
            log.info("Checking:" + contained.container);
            // Are we the container?
            if( delegate == contained.container ) {
                return true;
            }
            // Try the parent
            contained = contained.container[ContainedIn];
        }
        return false;
    }

    getType() { ->
        return objectTypes.getType(delegate);
    }

    getContext() { ->
        return objectTypes.getContext(delegate);
    }

    //run() { ActionEnvironment env, ...vargs ->
    //    if( log.isDebugEnabled() ) {
    //        log.debug("run:" + vargs);
    //    }
    //    return objectTypes.getContext(delegate).run(env, *vargs);
    //}
    //
    //canRun() { ActionEnvironment env, ...vargs ->
    //    if( log.isDebugEnabled() ) {
    //        log.debug("run:" + vargs);
    //    }
    //    return objectTypes.getContext(delegate).canRun(env, *vargs);
    //}
    //
    //runIfExists() { ActionEnvironment env, ...vargs ->
    //    if( log.isDebugEnabled() ) {
    //        log.debug("run:" + vargs);
    //    }
    //    return objectTypes.getContext(delegate).runIfExists(env, *vargs);
    //}
    //
    //hasAction() { ...vargs ->
    //    if( log.isDebugEnabled() ) {
    //        log.debug("hasAction:" + vargs);
    //    }
    //    return objectTypes.getContext(delegate).hasAction(*vargs);
    //}

    run() { ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return objectTypes.getContext(delegate).run(*vargs);
    }

    canRun() { ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return objectTypes.getContext(delegate).canRun(*vargs);
    }

    runIfExists() { ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return objectTypes.getContext(delegate).runIfExists(*vargs);
    }

    hasAction() { ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("hasAction:" + vargs);
        }
        return objectTypes.getContext(delegate).hasAction(*vargs);
    }

    getVars() { ->
        return LocalVars.get(delegate, entityData);
    }

    getPronouns() { ->
        def raceName = delegate[Race]?.getTypeName(entityData);
        return pronounsIndex.get(raceName);
    }

    relationship() { Object other ->
        // Should really get the newwest one
        def relationship = findEntity(CharacterRelationship.linkFilter(delegate, resolveEntityId(other)),
                                      CharacterRelationship.class);
        return relationship ? relationship[CharacterRelationship] : null;
    }

    locationRelationship() { String name ->
        def relationship = findEntity(LocationRelationship.typeFilter(delegate, name), LocationRelationship);
        return relationship ? relationship[LocationRelationship] : null;
    }

    getBody() { ->
        return getBody(delegate);
    }

    getBrain() { ->
        return getBrain(delegate);
    }

    setStatusText() { int type, String text, double duration ->
        delegate << new StatusText(type, text, now(), duration);
    }

    getLocation() { ->
        def bPos = delegate[BodyPosition];
        if( bPos ) {
            return bPos.lastLocation;
        }
        def pos = delegate[SpawnPosition];
        if( pos ) {
            return pos.location;
        }
        return null;
    }
}
//EntityId.metaClass.isCase = { Object o ->
//    log.info("EntityId.isCase(" + o + ")");
//    return false;
//}

// ObjectType enhancements
//-------------------------------------
ObjectType.metaClass {
    get() { String name ->
        if( log.isDebugEnabled() ) {
            log.debug("ObjectType.get(" + name + ")");
        }
        return delegate.getTypeVar(name, null);
    }

    getInfo() { ->
        return ObjectTypeInfo.create(delegate.name, entityData);
    }
}

ObjectType.metaClass.static.newInstance = { EntityComponent... init ->

    def result = entityData.createEntity();
    entityData.setComponents(result, delegate.info);
    def defaultComponents = delegate.getTypeVar("defaultComponents", null);
    if( defaultComponents ) {
        entityData.setComponents(result, defaultComponents);
    }
    // Set the newInstance() arguments after in case they override
    // defaults.
    entityData.setComponents(result, init);

    // Call the initialize action if it exists
    result.runIfExists("initialize");
    return result;
}


// ShapeInfo convenience methods... there is a lot to do with shapes that is
// sometimes complicated to write out manually.
//---------------------------------------------------------
ShapeInfo.metaClass.getShapeName << { ->
    return delegate.getShapeName(entityData);
}

ShapeInfo.metaClass.changeShapeName << { name ->
    return delegate.changeShapeName(name, entityData);
}

ShapeInfo.metaClass.static.create = { name, scale ->
    ShapeInfo.create(name, scale, entityData);
}

ShapeInfo.metaClass.static.shapeNameFilter = { name ->
    ShapeInfo.shapeNameFilter(name, entityData);
}

ShapeInfo.metaClass.loadCells << { ->
    return loadCellsForName(delegate.getShapeName(entityData));
}

ShapeInfo.metaClass.loadShape << { ->
    return loadShape(delegate.shapeName, delegate.scale, 0);
}

ShapeInfo.metaClass.getVolume << { ->
    def cells = loadCellsForName(delegate.getShapeName(entityData));
    def size = cells.size.toVec3d();
    size *= delegate.scale;
    return size;
}

ShapeInfo.metaClass.getCenterOffset << { ->
    def shape = delegate.loadShape();
    return shape.boundsCenter;
}

ShapeInfo.metaClass.getCenterBaseOffset << { ->
    def offset = delegate.centerOffset;
    offset.y = 0;
    return offset;
}



// Different EntityComponent convenience enhancements
//----------------------------------------------------
ObjectName.metaClass.static.create = { String name ->
    return ObjectName.create(name, entityData);
}

ObjectName.metaClass.getName = { ->
    return delegate.getName(entityData);
}

LocationRelationship.metaClass.static.createLocation = { EntityId source, Vec3d loc, String type ->
    return LocationRelationship.create(source, loc, type, entityData);
}

LocationRelationship.metaClass.getType = { ->
    return delegate.getType(entityData);
}

Decay.metaClass.static.seconds = { double seconds ->
    long startTime = now();
    long endTime = future(seconds);
    return new Decay(startTime, endTime);
}

// Stuff like this should probably be a separate groovy file
Holding.metaClass.static.create << { EntityId id, String manipulator ->
    Holding.create(id, manipulator, entityData);
}
Holding.metaClass.static.create << { Activator activator, String manipulator ->
    Holding.create(activator.id, manipulator, entityData);
}
Holding.metaClass.static.create << { ActionContext context, String manipulator ->
    Holding.create(context.target, manipulator, entityData);
}

AttachedTo.metaClass.static.create << { EntityId target, String attachPoint, Vec3d offset, Quatd rotation ->
    AttachedTo.create(target, attachPoint, offset, rotation, entityData);
}

SpawnPosition.metaClass.constructor << { Vec3d loc, Quatd rotation ->
    return new SpawnPosition(GameConstants.PHYSICS_GRID, loc, rotation);
}

SoundInfo.metaClass.static.create = { name, startTime, volume, looping ->
    return SoundInfo.create(name, startTime, volume, looping, entityData);
}

MapMarker.metaClass.static.create = { String type, int size, Vec3d pos ->
    return MapMarker.create(type, size, pos, entityData);
}

MapMarker.metaClass.static.create = { EntityId owner, String type, int size, Vec3d pos ->
    return MapMarker.create(owner, type, size, pos, entityData);
}

MapMarker.metaClass.getTypeName = { ->
    return delegate.getTypeName(entityData);
}

ObjectTypeInfo.metaClass.static.create = { String typeName ->
    return ObjectTypeInfo.create(typeName, entityData);
}

ObjectTypeInfo.metaClass.static.typeFilter = { String typeName ->
    return ObjectTypeInfo.typeFilter(typeName, entityData);
}

ObjectTypeInfo.metaClass.getTypeName = { ->
    return delegate.getTypeName(entityData);
}

EntityLink.metaClass.static.create = { def source, def target, String typeName ->
    return EntityLink.create(resolveEntityId(source), resolveEntityId(target), typeName, entityData);
}

Morph.metaClass.static.filter = { def target, String verb ->
    return Morph.filter(resolveEntityId(target), verb, entityData);
}

Morph.metaClass.static.create = { def target, String verb, double mix ->
    return Morph.create(resolveEntityId(target), verb, mix, entityData);
}

AgentType.metaClass.static.create = { String type, int level ->
    return AgentType.create(type, level, entityData);
}

LocationRelationship.metaClass.static.typeFilter = { EntityId self, String type ->
    return LocationRelationship.typeFilter(self, type, entityData);
}

Vec3d.metaClass.getLeafId = { ->
    return GameConstants.LEAF_GRID.worldToId(delegate);
}

ActionEnvironment.metaClass {
    // I get tired of not having an echo() on ActionEnvironment
    echo() { String msg ->
        delegate.showPrompt(resolveEntityId(delegate), PromptType.Message, msg);
    }
}

/**
 *  Class to better control what is "in scope" for the action
 *  closures since trying to let groovy sort out our delegation
 *  policy with respect to naked properties, getters, global bindings,
 *  etc. gets messy.
 */
class ActionDelegate {
    static Logger log = LoggerFactory.getLogger(ActionDelegate.class);

    private ObjectAction action;
    private ActionEnvironment env;
    private ActionContext context;
    private Logger actionLog;

    public Logger getLog() {
        return actionLog;
    }

    public boolean run( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return context.run(*vargs);
    }

    public boolean superRun( ...vargs ) {
        return context.superRun(action, *vargs);
    }

    public boolean runIfExists( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("runIfExists:" + vargs);
        }
        return context.runIfExists(*vargs);
    }

    public boolean canRun( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("canRun:" + vargs);
        }
        return context.canRun(*vargs);
    }

    public boolean hasAction( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("hasAction:" + vargs);
        }
        return context.hasAction(*vargs);
    }

    public ObjectType type( String name ) {
        return GlobalAccess.owner.objectTypes.getType(name);
    }

    public ActionEnvironment getEnv() {
        return env;
    }

    public ActionEnvironment getSession() {
        return env;
    }

    public ActionContext getContext() {
        return context;
    }

    public Object getActivator() {
        return env.getActivator();
    }

    public long getId() {
        return context.getTarget().getId();
    }

    public EntityId getTarget() {
        return context.getTarget();
    }

    public EntityId getSelf() {
        return context.getTarget();
    }

    public ActionContext getObject() {
        return context;
    }

    public ObjectType getType() {
        return context.getType();
    }

    public Object getTypeVar( String name ) {
        return getTypeVar(name, null);
    }

    public Object getTypeVar( String name, Object defaultValue ) {
        return context.getType().getTypeVar(name, defaultValue);
    }

    public void echo( String s ) {
        env.showPrompt(context.target, PromptType.Message, s);
    }
}

//class ActionOwner {
//    public Object get( String name ) {
//        log.info("Want to get:" + name);
//        Object result = GlobalAccess.owner.modManager.getGlobalBinding(name);
//        log.info("  global:" + result);
//        if( result != null  ) {
//            return result;
//        }
//        result = context.get(name);
//        log.info("  context:" + result);
//        if( result != null ) {
//            return result;
//        }
//        //result = env.getProperty(name);
//        //log.info("  env:" + result);
//        //if( result != null ) {
//        //    return result;
//        //}
//        // If we don't throw a missing property exception then
//        // the property resolution will think that we have this property
//        // and stop.  Since we are the owner that means it won't go
//        // to the delegate and we end up hiding environment properties
//        // like activator.
//        throw new MissingPropertyException(name, getClass());
//    }
//}

class ClosureAction extends AbstractObjectAction {
    static Logger log = LoggerFactory.getLogger(ClosureAction.class);
    Closure run;
    Closure canRun;
    Logger actionLog;

    public ClosureAction( ObjectType type, String name, Class[] types ) {
        super(name, types);
        // We'll pass this through to the action delegates that we create to
        // provide a better 'log' than the ActionDelegate's log.  This only becomes
        // an issue because we changed the closure resolution strategy for action
        // closures to 'delegate first'.  Prior to these changes, the logging
        // was going to the one defined by the mod pack... but note: not the groovy
        // file that contained the code.  In that sense, the object-type based log
        // is more specific though I'm a bit sad that we lose the module part.
        // Because we are resolving this at closure action creation time, there is
        // the possibility that we could somehow get that information from a static
        // variable somewhere with some clever code.  I'm going to live with it
        // for a little while first.  -pspeed:2023-12-10
        actionLog = LoggerFactory.getLogger("ObjectType.type:" + type.getName());
    }

    public boolean canRun( ActionEnvironment env, EntityId entity, Object... args ) {
        // This is not as efficient in the case where we already had the context but
        // makes the scripting code a lot easier.
        return canRun(env, entity.context, args);
    }

    @Override
    public boolean canRun( ActionEnvironment env, ActionContext context, Object... args ) {
        //log.info("ClosureAction.canRun(" + env + ", " + context + ", " + args + ")");
        if( run == null ) {
            return false;
        }
        if( canRun == null ) {
            // No special condition
            return true;
        }
        def actionDelegate = new ActionDelegate(action:this, env:env, context:context, actionLog:actionLog);
        Closure c = canRun.rehydrate(actionDelegate, canRun.owner, canRun);

        //MDC.pushByKey("stack", context.type.name + "." + name);
        def stack = MDC.get("stack");
        MDC.put("stack", context.type.name + "." + name);
        //NDC.push(context.type.name + "." + name);
        try {
            Object result;
            if( args.size() != 0 ) {
                result = c(*args);
            } else {
                result = c();
            }
            // Not as dumb as it looks because 'result' can be all kinds of
            // things and groovy's if() handles more than just primitive boolean.
            if( result ) {
                //log.info("Result is true:" + result);
                return true;
            } else {
                //log.info("Result is NOT true:" + result);
                return false;
            }
            return true;
        } finally {
            //MDC.popByKey("stack");
            MDC.put("stack", stack);
            //NDC.pop();
        }
    }

    public boolean run( ActionEnvironment env, EntityId entity, Object... args ) {
        // This is not as efficient in the case where we already had the context but
        // makes the scripting code a lot easier.
        return run(env, entity.context, args);
    }

    @Override
    public boolean run( ActionEnvironment env, ActionContext context, Object... args ) {
        //log.info("ClosureAction.run(" + env + ", " + context + ", " + args + ")");
        if( run == null ) {
            return false;
        }
        def actionDelegate = new ActionDelegate(action:this, env:env, context:context, actionLog:actionLog);
        Closure c = run.rehydrate(actionDelegate, run.owner, run);

        //MDC.pushByKey("stack", context.type.name + "." + name);
        def stack = MDC.get("stack");
        MDC.put("stack", context.type.name + "." + name);
        //NDC.push(context.type.name + "." + name);
        try {
            Object result;
            if( args.size() != 0 ) {
                result = c(*args);
            } else {
                result = c();
            }
            // Not as dumb as it looks because 'result' can be all kinds of
            // things and groovy's if() handles more than just primitive boolean.
            if( result ) {
                //log.info("Result is true:" + result);
                return true;
            } else {
                //log.info("Result is NOT true:" + result);
                return false;
            }
            return true;
        } catch( MissingMethodException e ) {
            // If we let this bubble up on its own then it will confuse groovy
            // sometimes and it will report the wrong method (the caller)
            throw new RuntimeException("Error running:" + this, e);
        } finally {
            //MDC.popByKey("stack");
            MDC.put("stack", stack);
            //NDC.pop();
        }
    }

    public ClosureAction onlyIf( Closure canRun ) {
        this.canRun = canRun;
        return this;
    }
}

/**
 *  Wraps a 'getter' closure so that we can keep some additional runtime
 *  state like loggers and stuff.  This is a sign that 'getters' should be
 *  first class parts of the object/actions framework.
 */
class ClosureGetter {
    Closure exec;
    Logger actionLog;

    public ClosureGetter( ObjectType type, Closure exec ) {
        this.exec = exec;
        this.actionLog = LoggerFactory.getLogger("ObjectType.type:" + type.getName());

        // See the comment in ObjectTypeWrapper about why 'delegate first'.
        exec.resolveStrategy = Closure.DELEGATE_FIRST;
    }

    public Object resolveProperty( ObjectType objectType, EntityId entity, String name ) {
        ActionEnvironment env = ActionEnvironment.getCurrentEnvironment(false);
        if( env == null ) {
            throw new IllegalStateException("No ActionEnvironment is current for getter call:" + entity + "." + name);
        }

        def getterDelegate = new GetterDelegate(objectType, entity, env, actionLog);
        Closure c = exec.rehydrate(getterDelegate, exec.owner, exec);
        //MDC.pushByKey("stack", objectType.name + "." + name);
        def stack = MDC.get("stack");
        MDC.put("stack", objectType.name + "." + name);
        //NDC.push(objectType.name + "." + name);
        try {
            return c();
        } finally {
            //MDC.popByKey("stack");
            MDC.put("stack", stack);
            //NDC.pop();
        }
    }
}

/**
 *  ObjectTypeWrapper is the type available during action configuration
 *  and at the root level of regular scripts.  Within action closures,
 *  actions will deal with the ObjectType directly and should never see
 *  an ObjectTypeWrapper.
 */
class ObjectTypeWrapper {
    static Logger log = LoggerFactory.getLogger(ObjectTypeWrapper.class);

    // This will hide the global 'log' from added actions and getters when
    // "with" is used because this class will be the owner.
    // If we want logging here then we should name it something else or
    // adjust the owner of the closure.  Future bridge to cross.

    def type;

    public ClosureAction addAction( String name, Closure exec ) {
        ClosureAction a = new ClosureAction(type, name, exec.getParameterTypes());
        a.run = exec;

        // Within the closure, we would prefer to resolve properties/methods
        // against the ActionDelegate itself versus hitting the owner first.
        // The 'owner' will be this ObjectTypeWrapper instance which might
        // be a super-class, etc..  This is a problem for things like getTypeVar()
        // which we'd prefer to run on the local real type instead of the super
        // type where the closure might have been defined.
        // I'm doing this as part of script API cleanup to reduce some of the extra
        // garbage in action scripts.  Resolution is tricky here so hopefully I
        // haven't created some other problems for myself.
        // For example, action closures will not be able to easily access the
        // direct type upon which they are defined except by accessing 'owner'.
        // I think this will be rare and 'advanced use' so I'm ok with it.
        // If we need it to look 'fancier' then we must have some idea when it
        // will be used and can make it a part of the ActionDelegate.
        // For more info see: https://groovy-lang.org/closures.html#_delegation_strategy_2
        exec.resolveStrategy = Closure.DELEGATE_FIRST;
        type.addAction(a);
        return a;
    }

    public void addGetter( String name, Closure exec ) {
        // Straight forward
        type.setTypeVar(name, new ClosureGetter(type, exec));
    }

    public void setTypeVar( String name, Object value ) {
        type.setTypeVar(name, value);
    }

    public void setTypeVar( String name, String... lines ) {
        type.setTypeVar(name, lines.join("\n"));
    }

    public Object getTypeVar( String name ) {
        return getTypeVar(name, null);
    }

    public Object getTypeVar( String name, Object defaultValue ) {
        return type.getTypeVar(name, defaultValue);
    }

    public void supertypes( String... types ) {
        def objectTypes = GlobalAccess.owner.objectTypes;
        for( String t : types ) {
            if( !objectTypes.exists(t) ) {
                throw new IllegalArgumentException("Unknown type:" + t);
            }
            ObjectType parent = objectTypes.getType(t);
            type.addSuperType(parent);
        }
    }

    public ObjectTypeInfo getInfo() {
        return type.info;
    }

    public EntityId newInstance( EntityComponent... init ) {
        return type.newInstance(init);
    }
}

// We so it this way so that it's easier to access entityData and other
// globals
//ObjectTypeWrapper.metaClass.getInfo = { ->
//    return ObjectTypeInfo.create(delegate.type.name, entityData);
//}

// Note: ObjectTypeWrapper is part of the delegate chain for executed actions
// and so anything we add to the metaClass will show up as a regular method.
// In this case, this closure was formerly called "createInstance" and was
// thus overriding any other regular createInstance() and not just type.createInstance().
// Problem showed when leaving off ObjectTypeInfo in createInstance() and suddenly
// seeing all kinds of ObjectTools around the world.
// This was added only 2 weeks ago so I don't think it's poisoned very many
// things in my development worlds.
// ObjectTypeWrapper is in scope because we want it to be for a lot of normal
// "just outside" action closures stuff and I don't think we can easily truncate it
// without losing the rest of the delegation chain.  So we'll just have to be careful
// with the type-specific methods we add.
// 2024-01-02: Not sure how much of the above is true anymore because of the ActionDelegate
// that we prefer over the parents and try to specifically clamp down what is available
// in an action closure.  Suffice to say, naming things in different scopes is
// still an important consideration.
//ObjectTypeWrapper.metaClass.newInstance = { EntityComponent... init ->
//    delegate.type.newInstance(init);
//    //def result = entityData.createEntity();
//    //entityData.setComponents(result, delegate.info);
//    //def defaultComponents = delegate.getTypeVar("defaultComponents", null);
//    //if( defaultComponents ) {
//    //    entityData.setComponents(result, defaultComponents);
//    //}
//    //// Set the newInstance() arguments after in case they override
//    //// defaults.
//    //entityData.setComponents(result, init);
//    //
//    //// Call the initialize action if it exists
//    //result.runIfExists("initialize");
//    //return result;
//}

type = { String name ->
    if( !objectTypes.exists(name) ) {
        throw new IllegalArgumentException("Unknown type:" + name);
    }
    return new ObjectTypeWrapper(type:objectTypes.getType(name));
}

createType = { String name ->
    if( objectTypes.exists(name) ) {
        throw new IllegalArgumentException("Type already exists:" + name);
    }
    // Else create an empty type in lieu of a builder pattern
    def type = new ObjectType(name);
    objectTypes.registerType(type);
    return new ObjectTypeWrapper(type:type);
}


