/*
 * Copyright (c) 2023, Simsilica, LLC
 * All rights reserved.
 */

findDefaultDesigns  = {
    def parent = worldManager.worldEntity;
    def filter = Filters.fieldEquals(AssemblyBlueprintInfo.class, "parent", parent);
    return entityData.findEntities(filter, AssemblyBlueprintInfo.class); 
} 

createBlueprintFromCells = { EntityId parent, String bpName, String objectName, CellArray cells, boolean carved, double scale ->

    // See if there is already an entity for the bpName
    EntityId entity = findBlueprint(parent, bpName);
    if( entity == null ) {
        entity = entityData.createEntity();
        log.info("Creating blueprint for:" + bpName);
    } else {
        BlueprintInfo bpInfo = entityData.getComponent(entity, BlueprintInfo.class); 
        log.info("Found existing blueprint for:" + bpName + " : " + bpInfo);
    }

    CellArrayId cellId = worldManager.getCellArrayStorage().store(cells);
    
    Vec3i offset = new Vec3i();
    offset.x = 5 - (cells.getSizeX()/2);
    offset.z = 5 - (cells.getSizeZ()/2);

    String shapeName = cellId.toFileName();
    if( carved ) {
        shapeName = "c_" + shapeName;
    }

    log.info("createing shapeId:" + shapeName);
    entityData.setComponents(entity,
            new BlueprintInfo(parent, bpName, offset),
            ObjectName.create(objectName, entityData),
            ShapeInfo.create(shapeName, scale, entityData),
            new CreatedBy(parent) 
            );

    return entity;                
}    


createDefaultDesign = { String bpName, String objectName, String assemblyName ->
    def parent = worldManager.worldEntity;
    
    // See if there is already an entity for the bpName
    EntityId entity = findDesign(parent, bpName);
    if( entity == null ) {
        entity = entityData.createEntity();
        log.info("Creating design for:" + bpName);
    } else {
        BlueprintInfo bpInfo = entityData.getComponent(entity, BlueprintInfo.class); 
        log.info("Found existing design for:" + bpName + " : " + bpInfo);
    }

    def assembly = loadSubassembly(assemblyName);
    def subId = worldManager.getSubassemblyStorage().store(assembly);
    String shapeName = subId.toFileName();
    
    log.info("Creating default blueprint:" + bpName + "  assembly:" + assembly);
    entityData.setComponents(entity,
        new AssemblyBlueprintInfo(parent, bpName),
        ObjectName.create(objectName),
        ShapeInfo.create(shapeName, 1),
        new CreatedBy(parent)
        );
    
    return entity;
}
 
copyDesign = { EntityId bp, EntityId target ->
    def bpInfo = bp[AssemblyBlueprintInfo];
    if( bpInfo == null ) {
        throw new IllegalArgumentException("Entity is not a design:" + bp);
    }
    EntityId entity = findDesign(target, bpInfo.getName());
    if( entity != null ) {
        log.info("Player:" + target + " already has bp:" + bpInfo);
        // So let's not blow away any changes they might have made already
        return;
    } else {
        log.info("Creating:" + bpInfo + " on player:" + target);
        entity = entityData.createEntity();
    }

    entityData.setComponents(entity,
        bpInfo.changeParent(target),
        bp[ObjectName],
        bp[ShapeInfo],
        bp[CreatedBy]
        );
        
    return entity;    
}
 
// Make sure we have some standard blueprints created
if( true ) {
    log.info("Creating default designs...");
    createDefaultDesign("Wood Door", "Door", "/Models/objects/wood-door1.assembly"); 
    createDefaultDesign("Wood Chest", "Chest", "/Models/objects/wood-chest1.assembly"); 
} 
 

// When a player entity is created, we want to add some
// standard blueprints for them.
on( playerEntityCreated ) { event ->
    //log.info("Adding default blueprints to:" + event);
}

on( playerEntityJoining ) { event ->
    log.info("blueprint tests...");
 
    // Copy the standard blueprints
    log.info("Adding default designs to:" + event.player);   
    findDefaultDesigns().each {
        log.info("Design:" + it);
        copyDesign(it, event.player);
    }
}
