/*
 * Copyright (c) 2024, Simsilica, LLC
 * All rights reserved.
 */

/*
Used to test the "remove losing context" bug that was written up as
   task T001108 but a little differently.  The problem was fixed in the action
   framework.

createType("TestType").with {
    addGetter("testGetter") { ->
        return "I am:" + self + " with type:" + self[ObjectTypeInfo];
    }

    addAction("testAction") { ->
        echo("test1 self:" + self + " self.testGetter:" + self.testGetter);
        removeEntity(self);
        echo("test2 self:" + self + " self.testGetter:" + self.testGetter);
    }
}
*/

createType("ButterflyEncounter").with {

    /**
     *  Returns the current mobs that were spawned and are being
     *  managed by this encounter.
     */
    addGetter("spawns") { ->
        // Keep this minimal to catch as many things as possible even
        // if something goes wrong.  Spawns don't always start with a
        // position so we don't include that.
        log.info("find spawns:" + OwnedBy.filter(self));
        findEntities(OwnedBy.filter(self),
                     OwnedBy.class,
                     ShapeInfo.class,
                     AgentType.class).each {
            log.info(" it:" + it);
        }

        return findEntities(OwnedBy.filter(self),
                     OwnedBy.class,
                     ShapeInfo.class,
                     AgentType.class);
    }

    addAction("activateEncounter") { ->
        log.info("activateEncounter:" + self);
        if( self[SpawnPosition] == null ) {
            log.error("Encounter does not have a spawn position:" + self, new Throwable("very visible"));
            return;
        }
        // Look for flowers, spawn butterflies based on frequency + dice roll

        // How many flowers are there in this zone and where are they?
        def pos = self[SpawnPosition].location;
        def colId = ColumnId.fromWorld(pos);
        def flowerTypes = worldStats.getBlockSet("flowers");
        List<Vec3i> flowers = worldStats.getSurfaceBlocks(colId, flowerTypes);

        double max = Math.ceil(Math.sqrt(flowers.size()));
        int count = (int)(1 + Math.random() * (max - 1));

        log.info("check for existing butterflies for:" + self);
        self.spawns.each {
            log.info("  Found existing butterfly:" + it);
            // If we already have a butterfly then take it away from the 'to spawn' count
            count--;
        }

        int butterFlyType = (int)(Math.random() * 2);

        def spawns = [:];

        // Spawn in the appropriate number of butterflies
        for( int i = 0; i < count; i++ ) {
            int index = (int)Math.random() * flowers.size();
            Vec3d loc = flowers.remove(index).toVec3d();

            double roll = Math.random();
            roll = roll * roll * roll;
            int variant = (int)(roll * 4);

            def entity = type("Butterfly").newInstance(
                                    new CreatedBy(self),
                                    // OwnedBy is better because if a player picks up the
                                    // butterfly then we can change the owned by and it won't
                                    // be managed by the encounter anymore.
                                    new OwnedBy(self),
                                    new SkinVariation(butterFlyType * 4 + variant)
                                );
            log.info("Created butterfly:" + entity);

            if( variant == 3 ) {
                float r = (float)Math.random();
                float g = (float)Math.random();
                float b = (float)Math.random();
                def color = new ColorRGBA(r, g, b, 1);
                log.info("Setting random skin color to:" + color);
                entity << new SkinColor(color);
            }

            spawns.put(entity, loc);
        }

        // We'll spool the actual placement but not the creation.
        // If we spool the creation then if we get deactivated  before they have
        // all spawned them some might get orphaned and just hang around the world
        // forever.
        spawns.entrySet().each {
            def entity = it.key;
            def loc = it.value;
            spool {
                // See if it's even still real
                if( entity[AgentType] ) {
                    entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                    log.info("  placed butterfly:" + entity + " at:" + entity[SpawnPosition]?.location);
                } else {
                    log.warn("Entity is gone by the time of placement:" + entity + " loc:" + loc);
                }
            }
        }

        self.spawns.each {
            log.info(" created:" + it);
        }
    }

    addAction("deactivateEncounter") { ->
        log.info("deactivateEncounter:" + self);
        // Despawn any of the spawned butterflies that are still alive
        // and remove the encounter.
        log.info("spawns:" + self.spawns);
        self.spawns.each {
            log.info("  Removing butterfly:" + it);
            removeEntity(it);
        }

        // If we remove ourselves first then we can't actually grab spawns and stuff
        // because we won't have a type yet.  Probably this should be spooled.
        // I fixed that, by the way.
        log.info("removing self:" + self);
        removeEntity(self);
    }
}

// Try to remove all of the butterflies to clean things up if something broke
if( true ) {
    boolean cleanup = false;

    // Try to find all of them... not just in this zone
    log.info("All entities...");
    findEntities(null, SpawnPosition.class, AgentType.class,
                 ObjectTypeInfo.class).each {
        log.info("Mobs:" + it + "  name:" + it[ObjectName].toString(entityData) + " type:" + it[ObjectTypeInfo].getTypeName());
        String type = it[ObjectTypeInfo].getTypeName();
        if( type == "Animal" ) {
            type = it[ObjectName].getName();
        }
        log.info("  type:" + type);
        log.info("  zone:" + it[SpawnPosition].binId);
        if( "Butterfly" == type ) {
            log.info("  bufferly agent type:" + it[AgentType]);
            log.info("  body pos:" + it[BodyPosition]);
            if( cleanup ) {
                log.info("  ...removing");
                removeEntity(it);
            }
        }
    }

    log.info("All buttefly encounters...");
    findEntities(ObjectTypeInfo.typeFilter("ButterflyEncounter"), ObjectTypeInfo.class).each {
        if( cleanup ) {
            log.info("Removing encounter:" + it);
            removeEntity(it);
        } else {
            log.info("Found existing butterfly encounter:" + it);
        }
    }
}


system(AgentActivationSystem.class).addActivationZoneListener(new ActivationZoneListener() {
    public void activateZone( long zoneId ) {
        log.info("activateZone(" + zoneId + ")");

        def trigger = findEntity(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class);
        if( !trigger ) {

            // Look at the world data to see if there are any flowers.
            // no flowers -> don't create it

            def colId = new ColumnId(zoneId);
            def flowerTypes = worldStats.getBlockSet("flowers");
            List<Vec3i> flowers = worldStats.getSurfaceBlocks(colId, flowerTypes);

            // Figure out how many max we would want to spawn based on season, climate, etc..
            // ...but for now spawn the encounter if there are any flowers at all.
            if( !flowers.isEmpty() ) {
                def butterflies = createEntity(
                    ObjectTypeInfo.create("ButterflyEncounter"),
                    new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd())
                );
                log.info("created:" + butterflies + " at:" + butterflies[SpawnPosition]);
                trigger = createEntity(
                    new EncounterTrigger(butterflies, zoneId)
                );
            }
        } else {
            log.info("Found existing:" + trigger);
        }
    }

    public void deactivateZone( long zoneId ) {
        log.info("deactivateZone(" + zoneId + ")");
        def trigger = findEntity(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class);
        if( trigger ) {
            //log.info("removing:" + trigger);
            removeEntity(trigger);
        }
    }
});

