/*
 * Copyright (c) 2024, Simsilica, LLC
 * All rights reserved.
 */


createType("BaseNPC").with {
    supertypes("BaseObject");

    addAction("resetHome") { ->
        def existing = findEntity(LocationRelationship.sourceFilter(self), LocationRelationship);
        if( !existing ) {
            existing = entityData.createEntity();
        }
        def pos = self.location;

        // See if this location already has a town or something
        def poiEntity = null;
        def poi = findCurrentPoi(pos);
        if( poi ) {
            poiEntity = new EntityId(poi.id.id);
            log.info("POI Entity:" + poiEntity);
            log.info(" name:" + poiEntity.name);
        } else {
            log.warn("NPC:" + self + " has no home town for location:" + pos);
        }

        existing << LocationRelationship.createEntityLocation(self, poiEntity, pos, "home");
        log.info("Set location to:" + existing[LocationRelationship].toString(entityData));

        // We might not have a brain yet
        self?.brain?.signal("Reset");
    }

    addAction("Reset Home") { ->
        run("resetHome");
        echo("Set home location to:" + self.locationRelationship("home")?.toString(entityData));
    }

    addAction("move") { loc, facing ->
        // Overridden to prevent externally moving NPCs.
    }

}

createType("Animal").with {
    supertypes("BaseNPC");

    addAction("Pet") { ->
        self.brain.signal("Pet", resolveEntityId(session));
    }

    addAction("Talk") { ->
        def brain = self.brain;
        brain.signal("Talking", resolveEntityId(session));
        if( self.name == "Dog" ) {
            brain.say("** woof **");
        }
    }
}

createType("Butterfly").with {
    supertypes("BaseNPC");

    setTypeVar("shape", ShapeInfo.create("animals/butterfly.rig", 1));

    addAction("Look") { ->
        superRun();
    }

    addGetter("objectDescription") { ->
        if( self[SkinColor] ) {
            return "A pretty butterfly with a rare color.";
        }
        return "A pretty butterfly.";
    }

    addAction("initialize") { ->
        superRun();

        self << ObjectName.create("Butterfly");
        self << new Mass(0.1);

        // Spawn in as passive and let the agent activation system
        // decide if it should be active
        self << AgentType.create("Butterfly", AgentType.LEVEL_PASSIVE);
    }

    //addAction("Randomize Appearance") { ->
    //    run("randomizeAppearance", npcRandom);
    //}

    addAction("randomizeAppearance") { Random rand ->
        int index = rand.nextInt(8);
        self << new SkinVariation(index);
        if( (index % 4) == 3 ) {
            float r = rand.nextFloat();
            float g = rand.nextFloat();
            float b = rand.nextFloat();
            def color = new ColorRGBA(r, g, b, 1); //SkinTone.stringToColor("ffc19620");
            self << new SkinColor(color);
        } else {
            self.remove(SkinColor.class);
        }
    }
}

createType("Gaefen").with {
    supertypes("Animal");

    setTypeVar("shape", ShapeInfo.create("animals/gaefen.rig", 1));

    addAction("Pet") { ->
        superRun();
    }

    addAction("Talk") { ->
        superRun();
    }

    addAction("initialize") { ->
        superRun();

        self << ObjectName.create("Gaefen", entityData);
        self << new Mass(10);
        self << AgentType.create("Gaefen", AgentType.LEVEL_PASSIVE);

        run("Randomize Coloring");
    }

    addAction("Randomize Coloring") { ->
        double baseOffset = 0.25;
        double base = 1.0 - baseOffset;
        double offset1 = Math.random() * (baseOffset * 1.25);
        double offset2 = offset1 * Math.random();
        double offset3 = offset1 * Math.random();
        float r = (float)Math.min(1.0, base + offset1);
        float g = (float)Math.min(1.0, base + offset2);
        float b = (float)Math.min(1.0, base + offset3);
        def color = new ColorRGBA(r, g, b, 1);
        log.info("Setting color to:" + color);
        self << new SkinColor(color);
    }
}

createType("Dog").with {
    supertypes("Animal");

    setTypeVar("shape", ShapeInfo.create("animals/dog/dog-lab.rig", 1));

    addAction("initialize") { ->
        superRun();

        self << ObjectName.create("Dog", entityData);
        self << new Mass(15);
        self << AgentType.create("Dog", AgentType.LEVEL_PASSIVE);
    }
}


createType("NPC").with {
    supertypes("BaseNPC");

    // For the cases where we really do want a non-contextual name.
    addGetter("characterName") { ->
        return self[CharacterName]?.fullName ?: "Entity@" + self?.getId();
    }

    addAction("randomize") { Random rand ->
        run("randomizeAppearance", rand);
        run("randomizeName", rand);
        run("randomizeClothing", rand);
    }

    addAction("randomizeAppearance") { Random rand ->
        def race = self[Race]?.getTypeName(entityData);
        def skin = skinToneIndex.getRandomTone(race, rand);
        def hair = hairToneIndex.getRandomTone(race, rand);
        log.info("Skin:" + skin.name + " - " + skin.description);
        log.info("Hair:" + hair.name + " - " + hair.description);
        self << new SkinColor(skin.getColor());
        self << new HairColor(hair.getColor());
    }

    addAction("randomizeName") { Random rand ->
        def race = self[Race]?.getTypeName(entityData);
        String[] names = characterNameGenerator.generateName(race, rand);
        log.info("name:" + names[0] + " " + names[1]);
        //self << new Name(names[0] + " " + names[1]);
        self << new CharacterName(names[0], names[1]);
    }

    addAction("randomizeClothing") { Random rand ->
        log.info("Randomize clothes for:" + self + "  " + self.name);

        // Remove and delete any existing clothing
        log.info("Removing existing clothes from:" + self.name);
        def wearing = getWornItems(self);
        wearing.each { item ->
            log.info("Removing:" + item.name);
            removeEntity(item);
        }

        def race = self[Race]?.getTypeName(entityData);
        def types = clothingTypeIndex.getRandomClothes(race, rand);
        int slot = 0;
        for( def type : types ) {
            type = type.randomize(similarFabricIndex, rand);
            log.info("Wearing:" + type.getFullName());

            def cellId = worldManager.getCellArrayStorage().store(type.cells);
            def shapeName = new ShapeName("fab", cellId.toIdString());

            def item = createEntity(
                    // Since we often treat designs and items interchangably,
                    // create the design and assign it to the NPC.
                    // Note that assigning it to the world will put it in every
                    // player's design list.
                    new ClothingInfo(self, type.getFullName()),
                    ObjectName.create(type.getName()),
                    ObjectTypeInfo.create("ClothingDesign"),
                    ShapeInfo.create(shapeName.toCompositeString(), 1),
                    new CreatedBy(worldManager.worldEntity),
                    new WornBy(self, slot++)
                );
        }
    }

    addAction("generateQuest") { ->
        //npcGenerateQuest.call(session, self);
        log.info("generateQuest(" + session + ", " + self + ")");

        def existing = findActiveQuest(self);
        log.info("Existing:" + existing);
        if( existing ) {
            log.info("" + self.characterName + " already has an active quest:" + existing.name);
            echo("" + self.characterName + " already has an active quest:" + existing.name);
            return;
        }

        def questTarget = findRandomFetchQuestTarget(self, self[SpawnPosition].location);
        if( !questTarget ) {
            echo("Quest target not found.");
            return;
        }
        log.info("Quest Target:" + questTarget.target[StructureInfo] + " loc:" + questTarget.location);
        log.info(" ground block:" + questTarget.block + " " + BlockTypeIndex.get(questTarget.block)?.name);

        def relLoc = new RelativeLocation(questTarget.town, questTarget.target, 99, questTarget.block);
        log.info("relLoc:" + relLoc);
        def item = type("QuestAxe").newInstance(
                new SpawnPosition(questTarget.location, new Quatd()),
                new Mass(5),
                ObjectName.create("Axe"),
                new Name("" + self.characterName + "'s Axe")
            );

        def quest = type("FetchQuest").newInstance(
                new Quest(self, worldTime),
                new QuestObjective(null, item),
                relLoc,
                new Name("Find " + item.name)
            );
    }

    addAction("Talk") { ->
        //echo("Want to talk?");
        //npcTalk.call(session, self)

        boolean knows = session.activator.relationship(self)?.level > 0;
        def whoAmI = knows ? session.activator.name : "Stranger";
        def opts = [];

        if( knows ) {
            opts.add(new Option(self, "Nice to see you again, " + self.name + ".", "answer", 1))
        } else {
            opts.add(new Option(self, "Hi, I'm " + session.activator.name + ".", "answer", 1))
        }

        def existing = findActiveQuest(self);

        def message = "Hail and well met.";
        if( existing ) {
            message = "It is too fine a day for the trouble I'm in...";
        }

        if( existing ) {
            def questEntity = findOurActiveQuest(self, resolveEntityId(session.activator));
            if( questEntity ) {
                log.info("Found questEntity:" + questEntity + "  quest:" + questEntity[Quest]);
                message = "Have you found my " + questEntity.questItem[ObjectName].name.toLowerCase() + " yet?";
                if( questEntity[Quest].status == 1 ) {
                    opts.add(new Option(self, "I found your item.", "answer", 4));
                } else {
                    opts.add(new Option(self, "No, I'm still looking.", "wishLuck"));
                }
            } else if( knows ) {
                opts.add(new Option(self, "What kind of trouble?", "answer", 2));
            }
        }
        session.showPrompt(self, PromptType.Dialog,
                            "## Greetings, ${whoAmI}!\n"
                            + message
                        );

        session.showOptions(self, opts, new Option(self, "Nevermind.", "sayGoodbye"));

        def brain = self.brain;
        brain.signal("Talking", resolveEntityId(session));
        brain.say("Greetings, ${whoAmI}!");
    }

    addAction("wishLuck") { ->
        def responses = [
            "I hope you have better luck than I did!",
            "Let me know when you find it.",
            "Thanks for continuing to look.",
            "I guess it will turn up eventually."
        ];
        def response = responses[(int)(Math.random() * responses.size())];
        session.showPrompt(self, PromptType.Dialog, "###" + response);
        self.brain.signal("Done Talking", resolveEntityId(session), response);
    }

    addAction("sayGoodbye") { ->
        self.brain.signal("Done Talking", resolveEntityId(session), "Safe travels.");
    }

    addAction("answer") { int index ->
        if( index == 1 ) {
            def relationship = findEntity(CharacterRelationship.linkFilter(resolveEntityId(session), self),
                                          CharacterRelationship.class);
            boolean knows = relationship && relationship[CharacterRelationship].level > 0;
            if( !knows ) {
                // Have to create the relationship first or self.name will fail to show
                // the character name.
                if( !relationship ) {
                    relationship = createEntity();
                }
                relationship << new CharacterRelationship(resolveEntityId(session), self, 1, 0, worldTime);

                session.showPrompt(self, PromptType.Dialog, "###Hi, I'm " + self.name + ".\nNice to meet you, " + activator.name);
            } else {
                session.showPrompt(self, PromptType.Dialog, "###Nice to see you, too.");
            }
            def opts = [];

            def questEntity = findOurActiveQuest(self, resolveEntityId(session.activator));
            if( questEntity ) {
                if( questEntity[Quest].status == 1 ) {
                    opts.add(new Option(self, "I found your item.", "answer", 4));
                } else {
                    opts.add(new Option(self, "I'm still looking for your " + questEntity.questItem[ObjectName].name.toLowerCase() + ".", "wishLuck"));
                }
            } else {
                def existing = findActiveQuest(self);
                if( existing ) {
                    opts.add(new Option(self, "You said something before about some trouble?", "answer", 2));
                }
            }

            session.showOptions(self, opts, new Option(self, "See you later.", "sayGoodbye"));

            // Still talking to us
            self.brain.signal("Talking", resolveEntityId(session));
            return;
        } else if( index == 2 ) {
            def questEntity = findActiveQuest(self);
            if( !questEntity ) {
                session.showPrompt(self, PromptType.Dialog, "I'm losing my mind.  I don't even know anymore.");
                self.brain.signal("Done Talking", resolveEntityId(session), "I'm losing my mind.  I don't even know anymore.");
                return;
            }
            def questGiver = self;
            def rand = new Random(self.getId() + questEntity.getId());

            def relativeLoc = questEntity[RelativeLocation]
            def objectName = questEntity.questItem[ObjectName].name.toLowerCase();

            // my 'type' axe
            def itemTypes = [
                    "grandfather's",
                    "grandmother's",
                    "father's",
                    "brother's",
                    "favorite",
                    "family's",
                    "specially made",
                    "hand-crafted",
                    "custom",
                    "guild signature-series"
                ]
            def itemType = itemTypes[rand.nextInt(itemTypes.size())];

            def verbs = [
                "walking",
                "exploring",
                "traveling",
                "adventuring",
                "foraging"
                ]
            def verb = verbs[rand.nextInt(verbs.size())];
            def mistakes = [
                "was startled and dropped the ${objectName}",
                "sat down to rest and the ${objectName} must have fallen",
                "tripped and then tossed the ${objectName} in frustration",
                "set the ${objectName} down while I adjusted my pack"
            ];
            def mistake = mistakes[rand.nextInt(mistakes.size())];

            def estimates = [
                "it might have been",
                "maybe",
                "probably",
                "I think",
                ""
            ]
            def estimate = estimates[rand.nextInt(estimates.size())];

            def excuses = [
                "I've been back a few times but can't seem to find exactly where I left it.",
                "Things have been keeping me busy here and I haven't had a chance to look for it.",
                "I don't know when I'll be able to head back that way.",
                "I'm not sure when I'll get a chance to look.",
                "I went back the next day but I couldn't spot it."
            ]
            def excuse = excuses[rand.nextInt(excuses.size())];

            def sourceLoc = relativeLoc.source[SpawnPosition]?.location;
            def targetLoc = relativeLoc.target[SpawnPosition]?.location;

            def targetPoi = relativeLoc.target[StructureInfo];
            def factoryName = targetPoi.getFactory(entityData);
            def poiDesc = worldManager.textDb.getText("poi/test-poi-desciptions.txt#${factoryName}", null);
            poiDesc = poiDesc.readLines()[0];

            def angle = Math.atan2(targetLoc.z - sourceLoc.z, targetLoc.x - sourceLoc.x);
            def distance = targetLoc.distance(sourceLoc);
            def angleDelta = 360 / 16; // n, nne, ne, ene, etc.
            angle = Math.toDegrees(angle);
            angle += (angleDelta * 0.5);
            if( angle < 0 ) {
                angle += 360;
            }
            def angleIndex = (int)(angle / angleDelta);
            def dir = directionNames[angleIndex];

            def description = /### I've lost my ${itemType} ${questEntity.questItem[ObjectName].name.toLowerCase()}!
                I was ${verb} in the area ${dir} of town when I ${mistake}.
                I remember I was at ${poiDesc} and
                ${estimate} ${findDistanceName(distance)} from here.
                ${excuse}
            /

            session.showPrompt(self, PromptType.Dialog, description);
            def opts = [
                new Option(self, "I can try to find it for you.", "answer", 3),
            ]
            session.showOptions(self, opts, new Option(self, "That's really too bad.", "sayGoodbye"));

            // Still talking to us
            self.brain.signal("Talking", resolveEntityId(session));
            return;
        } else if( index == 3 ) {
            def quest = findActiveQuest(self);
            if( !quest ) {
                session.showPrompt(self, PromptType.Dialog, "Wait... what were we talking about?");
                self.brain.signal("Done Talking", resolveEntityId(session), "Wait... what were we talking about?");
                return;
            }
            quest.run("acceptQuest");
            session.showPrompt(self, PromptType.Dialog, "Thanks so much!  Let me know when you've found it.");

            self.brain.signal("Done Talking", resolveEntityId(session), "Thanks so much!  Let me know when you've found it.");
            return;
        } else if( index == 4 ) {
            def quest = findOurActiveQuest(self, resolveEntityId(session));
            quest.run("questCompleted");
            session.showPrompt(self, PromptType.Dialog, "Thanks for bringing it back!");

            self.brain.signal("Done Talking", resolveEntityId(session), "Thanks for bringing it back!");
            return;
        }
        session.showPrompt(self, PromptType.Message, "Unhandled dialog answer type:" + index);
    }

    addAction("Generate Quest") { ->
        //npcGenerateQuest.call(session, self);
        run("generateQuest");
    }

    //addAction("Forget Me") { ->
    //    def relationship = findEntity(CharacterRelationship.linkFilter(resolveEntityId(session), self),
    //                                  CharacterRelationship.class);
    //    if( relationship ) {
    //        removeEntity(relationship);
    //        echo("Done!");
    //    } else {
    //        echo("I never knew you anyway.");
    //    }
    //}

    addAction("Randomize Name and Coloring") { ->
        run("randomizeAppearance", npcRandom);
        run("randomizeName", npcRandom);
    }

    addAction("Randomize Clothing") { ->
        run("randomizeClothing", npcRandom);
    }

    //addAction("Test Stuff") { ->
    //    npcTestStuff.call(session, self);
    //}
    //

    addAction("resetDefaultAppearance") { ->
        def race = self[Race]?.getTypeName(entityData);
        def model = self[BodyType]?.getTypeName(entityData);
        log.info("race:" + race + "  model:" + model);
        def bodyConfig = system(BodyConfigIndex).find(race, model);

        log.info("bc:" + bodyConfig);

        self << new SkinColor(bodyConfig.getSkinColor());
        self << new HairColor(bodyConfig.getHairColor());

        log.info("looking up:" + bodyConfig.getClothing());
        def clothing = findClothingDesign(worldManager.worldEntity, bodyConfig.getClothing());
        log.info("found:" + clothing);

        // Because for now the design IS the item, we'll cheat
        def item = copyClothingDesign(clothing, self);
        item << new WornBy(self, 0);
    }

    addAction("cleanup") { ->
        log.info("cleanup:" + self + " " + self.name);
        def wearing = getWornItems(self);
        wearing.each { item ->
            log.info("Removing:" + item.name);
            removeEntity(item);
        }
    }
}
