/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */


import mythruna.mblock.*;

// List of substance names that should not appear in the build wand status
substanceBaseIgnores = [
        "bad",
        "light",
        "light-blue",
        "light-green",
        "light-red",
        "plain-light",
        "test-normals",
        "test-pattern"
    ] as Set;

/**
 *  Retrieves all of the substance counters associated with the particular
 *  target object.
 */
getSubstanceCounters = { target ->
    return entityData.findEntities(SubstanceCounter.targetFilter(target), SubstanceCounter.class);
}

/**
 *  Retrieves the first matching type-specific substance counter associated to the
 *  particular target object.
 */
getSubstanceCounter = { target, type ->
    return entityData.findEntity(SubstanceCounter.typeNameFilter(target, type), SubstanceCounter.class);
}

/**
 *  Updates the type-specific substance counter associated with the particular
 *  target object.  Multiplier controls whether the value is increased or decreased.
 *  'max' is used to set the maximum amount if a new SubstanceCounter is created.
 */
changeSubstanceCounter = { target, substance, max, multiplier ->
    def typeName = substance.name.name;
    def existing = getSubstanceCounter(target, typeName);
    if( existing ) {
        def counter = existing[SubstanceCounter];
        double newAmount = counter.amount + substance.amount * multiplier;
        if( newAmount > counter.max) {
            newAmount = counter.max;
        } else if( newAmount < 0 ) {
            newAmount = 0;
        }
        counter = counter.changeAmount(newAmount);
        existing << counter;
    } else if( multiplier > 0 ) {
        def counter = SubstanceCounter.create(target, typeName, substance.amount * multiplier, max);
        existing = createEntity(counter);
    }
    return existing;
}

/**
 *  Returns the number of blocks that can be placed of the specified type
 *  from the target's substance inventory.
 */
getCountForSubstances = { target, type ->
    def parts = SubstanceTypeIndex.findParts(type);

    def minQuantity = Double.POSITIVE_INFINITY;
    parts.each { part ->
        //def existing = counters.get(part.name.name);
        def existing = getSubstanceCounter(target, part.name.name);
        if( !existing ) {
            minQuantity = 0;
            return;
        }
        def counter = existing[SubstanceCounter];
        def quantity = (int)((counter.amount / part.amount) + 0.00000001);
        if( quantity < minQuantity ) {
            minQuantity = quantity;
        }
    }
    return minQuantity;
}

createType("BuildWand").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Block");
    setTypeVar("description",
              "The Build Wand is used for placing and removing blocks.",
              "",
              "The right mouse button will place the current block type",
              "and the left mouse button will remove the block under the",
              "cursor (highlighted yellow).",
              "",
              "Press 'e' to open the block type selector menu.",
              "Press 'c' to select the block under the cursor as the",
              "current type.",
              "",
              "The selected block will appear in the bottom center of the",
              "screen.  If the block supports multiple rotations then the",
              "mouse wheel will change the block rotation."
            );
    setTypeVar("handleLocation", new Vec3d(1.5, 1.5, 3));

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/wand1.blocks", standardItemScale * 0.5));
    setTypeVar("volume", new ObjectVolume(new Vec3d(3, 3, 13).multLocal(standardItemScale * 0.5)));

    addAction("startRemoving") { BlockHit hit ->
        Vec3i loc = hit.block;
        log.info("" + context + ".mainPressBlock(" + loc + ")");
    }

    addAction("stopRemoving") { BlockHit hit ->
        Vec3i loc = hit.block;
        log.info("" + context + ".mainClickBlock(" + loc + ")");

        // Perform validation to make sure the player isn't cheating.
        // FIXME: add distance check
        // FIXME: add line of sight check

        // Are they allowed to edit here?
        if( !getPerms(activator, loc).canRemoveBlock() ) {
            echo("You do not have permission to remove blocks here.");
            return;
        }

        // Only adjust the wand values if the cell setting was successful
        if( world.setWorldCell(loc.toVec3d(), 0) == 0 ) {
            //testCollection.call(session, self, hit.type);
            // Add the substances to the wand inventory
            def parts = SubstanceTypeIndex.findParts(hit.type);
            parts.each { part ->
                changeSubstanceCounter(self, part, 1000, 1);
            }
            def selectedType = env.getVar("buildWand.selectedType", -1);
            if( selectedType == hit.type ) {
                run("resetFocusAnnotation", selectedType);
            }
        }
    }

    addAction("startFilling") { BlockHit hit, Vec3d loc, Integer selected ->
        log.info("" + context + ".startFilling()");
    }

    addAction("stopFilling") { BlockHit hit, Vec3d loc, Integer selected ->
        log.info("" + context + ".stopFilling()");

        // Perform validation to make sure the player isn't cheating.
        // FIXME: add distance check
        // FIXME: add line of sight check

        // Are they allowed to edit here?
        if( !getPerms(activator, loc).canAddBlock() ) {
            echo("You do not have permission to place blocks here.");
            return;
        }
        log.info("claim:" + getClaim(activator, loc));

        // Need to check to see if the type the client sent is actually
        // valid.  We send it with the action to avoid the possibility of
        // separate messages passing each other somehow... but it does mean
        // that we need to verify that the player can even use that type,
        // if they have the resources, and so on. (Which we'd have to do
        // anyway.)

        // Only adjust the wand values if the cell setting was successful
        def result = world.setWorldCell(loc, selected);
        if( result >= 0 ) {
            result = MaskUtils.getType(result);
        }
        //echo("set:" + selected + " result:" + result);
        if( result == selected ) {
            //testRemoval.call(session, self, selected);
            // Subtract the substances from the wand inventory
            def parts = SubstanceTypeIndex.findParts(selected);
            parts.each { part ->
                changeSubstanceCounter(self, part, 1000, -1);
            }
            def selectedType = env.getVar("buildWand.selectedType", selected);
            run("resetFocusAnnotation", selectedType);
        }
    }

    addAction("selectType") { Integer index ->
        log.info("" + context + ".selectType(" + index + ")");
        context << ObjectFocus.blockType(index);

        env.setVar("buildWand.selectedType", index);
        //testSelectType.call(session, self, index);
        run("resetFocusAnnotation", index);
    }

    addAction("resetFocusAnnotation") { Integer index ->
        def count = getCountForSubstances(self, index);
        echo("Amount:" + count);
        session.showPrompt(self, PromptType.Message, "focusAnnotation:" + count);
    }

    addAction("rotate") { Integer index ->
        log.info("" + context + ".rotate(" + index + ")");
        // Handled by the client right now... calls selectType
    }

    addAction("showStatus") { ->
        log.info("action: Show Status");

        def lines = [];

        def name = self.name;
        lines += "### $name Status";

        def counters = getSubstanceCounters(self);
        def statuses = new TreeMap();
        counters.each { entity ->
            def counter = entity[SubstanceCounter];
            if( counter ) {
                statuses.put(counter.typeName, String.format("%.03f/%.00f", counter.amount, counter.max));
            }
        }

        SubstanceTypeIndex.getBaseTypes().each { base ->
            if( statuses.containsKey(base.name) ) {
                return;
            }
            statuses.put(base.name, "0.000/1000");
        }
        statuses.keySet().removeAll(substanceBaseIgnores);

        int columnCount = 0;
        int columnBreak = (int)Math.ceil(statuses.size() / 3.0);
        lines += "{column}";
        statuses.entrySet().each { entry ->
            def type = entry.key;
            def image = type + "-counter.png";
            def title = type.capitalize();
            lines += String.format("![%s](Interface/%s \"%s\"){scale=0.5} %s %s",
                                    image, image, title, title, entry.value);
            columnCount++;
            if( columnCount >= columnBreak ) {
                lines += "{column}";
                columnCount = 0;
            }
        }

        session.showPrompt(self, PromptType.Dialog, lines.join("\n\n"));
        session.showOptions(self, []);
    }
}

// Temporary to keep my test game from blowing up
createType("BuildWand2").with {
    supertypes("BuildWand");
}
createType("BuildWandOriginal").with {
    supertypes("BuildWand");
}


