/*
 * Copyright (c) 2024, Simsilica, LLC
 * All rights reserved.
 */

//  Defines some base quest object types and convenience methods

import mythruna.world.tile.*;
import mythruna.world.town.*;

createRandomNpc = { EntityId createdBy, Vec3d loc, Random rand ->

    def raceIds = ["human-male", "human-female"];
    def raceId = raceIds[rand.nextInt(raceIds.size())];
    def bodyConfig = bodyConfigIndex.getFirst(raceId);
    def raceType = raceTypeIndex.getRace(raceId);

    def entity = createEntity(
                ShapeInfo.create(bodyConfig.model, 1),
                new SpawnPosition(loc, new Quatd()),
                ObjectName.create(raceType.name + " " + raceType.subtype, entityData),
                ObjectTypeInfo.create("NPC"),
                Race.create(raceId, entityData),
                new Mass(60),
                new CreatedBy(createdBy),
                AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)
            );
    entity.run("randomize", rand);
    entity.run("resetHome");
    entity.run("generateQuest");

    return entity;
}

createRandomPets = { npc, loc, typeName, count, rand ->

    // See if we already have that many
    int existing = 0;
    def typeInfo = ObjectTypeInfo.create(typeName);
    findEntities(OwnedBy.filter(npc), OwnedBy.class, ObjectTypeInfo.class, AgentType.class).each {
        if( it[ObjectTypeInfo] == typeInfo ) {
            existing++;
        }
    }
    if( existing == count ) {
        log.info("Already have " + count + " " + typeName + " pets");
        return;
    }
    count -= existing;

    for( int i = 0; i < count; i++ ) {
        def v = loc.add(rand.nextDouble() - 0.5, 0, rand.nextDouble() - 0.5);
        def pet = type(typeName).newInstance(
            new SpawnPosition(v, new Quatd()),
            new OwnedBy(npc)
        );
        pet.runIfExists("randomize", rand);
        pet.run("Reset Home");
        log.info("Created " + typeName + ":" + pet);
    }

}

generatePets = { town, npc ->
    log.info("generatePets(" + town + ", " + npc + ")");

    def home = npc.locationRelationship("home");
    def loc = home.location;
    def seed = loc.leafId;
    log.info("generatePets() NPC:" + npc + " home:" + home + "  seed:" + seed);
    def rand = new Random(seed);

    def roll = rand.nextDouble();
    log.info("roll:" + roll);
    if( roll < 0.25 ) {
        // 25% chance no pets at all
        log.info("no pets generated");
        return;
    }

    int total = 1 + rand.nextInt(3);
    int dogs = rand.nextInt(total);
    int gaefen = total - dogs;

    log.info("total:" + total + "  dogs:" + dogs + "  gaefen:" + gaefen);

    createRandomPets(npc, loc, "Dog", dogs, rand);
    createRandomPets(npc, loc, "Gaefen", gaefen, rand);
}

generateTownQuestGiver = { town ->
    log.info("generateTownQuestGiver(" + town + ")");

    // See if we've already got one
    def existing = null;
    def npcTypeInfo = ObjectTypeInfo.create("NPC");
    findEntities(CreatedBy.filter(town), CreatedBy.class, ObjectTypeInfo.class, AgentType.class).each { npc ->
        if( npc[ObjectTypeInfo] == npcTypeInfo ) {
            existing = npc;
        }
    }
    if( existing ) {
        //session.echo("This town already has a quest-giver NPC");
        log.info("" + town.name + " already has a quest-giver NPC:" + existing);
        generatePets(town, existing);
        return existing;
    }

    def townPoi = getPointOfInterest(town);
    log.info("town poi:" + townPoi);
    if( false ) {
        log.info(town + "->children:");
        townPoi.children.each {
            log.info(town + "  " + it);
        }
    }

    def candidates = [];

    // We prefer temples
    candidates.addAll(townPoi.children.findAll { it.type.startsWith("temple") });
    //log.info(town + " temples:" + candidates);

    if( candidates.isEmpty() ) {
        // Else we'll try graves
        candidates.addAll(townPoi.children.findAll { it.type.startsWith("graves") });
    }
    if( candidates.isEmpty() ) {
        // Else we'll try gallows
        candidates.addAll(townPoi.children.findAll { it.type.startsWith("gallows") });
    }
    if( candidates.isEmpty() ) {
        // Else we'll try warehouses
        candidates.addAll(townPoi.children.findAll { it.type.startsWith("warehouse") });
    }
    if( candidates.isEmpty() ) {
        // Else we'll try wells
        candidates.addAll(townPoi.children.findAll { it.type.startsWith("well") });
    }

    // Note: this is only stable for this world instance but not the world seed
    // in general.  Need to standardize a town seed because this even affects the
    // town layout repeatability.
    // FIXME: use official town seed when it exists
    def rand = new Random(town.getId());
    def target = candidates[rand.nextInt(candidates.size())];
    log.info(town + "->target:" + target);

    def structure = findInsertInfoEntity(town, target);
    log.info(town + "->found:" + structure);

    def spawnLoc = structure[SpawnPosition]?.location;
    def info = structure[TriggerInfo];

    log.info(town + "->base:" + spawnLoc + "  info:" + info);

    def loc = spawnLoc + info.size * 0.5;
    loc.y = spawnLoc.y;

    log.info(town + "->center:" + loc);

    loc = findEmptyCell(loc, Math.max(info.size.x, info.size.z));
    //def loc = findRandomLocationInStructure(structure);
    log.info(town + "->loc:" + loc);

    def entity = createRandomNpc(town, loc, rand);
    generatePets(town, entity);
    return entity;
}


directionNames = [
    "east", "east by southeast", "south east", "south by southeast",
    "south", "south by southwest", "southwest", "west by southwest",
    "west", "west by northwest", "northwest", "north by northwest",
    "north", "north by northeast", "northeast", "east by northeast"
]

distanceNames = [
    128:"nearby",
    256:"about a quarter km",
    512:"about half a km",
    768:"less than a km",
    1024:"about a km",
    1152:"just over a km",
    1280:"more than a km",
    1536:"about a km and a half",
    2048:"about two kms",
    3072:"about a days travel",
    3584:"more than a days travel",
    4000:"far"
] as TreeMap;

findDistanceName = { distance ->
    def lastMatch = null;
    def lastDistance = -10000;
    for( Map.Entry e : distanceNames.entrySet() ) {
        if( log.isTraceEnabled() ) {
            log.trace("checking:" + e + "   " + distance + " > " + e.key + " = " +(distance > e.key));
        }
        if( distance > e.key ) {
            if( log.isTraceEnabled() ) {
                log.trace("  no passed yet");
            }
            lastDistance = e.key;
            lastMatch = e.value;
        } else {
            // We've reached the end and have to make a decision
            double d1 = Math.abs(lastDistance - distance);
            double d2 = Math.abs(e.key - distance);
            if( log.isTraceEnabled() ) {
                log.trace("  last:" + d1 + " next:" + d2);
            }
            if( d1 < d2 ) {
                return lastMatch;
            } else {
                return e.value;
            }
        }
    }
    return lastMatch;
}

findActiveQuest = { EntityId questGiver ->
    for( EntityId quest : findEntities(Quest.questGiverFilter(questGiver), Quest.class) ) {
        if( quest[Quest].status >= 0 ) {
            return quest;
        }
    }
    return null;
}

findOurActiveQuest = { EntityId questGiver, EntityId quester ->
    for( EntityId quest : findEntities(Quest.specificQuestFilter(questGiver, quester), Quest.class) ) {
        if( log.isTraceEnabled() ){
            log.trace("checking quest:" + quest + "  status:" + quest.status);
        }
        if( quest[Quest].status >= 0 ) {
            return quest;
        }
    }
    return null;
}

findObjectiveQuests = { EntityId item, EntityId quester ->

    log.info("findObjectiveQuests(" + item + ", " + quester + ")");

    // A single item may actually be the objective of multiple quests
    def results = [];

    // First let's find all of the QuestObjectives for which this item is the object
    def filter = QuestObjective.objectiveFilter(item);
    findEntities(filter, QuestObjective.class).each { objective ->
        log.info("found objective:" + objective);
        def parentQuest = objective[QuestObjective].quest?:objective;
        log.info("parent quest:" + parentQuest + " " + parentQuest.name);
        // Find our specific child... there should only be one of these
        def childQuest = findEntity(Quest.childFilter(parentQuest, quester));
        log.info("child:" + childQuest);
        results.add(childQuest);
    }

    return results;
}

class QuestTarget {
    EntityId town;
    PointOfInterest townPoi;
    PointOfInterest questPoi;
    EntityId target;
    Vec3d location;
    int block;
}

findRandomFetchQuestTarget = { mob, loc ->
    log.info("----------------------------------findRandomFetchQuestTarget");
    QuestTarget result = new QuestTarget();
    result.townPoi = findCurrentPoi(loc);
    if( result.townPoi == null ) {
        log.info("No town near:" + loc + " Seeing if MOB:" + mob + " has creating town");
        def mobCreator = mob[CreatedBy]?.creatorId;
        log.info("found mobCreator:" + mobCreator + "  type:" + mobCreator?.type);
        if( "Town" != mobCreator?.type?.name ) {
            return;
        }
        result.townPoi = getPointOfInterest(mobCreator);
        result.town = mobCreator;
    } else {
        result.town = new EntityId(result.townPoi.featureId.id);
    }

    log.info("Inside:" + result.town.name);

    def candidates = [];
    findPointsOfInterest(loc, 1536).each { poi ->
        if( !result.townPoi.intersects(poi) ) {
            candidates.add(poi);
        }
    }
    log.info("Found " + candidates.size() + " candidates.");
    int index = (int)(candidates.size() * Math.random());
    log.info("index:" + index);
    result.questPoi = candidates[index];

    log.info("Quest POI:" + result.questPoi);

    // Find a child of that POI
    def poiId = new EntityId(result.questPoi.featureId.id);
    candidates = findEntities(StructureInfo.parentFilter(poiId), StructureInfo);
    log.info("Found " + candidates.size() + " candidates.");
    index = (int)(candidates.size() * Math.random());
    result.target = candidates[index];

    log.info("Quest Target:" + result.target[StructureInfo]);

    result.location = findRandomLocationInStructure(result.target);

    result.block = MaskUtils.getType(world.getWorldCell(result.location));
    if( result.block == 0 ) {
        result.block = MaskUtils.getType(world.getWorldCell(result.location.subtract(0, 1, 0)));
    }

    return result;
}


createType("FetchQuest").with {

    addGetter("name") { ->
        def n = self[Name];
        if( n != null ) {
            return n.getName();
        }
        def on = self[ObjectName];
        if( on != null ) {
            return on.getName(entityData);
        }
        return "Entity@" + self?.getId();
    }

    // Return the quest item
    addGetter("questItem") { ->
        def quest = self[Quest].parentQuest ?: self;
        def objective = quest[QuestObjective];
        if( objective ) {
            return objective.objective;
        }
        // Else search for the quest objective... could be more than one, too
        // ...for now we will return the first one
        return findEntity(QuestObjective.questFilter(quest), QuestObjective);
    }

    addAction("onTriggerApproached") { EntityId mob ->
        def quest = self[Quest];
        def item = self.questItem;
        echo("This looks near where " + item.name + " should be.");
    }

    addAction("acceptQuest") { ->
        //fetchAcceptQuest.call(session, self);

        def player = resolveEntityId(session.activator);
        def pos = self.questItem[SpawnPosition];

        // Get the target structure so that we can create a reasonable trigger
        // to let the player know when they've reached the right structure and not
        // just a 6 meter circle on the ground.

        def trigger = self[RelativeLocation].target;
        def triggerPos = trigger[SpawnPosition];
        def triggerInfo = trigger[TriggerInfo]

        // We should normalize this to the position that the NPC describes, though.
        // ...but then we'd need to make sure the radius contained the original point.
        def center = triggerPos.location.add(triggerInfo.xSize * 0.5, 0, triggerInfo.zSize * 0.5);
        def radius = 128;

        // For now, we'll just randomly move the center a bit to simulate uncertainty
        //log.info("original center:" + center);
        //def angle = Math.random() * Math.PI * 2;
        //def distance = 0.4 + Math.random() * radius * 0.4;
        //center.x += Math.cos(angle) * distance;
        //center.z += Math.sin(angle) * distance;
        ////center.x += (radius * 1 * Math.random()) - (radius * 0.5);
        ////center.z += (radius * 1 * Math.random()) - (radius * 0.5);
        //log.info("random offset center:" + center);
        // None of that matters because it's the spawn position of the quest
        // that controls it... and if we move that it will mess up the triggers.

        // Create a player-specific copy of the quest with a trigger
        // for that player
        def copy = self.type.newInstance(
                self[Quest].createChild(self, player, worldTime),
                // If we copy the objective then it makes it harder to find the quest
                // from the item because we'll (potentially) randomly find one of the children
                // instead of the parent
                //self[QuestObjective],
                self[Name],
                self[RelativeLocation],
                triggerPos,
                // Trigger private to the player
                triggerInfo.changeVisibility(player),
                //TriggerInfo.create(TileId.fromWorld(pos.location), 6, 2, player)
                MapMarker.create(player, "quest-area", radius, center)
        );
    }

    addAction("updateStatus") { int status ->
        def quest = self[Quest];
        self << quest.updateStatus(status, worldTime);
        if( quest.parentQuest ) {
            quest.parentQuest.run("updateStatus", status);
        }
    }

    addAction("itemTaken") { EntityId item ->
        //fetchItemTaken.call(session, self, item);
        log.info("itemTaken(" + session + ", " + self + ", " + item + ")");

        // If we've picked up the item then we don't need to have the trigger
        // anymore
        self.remove(TriggerInfo);

        // And the map marker
        self.remove(MapMarker);

        // Update this quest and the parent
        self.run("updateStatus", 1);

        echo("This can be returned to " + self[Quest].questGiver.name);
    }

    addAction("itemDropped") { EntityId item ->
        //fetchItemDropped.call(session, self, item);
        log.info("itemDropped(" + session + ", " + self + ", " + item + ")");

        // Quest is no longer progressed so we'll revert it back
        self.run("updateStatus", 0);
        echo(item.name + " will have to wait until later.");
    }

    addAction("abortQuest") { ->
        //fetchAbortQuest.call(session, self);
        log.info("fetchAbortQuest(" + session + ", " + self + ")");
    }

    addAction("questCompleted") { ->
        //fetchQuestCompleted.call(session, self);
        log.info("questCompleted(" + session + ", " + self + ")");

        // Make sure the trigger is removed... though it should have been
        // removed when we picked up the item.
        self.remove(TriggerInfo);

        // We'll also remove the quest item instead of transferring it to the NPC (for now)
        def item = self.questItem;
        log.info("Removing:" + item + " " + item?.name);
        removeEntity(item);

        // Now mark this quest child complete
        Quest quest = self[Quest];
        self << quest.updateStatus(-1, worldTime);

        def parentQuest = quest.parentQuest;
        parentQuest << parentQuest[Quest].updateStatus(-1, worldTime);
    }
}


