/*
 * $Id$
 * 
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */
import mythruna.shell.*;
import mythruna.net.server.AccountHostedService;
import mythruna.net.server.GameSessionHostedService;

addShellCommand = { CommandShell target, String name, List help, Closure doIt ->
    ShellCommand cmd = new AbstractShellCommand(help as String[]) {
                public Object execute( CommandShell shell, String args ) {
                    // rehydrate(Object delegate, Object owner, Object thisObject) 
                    Closure c = doIt.rehydrate(shell, doIt.owner, doIt);
                    int length = doIt.getParameterTypes().length;
                    if( length == 2 ) {
                        c(shell, args);
                    } else { 
                        c(args);
                    }                    
                    return null;
                }        
            }; 
    target.addCommand(name, cmd); 
    return cmd;           
} 

updateShellCommand = { CommandShell target, String name, List help, Closure doIt ->
    ShellCommand cmd = new AbstractShellCommand(help as String[]) {
                public Object execute( CommandShell shell, String args ) {
                    // rehydrate(Object delegate, Object owner, Object thisObject) 
                    Closure c = doIt.rehydrate(shell, doIt.owner, doIt);
                    int length = doIt.getParameterTypes().length;
                    if( length == 2 ) {
                        c(shell, args);
                    } else { 
                        c(args);
                    }                    
                    return null;
                }        
            }; 
    target.updateCommand(name, cmd); 
    return cmd;           
} 

adminCommand = { String name, List help, Closure doIt ->
    addShellCommand(server.adminShell, name, help, doIt);
} 

updateAdminCommand = { String name, List help, Closure doIt ->
    updateShellCommand(server.adminShell, name, help, doIt);
} 

// Add some standard shell commands
def help = [
    "<connection#> [message] - disconnects the specified connection.",
    "Where:",
    "  <connection#> - is the integer connection number.",
    "  [message] - is an optional message to send with the disconnect."
    ];
adminCommand("kick", help) { args ->
    def id;
    def message;
    int split = args.indexOf(' ');
    if( split < 0 ) {
        id = args as int;
        message = "The admin has closed your connection."
    } else {
        id = args.substring(0, split) as int;
        message = args.substring(split + 1);
    }
    def conn = server.server.getConnection(id);
    if( conn == null ) {
        throw new IllegalArgumentException("Invalid connection ID:" + id);
    }
    log.info("Disconnecting:" + conn);
    conn.close(message);
}    

getAccount = { String id ->
    if( id.isInteger() ) {
        def conn = server.server.getConnection(id as int);
        if( conn == null ) {
            throw new IllegalArgumentException("ID is not an active connection:" + id);
        }
        def account = AccountHostedService.getAccount(conn);
        if( account == null ) {
            throw new IllegalArgumentException("Connection is not logged in yet:" + id);
        }
        return account;      
    } else {
        def account = accountManager.getAccount(id);
        if( account == null ) {
            throw new IllegalArgumentException("User ID is not a valid user:" + id)
        }
        return account; 
    }
}

gameSession = { conn ->
    return GameSessionHostedService.getGameSession(conn);
}

help = [
    "<connection#|userId> - bans the specified user from future logins.",
    "Where:",
    "  <connection#|userId> - is the connection number of the user to",
    "       ban or their user ID."
    ];
adminCommand("ban", help) { shell, args ->
    def account = getAccount(args);
    account.setProperty("banned", true);
    account.setProperty("bannedUpdateTime", System.currentTimeMillis());
    account.save();
    shell.echo("Banned:" + account.getUserId());    
}

help = [
    "<connection#|userId> - unbans a player that was previously banned.",
    "Where:",
    "  <connection#|userId> - is the connection number of the user to",
    "       unban or their user ID."
    ];
adminCommand("unban", help) { shell, args ->
    def account = getAccount(args);
    account.setProperty("banned", false);
    account.setProperty("bannedUpdateTime", System.currentTimeMillis());
    account.save();
    shell.echo("Unbanned:" + account.getUserId());
}

help = [
    "- Shows network stats about the connected players"
]
adminCommand("stats", help) { args ->
    if( server.server.getConnections().isEmpty() ) {
        echo("No connections")
        return;
    } 
    def host = server.server.getServices().getService(com.simsilica.ethereal.EtherealHost.class);
    for( def conn : server.server.getConnections() ) {
        echo("Client[" + conn.getId() + "] address:" + conn.getAddress());
        def listener = host.getStateListener(conn);
        if( listener == null ) {
            echo("    [" + conn.getId() + "] No stats");
            continue;
        }
        echo("    [" + conn.getId() + "] Ping time: " + (listener.getConnectionStats().getAveragePingTime() / 1000000.0) + " ms");
        String miss = String.format("%.02f", listener.getConnectionStats().getAckMissPercent());
        echo("    [" + conn.getId() + "] Ack miss: " + miss + "%");
        echo("    [" + conn.getId() + "] Average msg size: " + listener.getConnectionStats().getAverageMessageSize() + " bytes");
    }
} 

help = [
    "- Refreshed any dynamically loadable config."
]
adminCommand("reload", help) { args ->
    echo("Checking:" + server.fileMods.size() + " mods");
    int count = 0;
    int errors = 0; 
    server.fileMods.each { mod ->
        log.info("Checking:" + mod);
        if( mod.hasChanged() ) {
            log.info("Reloading:" + mod);
            echo("Reloading:" + mod.info.id);
            try {
                mod.reload();
                count++;
            } catch( Exception e ) {
                log.error("Error reloading:" + mod, e);
                errors++;
            }
        }
    }
    echo("Reloaded:" + count + " changed mods with:" + errors + " errors");
}

help = [
    "- shows current server memory usage."
]
updateAdminCommand("mem", help) { args ->
    Runtime rt = Runtime.getRuntime();

    double BYTE_TO_MB = 1.0/(1024 * 1024);
    long free = rt.freeMemory();
    long total = rt.totalMemory();
    long max = rt.maxMemory();
    long used = total - free;
    double percent = 100.0 * used / max;
    echo(String.format("%.2f%%  %.2f/%.2f mb", percent, used * BYTE_TO_MB, max * BYTE_TO_MB))
}

help = [
    "- Lists the connected players with additional details"
]
updateAdminCommand("who", help) { args ->
    if( server.server.getConnections().isEmpty() ) {
        echo("No connections")
        return;
    }
    def host = server.server.getServices().getService(com.simsilica.ethereal.EtherealHost.class);
    for( def conn : server.server.getConnections() ) {
        def session = gameSession(conn);
        def loc = session.avatar[SpawnPosition]?.location;
        def locString = loc == null ? "null" : String.format("%.2f, %.2f, %.2f", loc.x, loc.y, loc.z);
        echo("[" + conn.id + "] = " + session.avatar.name + " at:(" + locString + ")  " + conn.address);
    }
}

help = [
    "- dumps the current ObjectType info to object-types.txt file"
]
adminCommand("dumpTypes", help) { args ->
    new File("object-types.txt").withPrintWriter("UTF-8") { report ->
        def names = [] as TreeSet;
        objectTypes.each { it -> names.add(it.name) };
        names.each { name ->
            echo("Dumping type:" + name);
            def type = objectTypes.getType(name);
            report.println("" + type.name + " {");
            report.println("    superTypes [");
            type.flattenedTypes.each { superType ->
                report.println("        " + superType.name);
            }
            report.println("    ]");
            report.println("    vars {");
            type.vars.entrySet().each {
                if( it.value instanceof Closure ) {
                    report.println("        " + it.key + " -> dynamic getter");
                } else {
                    report.println("        " + it);
                }
            }
            report.println("    }");
            def visited = [] as Set;
            report.println("    contextActions [");
            type.flattenedIndex.values().each { action ->
                if( action.paramTypes.length > 0 ) {
                    return;
                }
                // Hacky but works
                if( action.name == action.name.capitalize() ) {
                    visited.add(action);
                    report.println("        " + action.name);
                }
            }
            report.println("    ]");
            report.println("    actions [");
            type.flattenedIndex.values().each { action ->
                if( visited.contains(action) ) {
                    return;
                }
                report.print("        " + action.name + "(");
                for( int i = 0; i < action.paramTypes.length; i++ ) {
                    if( i > 0 ) {
                        report.print(", ");
                    }
                    def paramType = action.paramTypes[i];
                    if( paramType == EntityId.class ) {
                        report.print("#ENTITY");
                    } else if( paramType == Object.class ) {
                        report.print("*");
                    } else if( paramType == Closure.class ) {
                        report.print("#CALLBACK");
                    } else {
                        report.print(paramType.simpleName);
                    }
                }
                report.println(")");
            }
            report.println("    ]");
            report.println("}");
        }
    }
    
//        private String name;
//    private ListMultimap<String, ObjectAction<A, T>> actionIndex;
//    private Map<String, Object> vars = new HashMap<>();
//    private Set<ObjectType<A, T>> superTypes = new LinkedHashSet<>();
//    private Set<ObjectType<A, T>> flattenedTypes = null;
//    private ListMultimap<String, ObjectAction<A, T>> flattenedIndex;

}

on( playerEntityJoined ) { event ->

    def account = AccountHostedService.getAccount(event.connection);
    if( account ) {
        log.info("Entity joined for account:" + account);
    
        def perms = account.getProperty("permissions", List.class);
        if( perms?.contains("admin") ) {
            log.info("Need to add admin commands to player:" + account.getName());
        
            def shell = GameSessionHostedService.getShell(event.connection);
            server.adminShell.getCommands().entrySet().each {
                log.info("Adding command:" + it);
                shell.addCommand(it.key, it.value);
            }
        }
    }
    
    // And add any standard player server-side commands
    def shell = GameSessionHostedService.getShell(event.connection);

    def cmdHelp = [
            "- Shows stats about this client connection."
        ];
    addShellCommand(shell, "ping", cmdHelp) {
        def conn = event.connection;
        def host = server.server.getServices().getService(com.simsilica.ethereal.EtherealHost.class);
        def listener = host?.getStateListener(conn);
        if( listener == null ) {
            echo("No stats");
            return;
        }
        echo("Client[" + conn.getId() + "] address:" + conn.getAddress());
        echo("    Ping time: " + (listener.getConnectionStats().getAveragePingTime() / 1000000.0) + " ms");
        String miss = String.format("%.02f", listener.getConnectionStats().getAckMissPercent());
        echo("    Ack miss: " + miss + "%");
        echo("    Average msg size: " + listener.getConnectionStats().getAverageMessageSize() + " bytes");
    }

    cmdHelp = [
            "- Dumps info about the mod manager."
        ];
    addShellCommand(shell, "mods", cmdHelp) { args ->
        echo("Mod packs:");
        modManager.modPackIds.each {
            echo("    " + it);
        }
    }   
 
    cmdHelp = [
            "[location] - Warps to a specified location.",
            "Location is in x y z or x z form.",
            "If location is not specified then the player is warped to the",
            "current location at the max build height."
        ];
    addShellCommand(shell, "warp", cmdHelp) { args ->
        
        def phys = system(PhysicsSpace)
        def target;
        if( args.startsWith("#") ) {
            // Look up the location
            def name = args.substring(1);
            echo("Warp:" + event.player + "  to mark:" + name);
            def mark = getMark(name);
            if( mark == null ) {
                echo("Mark not found:" + name);
                return;
            }
            target = mark[SpawnPosition]?.location; 
            echo("Warping to:" + target);
        } else {
            def tokens = args?.split("\\s|,");
            echo("Warp:" + event.player + "  args:" + tokens);
            def current = event.player[SpawnPosition]?.location;
            echo("Current Position:" + current);
 
            target = current.clone();
            if( !tokens ) {
                target.y = GameConstants.MAX_BUILD_HEIGHT + 10;
            } else if( tokens.size() > 2 ) {
                target.x = Double.parseDouble(tokens[0]);
                target.y = Double.parseDouble(tokens[1]);
                target.z = Double.parseDouble(tokens[2]);
            } else if( tokens.size() > 1 ) {
                target.x = Double.parseDouble(tokens[0]);
                // This will work until we have fall damage
                target.y = GameConstants.MAX_BUILD_HEIGHT + 10;
                target.z = Double.parseDouble(tokens[1]);
            } else {
                echo("Invalid number of arguments:" + tokens);
                return;
            }
        }
        //def target = current.add(0, 400, 0) 
 
        phys.teleport(event.player, target, new Quatd());
        //echo("session:" + gameSession(event.connection));       
    }   

    cmdHelp = [
            "- Respawns at the starting location."
        ];
    addShellCommand(shell, "respawn", cmdHelp) { args ->
        //echo("World:" + world + "  thread:" + Thread.currentThread());
        def target = system(WorldManager).info.spawnPoint;
        
        // FIXME: build the spawn tower height into the spawn location
        // instead of adding it back all the time
        target.y += 20;
        target.x += 0.5;
        target.z += 0.5;
        echo("Respawning at:" + target);
        system(PhysicsSpace).teleport(event.player, target, new Quatd());
    }
    
    cmdHelp = [
            "[name] - Marks the current location with the specified name.",
            "This can be used for later commands that take a location name.",
            "If no name is specified then the current marks are listed."
            ]
    addShellCommand(shell, "mark", cmdHelp) { args ->
        if( !args ) {
            // Just list the current marks
            echo("No mark name specified, listing current marks:")
            
            // Search for the right kind of entities
            getMarks().each { mark ->
                echo("  " + mark.name + " - " + mark[SpawnPosition]?.location);
                log.info("  " + mark.name + " - " + mark[SpawnPosition]?.location);
            }
            return;
        } 
        def current = event.player[SpawnPosition];
        if( current == null ) {
            echo("Error getting current location");
            return;
        }
        echo("Setting mark:" + args + " to position:" + current?.location);
        setMark(args, current);
    } 

    cmdHelp = [
            "[name] - Removes the specified mark.",
            ]
    addShellCommand(shell, "unmark", cmdHelp) { args ->
        if( !args ) {
            echo("No mark name specified")
            return;
        }
        if( removeMark(args) ) {
            echo("Mark removed");
        } else {
            echo("Error: no existing mark found for:" + args);
        } 
    } 
      
}


