/*
 * Copyright (c) 2024, Simsilica, LLC
 * All rights reserved.
 */

/*
Used to test the "remove losing context" bug that was written up as
   task T001108 but a little differently.  The problem was fixed in the action
   framework.

createType("TestType").with {
    addGetter("testGetter") { ->
        return "I am:" + self + " with type:" + self[ObjectTypeInfo];
    }

    addAction("testAction") { ->
        echo("test1 self:" + self + " self.testGetter:" + self.testGetter);
        removeEntity(self);
        echo("test2 self:" + self + " self.testGetter:" + self.testGetter);
    }
}
*/

checkEncounter = { encounter ->
    if( encounter[SpawnPosition] == null ) {
        log.error("Encounter does not have a spawn position:" + encounter, new Throwable("very visible"));
        // What does it have?
        entityData.getComponentHandlers().keySet().each { componentType ->
            def val = encounter[componentType]
            String text = String.valueOf(val);
            log.info("" + encounter + "->" + componentType.simpleName + " = " + text);
        }
        System.exit(0);
    }
}

createType("ButterflyEncounter").with {

    /**
     *  Returns the current mobs that were spawned and are being
     *  managed by this encounter.
     */
    addGetter("spawns") { ->
        // Keep this minimal to catch as many things as possible even
        // if something goes wrong.  Spawns don't always start with a
        // position so we don't include that.
        if( log.isTraceEnabled() ) {
            log.trace("find spawns:" + OwnedBy.filter(self));
            findEntities(OwnedBy.filter(self),
                     OwnedBy.class,
                     ShapeInfo.class,
                     AgentType.class).each {
                if( log.isTraceEnabled() ) {
                    log.trace(" it:" + it);
                }
            }
        }

        return findEntities(OwnedBy.filter(self),
                     OwnedBy.class,
                     ShapeInfo.class,
                     AgentType.class);
    }

    addAction("activateEncounter") { ->
        if( log.isTraceEnabled() ) {
            log.trace("activateEncounter:" + self);
        }
        checkEncounter.call(self);
    }

    addAction("initializeEncounter") { flowers ->
        if( log.isTraceEnabled() ) {
            log.trace("initializeEncounter:" + self);
        }
        checkEncounter.call(self);
        if( self[SpawnPosition] == null ) {
            log.error("Encounter does not have a spawn position:" + self, new Throwable("very visible"));
            return;
        }
        // Look for flowers, spawn butterflies based on frequency + dice roll

        //// How many flowers are there in this zone and where are they?
        //def pos = self[SpawnPosition].location;
        //def colId = ColumnId.fromWorld(pos);
        //def flowerTypes = worldStats.getBlockSet("flowers");
        //List<Vec3i> flowers = worldStats.getSurfaceBlocks(colId, flowerTypes);

        double max = Math.ceil(Math.sqrt(flowers.size()));
        // But never more than 4 per encounter, at least for now
        max = Math.min(4, max);
        int count = (int)(1 + Math.random() * (max - 1));

        if( log.isTraceEnabled() ) {
            log.trace("check for existing butterflies for:" + self);
        }
        self.spawns.each {
            if( log.isTraceEnabled() ) {
                log.trace("  Found existing butterfly:" + it);
            }
            // If we already have a butterfly then take it away from the 'to spawn' count
            count--;
        }

        int butterFlyType = (int)(Math.random() * 2);

        def spawns = [:];

        // Spawn in the appropriate number of butterflies
        for( int i = 0; i < count; i++ ) {
            int index = (int)Math.random() * flowers.size();
            Vec3d loc = flowers.remove(index).toVec3d();

            double roll = Math.random();
            roll = roll * roll * roll;
            int variant = (int)(roll * 4);

            def entity = type("Butterfly").newInstance(
                                    new CreatedBy(self),
                                    // OwnedBy is better because if a player picks up the
                                    // butterfly then we can change the owned by and it won't
                                    // be managed by the encounter anymore.
                                    new OwnedBy(self),
                                    new SkinVariation(butterFlyType * 4 + variant)
                                );
            if( log.isTraceEnabled() ) {
                log.trace("Created butterfly:" + entity);
            }

            if( variant == 3 ) {
                float r = (float)Math.random();
                float g = (float)Math.random();
                float b = (float)Math.random();
                def color = new ColorRGBA(r, g, b, 1);
                if( log.isTraceEnabled() ) {
                    log.trace("Setting random skin color to:" + color);
                }
                entity << new SkinColor(color);
            }

            spawns.put(entity, loc);
        }

        // We'll spool the actual placement but not the creation.
        // If we spool the creation then if we get deactivated  before they have
        // all spawned them some might get orphaned and just hang around the world
        // forever.
        spawns.entrySet().each {
            def entity = it.key;
            def loc = it.value;
            spool {
                // See if it's even still real
                if( entity[AgentType] ) {
                    entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                    if( log.isTraceEnabled() ) {
                        log.trace("  placed butterfly:" + entity + " at:" + entity[SpawnPosition]?.location);
                    }
                } else {
                    log.warn("Entity is gone by the time of placement:" + entity + " loc:" + loc);
                }
            }
        }

        if( log.isTraceEnabled() ) {
            self.spawns.each {
                log.trace(" created:" + it);
            }
        }
    }

    addAction("deactivateEncounter") { ->

        if( log.isTraceEnabled() ) {
            log.trace("deactivateEncounter:" + self);
        }
        log.info("deactivateEncounter:" + self);
        // Despawn any of the spawned butterflies that are still alive
        // and remove the encounter.
        def spawns = self.spawns;
        if( log.isTraceEnabled() ) {
            log.trace("spawns:" + spawns);
        }
        spawns.each {
            if( log.isTraceEnabled() ) {
                log.trace("  Removing butterfly:" + it);
            }
            removeEntity(it);
        }

        // If we remove ourselves first then we can't actually grab spawns and stuff
        // because we won't have a type yet.  Probably this should be spooled.
        // I fixed that, by the way.
        if( log.isTraceEnabled() ) {
            log.trace("removing self:" + self);
        }
        log.info("removing self:" + self);
        removeEntity(self);
    }
}

createType("BirdEncounter").with {

    /**
     *  Returns the current mobs that were spawned and are being
     *  managed by this encounter.
     */
    addGetter("spawns") { ->
        // Keep this minimal to catch as many things as possible even
        // if something goes wrong.  Spawns don't always start with a
        // position so we don't include that.
        if( log.isTraceEnabled() ) {
            log.trace("find spawns:" + OwnedBy.filter(self));
            findEntities(OwnedBy.filter(self),
                        OwnedBy.class,
                        ShapeInfo.class,
                        AgentType.class).each {
                log.trace(" it:" + it);
            }
        }

        return findEntities(OwnedBy.filter(self),
                     OwnedBy.class,
                     ShapeInfo.class,
                     AgentType.class);
    }

    addAction("activateEncounter") { ->
        if( log.isTraceEnabled() ) {
            log.trace("activateEncounter:" + self);
        }
        checkEncounter.call(self);
    }

    addAction("initializeEncounter") { flowers ->
        if( log.isTraceEnabled() ) {
            log.trace("initializeEncounter:" + self);
        }
        checkEncounter.call(self);
        if( self[SpawnPosition] == null ) {
            log.error("Encounter does not have a spawn position:" + self, new Throwable("very visible"));
            return;
        }
        // Look for leaves, spawn birds based on frequency + dice roll

        // How many blocks are there in this zone and where are they?
        def pos = self[SpawnPosition].location;
        def colId = ColumnId.fromWorld(pos);
        def leafTypes = worldStats.getBlockSet("leaves");
        List<Vec3i> blocks = worldStats.getSurfaceBlocks(colId, leafTypes);

        double max = 3;
        int count = (int)(1 + Math.random() * (max - 1));

        if( log.isTraceEnabled() ) {
            log.trace("check for existing spawns for:" + self);
        }
        self.spawns.each {
            if( log.isTraceEnabled() ) {
                log.trace("  Found existing spawns:" + it);
            }
            // If we already have a spawn then take it away from the 'to spawn' count
            count--;
        }

        int birdType = (int)(Math.random() * 2);

        def spawns = [:];

        // Spawn in the appropriate number of butterflies
        for( int i = 0; i < count; i++ ) {
            int index = (int)Math.random() * blocks.size();
            Vec3d loc = blocks.remove(index).toVec3d();

            double roll = Math.random();
            roll = roll * roll * roll;
            int variant = (int)(roll * 4);
            boolean randomColor = Math.random() < 0.2;
            if( birdType == 1 ) {
                variant = 0;
                birdType = 3;
            } else if( variant < 3 ) {
                variant = 0;
            }
            def entity = type("Bird").newInstance(
                                    new CreatedBy(self),
                                    // OwnedBy is better because if a player picks up the
                                    // butterfly then we can change the owned by and it won't
                                    // be managed by the encounter anymore.
                                    new OwnedBy(self),
                                    new SkinVariation(birdType * 4 + variant)
                                );
            if( log.isTraceEnabled() ) {
                log.trace("Created bird:" + entity);
            }

            //if( variant == 3 ) {
            if( randomColor ) {
                float r = (float)Math.random();
                float g = (float)Math.random();
                float b = (float)Math.random();
                def color = new ColorRGBA(r, g, b, 1);
                if( log.isTraceEnabled() ) {
                    log.trace("Setting random skin color to:" + color);
                }
                entity << new SkinColor(color);
            }

            spawns.put(entity, loc);
        }

        // We'll spool the actual placement but not the creation.
        // If we spool the creation then if we get deactivated  before they have
        // all spawned them some might get orphaned and just hang around the world
        // forever.
        spawns.entrySet().each {
            def entity = it.key;
            def loc = it.value;
            spool {
                // See if it's even still real
                if( entity[AgentType] ) {
                    entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                    if( log.isTraceEnabled() ) {
                        log.trace("  placed spawn:" + entity + " at:" + entity[SpawnPosition]?.location);
                    }
                } else {
                    log.warn("Entity is gone by the time of placement:" + entity + " loc:" + loc);
                }
            }
        }

        if( log.isTraceEnabled() ) {
            self.spawns.each {
                log.trace(" created:" + it);
            }
        }
    }

    addAction("deactivateEncounter") { ->
        if( log.isTraceEnabled() ) {
            log.trace("deactivateEncounter:" + self);
        }
        log.info("deactivateEncounter:" + self);
        // Despawn any of the spawned butterflies that are still alive
        // and remove the encounter.
        def spawns = self.spawns;
        if( log.isTraceEnabled() ) {
            log.trace("spawns:" + spawns);
        }
        spawns.each {
            if( log.isTraceEnabled() ) {
                log.trace("  Removing spawn:" + it);
            }
            removeEntity(it);
        }

        // If we remove ourselves first then we can't actually grab spawns and stuff
        // because we won't have a type yet.  Probably this should be spooled.
        // I fixed that, by the way.
        if( log.isTraceEnabled() ) {
            log.trace("removing self:" + self);
        }
        log.info("removing self:" + self);
        removeEntity(self);
    }
}


// Try to remove all of the butterflies to clean things up if something broke
if( true ) {
    boolean cleanup = false;

    // Try to find all of them... not just in this zone
    log.info("All entities...");
    findEntities(null, SpawnPosition.class, AgentType.class,
                 ObjectTypeInfo.class).each {
        log.info("Mobs:" + it + "  name:" + it[ObjectName].toString(entityData) + " type:" + it[ObjectTypeInfo].getTypeName());
        String type = it[ObjectTypeInfo].getTypeName();
        if( type == "Animal" ) {
            type = it[ObjectName].getName();
        }
        log.info("  type:" + type);
        log.info("  zone:" + it[SpawnPosition].binId);
        if( "Butterfly" == type ) {
            log.info("  bufferly agent type:" + it[AgentType]);
            log.info("  body pos:" + it[BodyPosition]);
            if( cleanup ) {
                log.info("  ...removing");
                removeEntity(it);
            }
        }
        if( "Bird" == type ) {
            log.info("  bird agent type:" + it[AgentType]);
            log.info("  body pos:" + it[BodyPosition]);
            if( cleanup ) {
                log.info("  ...removing");
                removeEntity(it);
            }
        }
    }

    log.info("All buttefly encounters...");
    findEntities(ObjectTypeInfo.typeFilter("ButterflyEncounter"), ObjectTypeInfo.class).each {
        if( cleanup ) {
            log.info("Removing encounter:" + it);
            removeEntity(it);
        } else {
            log.info("Found existing butterfly encounter:" + it);
        }
    }

    log.info("All bird encounters...");
    findEntities(ObjectTypeInfo.typeFilter("BirdEncounter"), ObjectTypeInfo.class).each {
        if( cleanup ) {
            log.info("Removing encounter:" + it);
            removeEntity(it);
        } else {
            log.info("Found existing bird encounter:" + it);
        }
    }

    if( true ) {
        // Clean out all of the lingering encounter triggers
        findEntities(null, EncounterTrigger.class).each { trigger ->
            if( cleanup ) {
                log.info("Removing trigger:" + trigger + " tigger:" + trigger[EncounterTrigger]);
                removeEntity(trigger);
            } else {
                log.info("Found trigger:" + trigger + " tigger:" + trigger[EncounterTrigger]);
            }
        }
    }
}

findEncounters = { long zoneId, ObjectTypeInfo encounterType ->
    def triggers = findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class);
    return triggers.findAll {
        def trigger = it[EncounterTrigger];
        log.info("" + zoneId + " checking:" + it + "  trigger:" + trigger);
        return encounterType == trigger.encounterId[ObjectTypeInfo];
    };
}

// Random encounters are special because they are their own triggers
// so we can narrow the search down quite a bit.
findRandomEncounters = { long zoneId, ObjectTypeInfo encounterType ->
    def triggers = findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class, ObjectTypeInfo.class);
    return triggers.findAll {
        log.info("" + zoneId + " checking:" + it + "  type:" + it[ObjectTypeInfo]);
        return encounterType == it[ObjectTypeInfo]
    };
}

boolean disableEncounters = false;
boolean enableButterflies = true;
boolean enableBirds = true;

mobEncounterSystem = system(MobEncounterSystem.class);

system(AgentActivationSystem.class).addActivationZoneListener(new ActivationZoneListener() {
    // Called from the agents thread to notify about zone activation.
    // Beware of the threading... and asynchronous state.
    public void activateZone( long zoneId ) {
        log.info("" + zoneId + " activateZone()");
        if( disableEncounters ) {
            return;
        }

        onBackground {
            // See if we should try to create encounters...
            // Note: as it stands, this is blocking other listeners and we may want to
            // move it to a background worker pool.
            def colId = new ColumnId(zoneId);
            if( enableButterflies ) {
                def flowerTypes = worldStats.getBlockSet("flowers");
                List<Vec3i> blocks = worldStats.getSurfaceBlocks(colId, flowerTypes);
                if( blocks.size() > 1 ) {
                    // Run this on some future frame
                    spoolAsWorld {
                        log.info("" + zoneId + " Checking butterflies zone");
                        if( !mobEncounterSystem.isActiveZone(zoneId) ) {
                            log.info("" + zoneId + " is no longer active");
                            return;
                        }
                        def objectTypeInfo = ObjectTypeInfo.create("ButterflyEncounter");
                        def existing = findRandomEncounters(zoneId, objectTypeInfo);
                        log.info("existing:" + existing);

                        def butterflies = createEntity(
                            ObjectTypeInfo.create("ButterflyEncounter"),
                            new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd())
                        );
                        log.info("" + zoneId + " created butterflies:" + butterflies + " at:" + butterflies[SpawnPosition]);
                        // Random encounters are their own triggers
                        butterflies << new EncounterTrigger(butterflies, zoneId);
                        butterflies.run("initializeEncounter", blocks);
                    }
                }
            }

            if( enableBirds ) {
                def leafTypes = worldStats.getBlockSet("leaves");
                List<Vec3i> blocks = worldStats.getSurfaceBlocks(colId, leafTypes);
                if( blocks.size() > 10 ) {
                    // Run this on some future frame
                    spoolAsWorld {
                        log.info("" + zoneId + " Checking birds zone");
                        if( !mobEncounterSystem.isActiveZone(zoneId) ) {
                            log.info("" + zoneId + " is no longer active");
                            return;
                        }

                        def objectTypeInfo = ObjectTypeInfo.create("BirdEncounter");
                        def existing = findRandomEncounters(zoneId, objectTypeInfo);
                        log.info("existing:" + existing);

                        def birds = createEntity(
                            ObjectTypeInfo.create("BirdEncounter"),
                            new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd())
                        );
                        log.info("" + zoneId + " created birds:" + birds + " at:" + birds[SpawnPosition]);
                        // Random encounters are their own triggers
                        birds << new EncounterTrigger(birds, zoneId);
                        birds.run("initializeEncounter", blocks);
                    }
                }
            }
        }
    }

    public void deactivateZone( long zoneId ) {
        log.info("" + zoneId + ":deactivateZone()");
        if( disableEncounters ) {
            return;
        }
    }
});

//system(AgentActivationSystem.class).addActivationZoneListener(new ActivationZoneListener() {
//    public void activateZone( long zoneId ) {
//        log.info("" + zoneId + " activateZone()");
//
//        if( disableEncounters ) {
//            return;
//        }
//
//        def triggers = findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class);
//        log.info("" + zoneId + " existing triggers:" + triggers);
//        triggers.each { trigger ->
//            log.info("" + zoneId + " found existing:" + trigger + "  trigger:" + trigger[EncounterTrigger]);
//        }
//        if( !triggers ) {
//
//            // Look at the world data to see if there are any flowers.
//            // no flowers -> don't create it
//            def colId = new ColumnId(zoneId);
//            if( enableButterflies ) {
//                def flowerTypes = worldStats.getBlockSet("flowers");
//                //List<Vec3i> flowers = worldStats.getSurfaceBlocks(colId, flowerTypes);
//                boolean found = worldStats.hasSurfaceBlocks(colId, flowerTypes, 1);
//
//                // Figure out how many max we would want to spawn based on season, climate, etc..
//                // ...but for now spawn the encounter if there are any flowers at all.
//                if( found ) {
//                    def butterflies = createEntity(
//                        ObjectTypeInfo.create("ButterflyEncounter"),
//                        new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd())
//                    );
//                    log.info("" + zoneId + " created butterflies:" + butterflies + " at:" + butterflies[SpawnPosition]);
//                    def trigger = createEntity(
//                        new EncounterTrigger(butterflies, zoneId)
//                    );
//                }
//            }
//
//            if( enableBirds ) {
//                def leafTypes = worldStats.getBlockSet("leaves");
//                //List<Vec3i> leaves = worldStats.getSurfaceBlocks(colId, leafTypes);
//                boolean found = worldStats.hasSurfaceBlocks(colId, leafTypes, 10);
//                if( found ) {
//                    def birds = createEntity(
//                        ObjectTypeInfo.create("BirdEncounter"),
//                        new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd())
//                    );
//                    log.info("" + zoneId + " created birds:" + birds + " at:" + birds[SpawnPosition]);
//                    def trigger = createEntity(
//                        new EncounterTrigger(birds, zoneId)
//                    );
//                }
//            }
//        //} else {
//        //    log.info("Found existing:" + trigger);
//        }
//
//        log.info("kilroy butterflies:" + findEncounters(zoneId, ObjectTypeInfo.create("ButterflyEncounter")));
//        log.info("kilroy birds:" + findEncounters(zoneId, ObjectTypeInfo.create("BirdEncounter")));
//    }
//
//    public void deactivateZone( long zoneId ) {
//        log.info("" + zoneId + ":deactivateZone()");
//        findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class).each { trigger ->
//            // Note: removing the trigger is always necessary... this is not necessarily
//            // removing the encounter itself.  That's up to what the encounter does when
//            // deactivated.  ie: trigger != encounter
//            log.info("" + zoneId + " removing:" + trigger + " tigger:" + trigger[EncounterTrigger]);
//            removeEntity(trigger);
//        }
//    }
//});
