/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

import java.util.function.Consumer;
import mythruna.sim.time.*;

// Get the time settings system to which we'll register
// time-based update events.
timeSettings = system(WorldTimeSettingsSystem.class);

worldTime = system(WorldTime.class);

// Keep track of the registered callbacks so that we can remove
// them again
registeredTimeSettings = [];

addTimeSetting = { Consumer<WorldTime> callback ->
    timeSettings.addSetting(callback);
    registeredTimeSettings.add(callback);
    return callback;
}

removeTimeSetting = { Consumer<WorldTime> callback ->
    timeSettings.removeSetting(callback);
    registeredTimeSettings.remove(callback);
    return callback;
}

cleanupTimeSettings = {
    registeredTimeSettings.each { setting ->
        timeSettings.removeSetting(setting);
    }
    registeredTimeSettings.clear();
}

TweenSchedule.metaClass {
    interpolate() { Closure interp ->
log.info("interpolate:" + interp);
        if( interp.getParameterTypes().length != 3 ) {
            throw new IllegalArgumentException("Interpolation closure must have three parameters.");
        }
        delegate.tweenFactory = { length, start, end ->
            Tween result = new AbstractTween(length) {
                @Override
                protected void doInterpolate( double t ) {
                    interp(t, start, end);
                }
            }
        }
        return delegate;
    }
}

addSchedule = { String interval ->
    def sched = new TweenSchedule();
    sched.interval = TimeInterval.parseInterval(interval);
    addTimeSetting(sched);
    return sched;
}

scheduledDouble = { String interval, Closure config ->
    def supplier = new ScheduledSupplier(worldTime, TimeInterval.parseInterval(interval));
    supplier.with config;
    return supplier;
}

/**
 *  Converts a moon phase between 0 and 24 to a moon strength.
 */
moonPhaseToStrength = { phase ->
    return 1.0 - (int)Math.abs(12.5 - phase)/12;
}

/**
 *  Gets the current moon strength.
 */
getMoonStrength = { ->
    return moonPhaseToStrength(worldTime.moonPhase);
}

/**
 *  Returns the next moon rise or if the moon is already up,
 *  this will return the previous moon rise.
 */
getMoonRise = { ->
    def now = worldTime.clone();
    int day = now.day;
    if( now.hour < 6 ) {
        // We are already in the next morning after midnight so we really
        // want to start the schedule yesterday.
        day--;
    }

    return new WorldTime(now.year, now.month, day, 20, 0);
}


//def sched = new TweenSchedule();
//sched.interval = TimeInterval.parseInterval("0:30");
//sched.tweenFactory = { length, start, end ->
//    Tween result = new AbstractTween(length) {
//        @Override
//        protected void doInterpolate( double t ) {
//            def val = start + (end - start) * t;
//            //println "Test:" + val;
//            env.setMoonPhase((int)val);
//        }
//    }
//    return result;
//}

//def sched = addSchedule("0:30").with {
//    interpolate = { t, start, end ->
//        log.into("t:" + t + " start:" + start + " end:" + end);
//    }
//    at("0:00", 0)
//    at("0:15", 24)
//}



// Cleanup any of the registered settings
mod.onTerminate {
    cleanupTimeSettings();
}


