/*
 * Copyright (c) 2024, Simsilica, LLC
 * All rights reserved.
 */

/*
Used to test the "remove losing context" bug that was written up as
   task T001108 but a little differently.  The problem was fixed in the action
   framework.

createType("TestType").with {
    addGetter("testGetter") { ->
        return "I am:" + self + " with type:" + self[ObjectTypeInfo];
    }

    addAction("testAction") { ->
        echo("test1 self:" + self + " self.testGetter:" + self.testGetter);
        removeEntity(self);
        echo("test2 self:" + self + " self.testGetter:" + self.testGetter);
    }
}
*/

findEncounters = { long zoneId, ObjectTypeInfo encounterType ->
    def triggers = findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class);
    return triggers.findAll {
        def trigger = it[EncounterTrigger];
        log.info("" + zoneId + " checking:" + it + "  trigger:" + trigger);
        return encounterType == trigger.encounterId[ObjectTypeInfo];
    };
}

// Random encounters are special because they are their own triggers
// so we can narrow the search down quite a bit.
findRandomEncounters = { long zoneId, ObjectTypeInfo encounterType ->
    def triggers = findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class, ObjectTypeInfo.class);
    return triggers.findAll {
        log.info("" + zoneId + " checking:" + it + "  type:" + it[ObjectTypeInfo]);
        return encounterType == it[ObjectTypeInfo]
    };
}

checkEncounter = { encounter ->
    if( encounter[SpawnPosition] == null ) {
        log.error("Encounter does not have a spawn position:" + encounter, new Throwable("very visible"));
        // What does it have?
        entityData.getComponentHandlers().keySet().each { componentType ->
            def val = encounter[componentType]
            String text = String.valueOf(val);
            log.info("" + encounter + "->" + componentType.simpleName + " = " + text);
        }
        //System.exit(0);
    }
}

/**
 *  Returns the set of "OwnedBy" MOBs for the given spawner.
 */
findSpawns = { spawner ->
    // Keep this minimal to catch as many things as possible even
    // if something goes wrong.  Spawns don't always start with a
    // position so we don't include that.
    if( log.isTraceEnabled() ) {
        log.trace("find spawns:" + OwnedBy.filter(spawner));
        findEntities(OwnedBy.filter(spawner),
                     OwnedBy.class,
                     ShapeInfo.class,
                     AgentType.class).each {
                if( log.isTraceEnabled() ) {
                    log.trace(" it:" + it);
                }
            }
    }

    return findEntities(OwnedBy.filter(spawner),
                 OwnedBy.class,
                 ShapeInfo.class,
                 AgentType.class);
}

/**
 *  Common behaviors for non-durable random encounters.
 */
createType("BaseRandomEncounter").with {

    /**
     *  By default, activateEncounter does nothing for random encounters because
     *  all of the mob creation was done during initialize.
     */
    addAction("activateEncounter") { ->
        if( log.isTraceEnabled() ) {
            log.trace("activateEncounter:" + self);
        }
        log.info("activateEncounter:" + self);
        checkEncounter.call(self);
    }

    addAction("deactivateEncounter") { ->
        if( log.isTraceEnabled() ) {
            log.trace("deactivateEncounter:" + self);
        }
        log.info("deactivateEncounter:" + self);
        long start = System.nanoTime();

        // Querying the spawns can take up to 20-30 ms... and removing them can
        // take even longer... so we'll do the whole thing on a background thread.
        onBackground {
            // Despawn any of the spawned mobs that are still alive
            // and remove the encounter.
            def spawns = findSpawns(self);
            log.info("remove spawns:" + spawns);
            if( log.isTraceEnabled() ) {
                log.trace("spawns:" + spawns);
            }
            spawns.each { mob ->
                if( log.isTraceEnabled() ) {
                    log.trace("  Removing spawn:" + mob);
                }
                log.info("  Removing spawn:" + mob);
                // Turns out, removeEntity() can take a while and in this case we don't
                // need to check for existence later so we can run them on the background thread pool
                // Let it take as long as it needs.  Usually, when long it's only 20-30 ms... which
                // is long enough to drop frames... but it's occasionally as long as 130 ms.
                removeEntity(mob);
            }
        }

        // If we remove ourselves first then we can't actually grab spawns and stuff
        // because we won't have a type yet.  Probably this should be spooled.
        // I fixed that, by the way.
        if( log.isTraceEnabled() ) {
            log.trace("removing self:" + self);
        }
        log.info("removing self trigger:" + self);
        long start2 = System.nanoTime();
        // Need to remove the encounter trigger synchronously so we don't see ourselves as an
        // existing encounter if the zone is reactivated... but the rest of the entity can
        // be removed on the background thread.
        self.remove(EncounterTrigger.class);
        long mid = System.nanoTime();
        onBackground {
            removeEntity(self);
        }
        long end = System.nanoTime();
        if( (end - start) > 16000000L ) {
            log.warn(String.format("Deactivate encounter: %s took: %.03f ms, mobs: %.03f, comp: %.03f, remove: %.03f",
                            self,
                            (end - start)/1000000.0,
                            (start2 - start)/1000000.0,
                            (mid - start2)/1000000.0,
                            (end - mid)/1000000.0
                         ));
        }
    }

}

createType("ButterflyEncounter").with {
    supertypes("BaseRandomEncounter");

    addAction("initializeEncounter") { flowers ->
        if( log.isTraceEnabled() ) {
            log.trace("initializeEncounter:" + self);
        }
        long start = System.nanoTime();
        checkEncounter.call(self);
        if( self[SpawnPosition] == null ) {
            log.error("Encounter does not have a spawn position:" + self, new Throwable("very visible"));
            return;
        }
        long time1 = System.nanoTime();

        // Look for flowers, spawn butterflies based on frequency + dice roll
        double max = Math.ceil(Math.sqrt(flowers.size()));
        // But never more than 4 per encounter, at least for now
        max = Math.min(4, max);
        int count = (int)(1 + Math.random() * (max - 1));

        // We don't reinitialize an existing encounter so this is wasted here
        //if( log.isTraceEnabled() ) {
        //    log.trace("check for existing butterflies for:" + self);
        //}
        //self.spawns.each {
        //    if( log.isTraceEnabled() ) {
        //        log.trace("  Found existing butterfly:" + it);
        //    }
        //    // If we already have a butterfly then take it away from the 'to spawn' count
        //    count--;
        //}

        int butterFlyType = (int)(Math.random() * 2);

        def spawns = [:];

        // Spawn in the appropriate number of butterflies
        for( int i = 0; i < count; i++ ) {
            int index = (int)Math.random() * flowers.size();
            Vec3d loc = flowers.remove(index).toVec3d();

            double roll = Math.random();
            roll = roll * roll * roll;
            int variant = (int)(roll * 4);

            def entity = type("Butterfly").newInstance(
                                    new CreatedBy(self),
                                    // OwnedBy is better because if a player picks up the
                                    // butterfly then we can change the owned by and it won't
                                    // be managed by the encounter anymore.
                                    new OwnedBy(self),
                                    new SkinVariation(butterFlyType * 4 + variant),
                                    new TemporaryObject()
                                );
            if( log.isTraceEnabled() ) {
                log.trace("Created butterfly:" + entity);
            }

            if( variant == 3 ) {
                float r = (float)Math.random();
                float g = (float)Math.random();
                float b = (float)Math.random();
                def color = new ColorRGBA(r, g, b, 1);
                if( log.isTraceEnabled() ) {
                    log.trace("Setting random skin color to:" + color);
                }
                entity << new SkinColor(color);
            }

            spawns.put(entity, loc);
        }

        long time2 = System.nanoTime();

        // We'll spool the actual placement but not the creation.
        // If we spool the creation then if we get deactivated  before they have
        // all spawned them some might get orphaned and just hang around the world
        // forever.
        spawns.entrySet().each {
            def entity = it.key;
            def loc = it.value;
            spool {
                // See if it's even still real
                if( entity[AgentType] ) {
                    entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                    if( log.isTraceEnabled() ) {
                        log.trace("  placed butterfly:" + entity + " at:" + entity[SpawnPosition]?.location);
                    }
                } else {
                    log.warn("Entity is gone by the time of placement:" + entity + " loc:" + loc);
                }
            }
        }

        long end = System.nanoTime();
        log.info(String.format("" + self + " init in: %.03f ms, check: %.03f ms, create: %.03f ms, place: %.03f ms",
                            (end - start)/1000000.0,
                            (time1 - start)/1000000.0,
                            (time2 - time1)/1000000.0,
                            (end - time2)/1000000.0
                        ));

        if( log.isTraceEnabled() ) {
            findSpawns(self).each {
                log.trace(" created:" + it);
            }
        }
    }
}

createType("BirdEncounter").with {
    supertypes("BaseRandomEncounter");

    addAction("initializeEncounter") { blocks ->
        if( log.isTraceEnabled() ) {
            log.trace("initializeEncounter:" + self);
        }
        long start = System.nanoTime();
        checkEncounter.call(self);
        if( self[SpawnPosition] == null ) {
            log.error("Encounter does not have a spawn position:" + self, new Throwable("very visible"));
            return;
        }
        long time1 = System.nanoTime();

        // Look for leaves, spawn birds based on frequency + dice roll
        double max = 3;
        int count = (int)(1 + Math.random() * (max - 1));

        // We don't reinitialize an existing encounter so this is wasted here
        //if( log.isTraceEnabled() ) {
        //    log.trace("check for existing spawns for:" + self);
        //}
        //self.spawns.each {
        //    if( log.isTraceEnabled() ) {
        //        log.trace("  Found existing spawns:" + it);
        //    }
        //    // If we already have a spawn then take it away from the 'to spawn' count
        //    count--;
        //}

        int birdType = (int)(Math.random() * 2);

        def spawns = [:];

        // Spawn in the appropriate number of butterflies
        for( int i = 0; i < count; i++ ) {
            int index = (int)Math.random() * blocks.size();
            Vec3d loc = blocks.remove(index).toVec3d();

            double roll = Math.random();
            roll = roll * roll * roll;
            int variant = (int)(roll * 4);
            boolean randomColor = Math.random() < 0.2;
            if( birdType == 1 ) {
                variant = 0;
                birdType = 3;
            } else if( variant < 3 ) {
                variant = 0;
            }
            def entity = type("Bird").newInstance(
                                    new CreatedBy(self),
                                    // OwnedBy is better because if a player picks up the
                                    // butterfly then we can change the owned by and it won't
                                    // be managed by the encounter anymore.
                                    new OwnedBy(self),
                                    new SkinVariation(birdType * 4 + variant),
                                    new TemporaryObject()
                                );
            if( log.isTraceEnabled() ) {
                log.trace("Created bird:" + entity);
            }

            //if( variant == 3 ) {
            if( randomColor ) {
                float r = (float)Math.random();
                float g = (float)Math.random();
                float b = (float)Math.random();
                def color = new ColorRGBA(r, g, b, 1);
                if( log.isTraceEnabled() ) {
                    log.trace("Setting random skin color to:" + color);
                }
                entity << new SkinColor(color);
            }

            spawns.put(entity, loc);
        }
        long time2 = System.nanoTime();

        // We'll spool the actual placement but not the creation.
        // If we spool the creation then if we get deactivated  before they have
        // all spawned them some might get orphaned and just hang around the world
        // forever.
        spawns.entrySet().each {
            def entity = it.key;
            def loc = it.value;
            spool {
                // See if it's even still real
                if( entity[AgentType] ) {
                    entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                    if( log.isTraceEnabled() ) {
                        log.trace("  placed spawn:" + entity + " at:" + entity[SpawnPosition]?.location);
                    }
                } else {
                    log.warn("Entity is gone by the time of placement:" + entity + " loc:" + loc);
                }
            }
        }

        long end = System.nanoTime();
        log.info(String.format("" + self + " init in: %.03f ms, check: %.03f ms, create: %.03f ms, place: %.03f ms",
                            (end - start)/1000000.0,
                            (time1 - start)/1000000.0,
                            (time2 - time1)/1000000.0,
                            (end - time2)/1000000.0
                        ));
        if( log.isTraceEnabled() ) {
            findSpawns(self).each {
                log.trace(" created:" + it);
            }
        }
    }
}


// Do some cleanup of old leaked triggers that don't have all of their data...
// but only the first time we have to.
def triggerCleanupLevelKey = "trigger.cleanup.level";
def triggerCleanupLevel = worldManager.worldEntity.vars.loadInt(triggerCleanupLevelKey, -1);
def latestCleanupLevel = 1;
log.info("trigger.cleanup.level:" + triggerCleanupLevel);

if( triggerCleanupLevel == -1 ) {
    log.info("Need to cleanup old leaked triggers and encounters.");

    // Prior to this moment in time, all encounter triggers and encounters were temporary.
    // As of cleanup level "none", any trigger is leaked from bad processing... and
    // thus so is any linked encounter.
    def triggerEncounters = [:];
    log.info("Checking for leaked triggers...");
    findEntities(null, EncounterTrigger.class).each { trigger ->
        def et = trigger[EncounterTrigger];
        triggerEncounters.put(trigger, et.encounterId);
        log.info("trigger:" + trigger);
        int count1 = logComponents(log, trigger, "  ", 80);
        log.info("  encounter:" + et.encounterId);
        int count2 = logComponents(log, et.encounterId, "    ", 80);
        if( count1 == 1 && count2 == 0 ) {
            // This really is just dead space... the entity only has a trigger
            // component and the trigger points to a non-existent entity.
            log.info("  **removing trigger:" + trigger);
            entityData.removeEntity(trigger);
        }
    }
    log.info("Leaked trigger count:" + triggerEncounters.size());

    // Find all of the existing random encounters.  At the time of cleanup,
    // there are only two so its easy... and they should ALWAYS have been cleaned up
    // if they were legitimate.  So the only encounters we will see here are either
    // legacy leaks from before the issues were fixed or the last run failed to shutdown
    // properly.
    log.info("Checking for leaked encounters...");
    def randomEncounters = [] as Set;
    findEntities(ObjectTypeInfo.typeFilter("ButterflyEncounter"), ObjectTypeInfo.class).each {
        randomEncounters.add(it);
    }
    findEntities(ObjectTypeInfo.typeFilter("BirdEncounter"), ObjectTypeInfo.class).each {
        randomEncounters.add(it);
    }

    def leakedSpawns = [] as Set;
    randomEncounters.each { it ->
        log.info("encounter:" + it + "  hasTrigger:" + triggerEncounters.values().contains(it));
        logComponents(log, it, "  ", 80);
        findSpawns(it).each { spawn ->
            leakedSpawns.add(spawn);
            log.info("  spawn:" + spawn);
            logComponents(log, spawn, "    ", 80);
            log.info("    **removing spawn:" + spawn);
        }
        log.info("  **removing encounter:" + it);
        entityData.removeEntity(it);
    }
    log.info("Leaked spawn count:" + leakedSpawns.size());

    // Check for orphaned MOBs
    log.info("Checking for orphans...");
    def orphans = [] as Set;
    findEntities(null, AgentType.class, ObjectTypeInfo.class, SpawnPosition.class).each {
        String type = it[ObjectTypeInfo].getTypeName();
        if( type == "Animal" ) {
            type = it[ObjectName].getName();
        }
        def ownedBy = it[OwnedBy]?.owner;
        def createdBy = it[CreatedBy]?.creatorId;
        def isOrphan = ownedBy == createdBy && (ownedBy == null || !hasComponents(ownedBy));

        if( "Butterfly" == type ) {
            log.info("Butterfly:" + it + " created by:" + createdBy + " owned by:" + ownedBy);
            if( isOrphan ) {
                orphans.add(it);
            } else {
                log.info(" -Not an orphan.");
                if( ownedBy ) {
                    log.info("  ownedBy:" + ownedBy);
                    logComponents(log, ownedBy, "    ", 80);
                }
                if( createdBy && createdBy != ownedBy ) {
                    log.info("  createdBy:" + createdBy);
                    logComponents(log, createdBy, "    ", 80);
                }
            }
        }
        if( "Bird" == type ) {
            log.info("Bird:" + it + "  created by:" + createdBy + " owned by:" + ownedBy);
            if( isOrphan ) {
                orphans.add(it);
            } else {
                log.info(" -Not an orphan.");
                if( ownedBy ) {
                    log.info("  ownedBy:" + ownedBy);
                    logComponents(log, ownedBy, "    ", 80);
                }
                if( createdBy && createdBy != ownedBy ) {
                    log.info("  createdBy:" + createdBy);
                    logComponents(log, createdBy, "    ", 80);
                }
            }
        }
    }
    // So we can do script things if we want
    orphans.each { orphan ->
        // Unfortunately can't run actions or custom getters during this phase
        // of startup.
        String name = "Unknown";
        if( orphan[Name] ) {
            name = orphan[Name].name;
        } else if( orphan[ObjectName] ) {
            name = orphan[ObjectName].name;
        }
        log.info("Orphan:" + orphan + "  name:" + name);
        logComponents(log, orphan, "  ", 80);
        log.info("  **removing orphan:" + orphan);
        entityData.removeEntity(orphan);
    }

    // Now mark this cleanup as done
    worldManager.worldEntity.vars.storeInt(triggerCleanupLevelKey, latestCleanupLevel);
}

// Try to remove all of the butterflies to clean things up if something broke
if( true ) {
    boolean cleanup = false;

    // We will always try to cleanup the random encounters from before we tagged
    // them with TemporaryObject components.
    def oldLeakedEncounters = [] as Set;

    def leakedEncounterSpawnOwners = [] as Set;
    int leakedMobCount = 0;

    // Try to find all of them... not just in this zone
    log.info("All entities...");
    findEntities(null, AgentType.class, ObjectTypeInfo.class, SpawnPosition.class).each {
        log.info("Mobs:" + it + "  name:" + it[ObjectName].toString(entityData) + " type:" + it[ObjectTypeInfo].getTypeName());
        String type = it[ObjectTypeInfo].getTypeName();
        if( type == "Animal" ) {
            type = it[ObjectName].getName();
        }
        log.info("  type:" + type);
        log.info("  zone:" + it[SpawnPosition].binId);
        if( "Butterfly" == type ) {
            log.info("  ownedBy:" + it[OwnedBy]);
            leakedMobCount++;
            if( it[OwnedBy] ) {
                leakedEncounterSpawnOwners.add(it[OwnedBy].owner);
                logComponents(log, it[OwnedBy].owner, "    ", 80);
            } else {
                log.warn("  has no owner");
            }
            log.info("  bufferly agent type:" + it[AgentType]);
            log.info("  body pos:" + it[BodyPosition]);
            if( cleanup ) {
                log.info("  ...removing");
                removeEntity(it);
            }
        }
        if( "Bird" == type ) {
            log.info("  ownedBy:" + it[OwnedBy]);
            leakedMobCount++;
            if( it[OwnedBy] ) {
                leakedEncounterSpawnOwners.add(it[OwnedBy].owner);
                logComponents(log, it[OwnedBy].owner, "    ", 80);
            } else {
                log.warn("  has no owner");
            }
            log.info("  bird agent type:" + it[AgentType]);
            log.info("  body pos:" + it[BodyPosition]);
            if( cleanup ) {
                log.info("  ...removing");
                removeEntity(it);
            }
        }
    }

    def leakedEncounters = [] as Set;

    log.info("All buttefly encounters...");
    findEntities(ObjectTypeInfo.typeFilter("ButterflyEncounter"), ObjectTypeInfo.class).each {
        leakedEncounters.add(it);
        if( it[TemporaryObject] == null ) {
            oldLeakedEncounters.add(it);
        }
        if( cleanup ) {
            log.info("Removing encounter:" + it);
            removeEntity(it);
        } else {
            log.info("Found existing butterfly encounter:" + it);
            entityData.getComponentHandlers().keySet().each { componentType ->
                def val = it[componentType]
                if( val ) {
                    String text = String.valueOf(val);
                    int maxLength = 120;
                    if( text.length() > maxLength ) {
                        text = text.substring(0, maxLength - 3) + "...";
                    }
                    log.info("  " + componentType.simpleName + " = " + text);
                }
            }
        }
    }

    log.info("All bird encounters...");
    findEntities(ObjectTypeInfo.typeFilter("BirdEncounter"), ObjectTypeInfo.class).each {
        leakedEncounters.add(it);
        if( it[TemporaryObject] == null ) {
            oldLeakedEncounters.add(it);
        }
        if( cleanup ) {
            log.info("Removing encounter:" + it);
            removeEntity(it);
        } else {
            log.info("Found existing bird encounter:" + it);
            entityData.getComponentHandlers().keySet().each { componentType ->
                def val = it[componentType]
                if( val ) {
                    String text = String.valueOf(val);
                    int maxLength = 120;
                    if( text.length() > maxLength ) {
                        text = text.substring(0, maxLength - 3) + "...";
                    }
                    log.info("  " + componentType.simpleName + " = " + text);
                }
            }
        }
    }

    def leakedTriggers = [] as Set;
    def leakedTriggerEncounters = [:];
    if( true ) {
        // Clean out all of the lingering encounter triggers
        findEntities(null, EncounterTrigger.class).each { trigger ->
            leakedTriggers.add(trigger);
            leakedTriggerEncounters.put(trigger, trigger[EncounterTrigger].encounterId);
            if( cleanup ) {
                log.info("Removing trigger:" + trigger + " tigger:" + trigger[EncounterTrigger]);
                removeEntity(trigger);
            } else {
                String type = trigger[ObjectTypeInfo]?.getTypeName(entityData);
                log.info("Found trigger:" + trigger + " tigger:" + trigger[EncounterTrigger] + " type:" + type);
                entityData.getComponentHandlers().keySet().each { componentType ->
                    def val = trigger[componentType]
                    if( val ) {
                        String text = String.valueOf(val);
                        int maxLength = 120;
                        if( text.length() > maxLength ) {
                            text = text.substring(0, maxLength - 3) + "...";
                        }
                        log.info("  " + componentType.simpleName + " = " + text);
                    }
                }
                findEntities(OwnedBy.filter(trigger),
                             OwnedBy.class,
                             ShapeInfo.class,
                             AgentType.class).each {
                    log.info("  ----spawn:" + it);
                }
            }
        }
    }

    log.info("Leaked mobs:" + leakedMobCount + " leakedEncounters:" + leakedEncounters.size());

    def check
    check = new HashSet(leakedEncounterSpawnOwners);
    check.removeAll(leakedEncounters);
    log.info("Unmatched encounter spawn owners:" + check);
    def triggerCheck = leakedTriggerEncounters.clone();
    triggerCheck.values().removeAll(leakedEncounters);
    log.info("Triggers without encounters:" + triggerCheck.keySet());
    check = new HashSet(leakedEncounters);
    check.removeAll(leakedTriggerEncounters.values())
    log.info("Encounters without triggers:" + check);
}

boolean disableEncounters = false;
boolean enableButterflies = true;
boolean enableBirds = true;

mobEncounterSystem = system(MobEncounterSystem.class);

system(AgentActivationSystem.class).addActivationZoneListener(new ActivationZoneListener() {
    // Called from the agents thread to notify about zone activation.
    // Beware of the threading... and asynchronous state.
    public void activateZone( long zoneId ) {
        log.info("" + zoneId + " activateZone()");
        if( disableEncounters ) {
            return;
        }

        onBackground {
            // See if we should try to create encounters...
            // Note: as it stands, this is blocking other listeners and we may want to
            // move it to a background worker pool.
            def colId = new ColumnId(zoneId);
            if( enableButterflies ) {
                def flowerTypes = worldStats.getBlockSet("flowers");
                List<Vec3i> blocks = worldStats.getSurfaceBlocks(colId, flowerTypes);
                if( blocks.size() > 1 ) {
                    // Run this on some future frame
                    spoolAsWorld {
                        log.info("" + zoneId + " Checking butterflies zone");
                        long start = System.nanoTime();
                        if( !mobEncounterSystem.isActiveZone(zoneId) ) {
                            log.info("" + zoneId + " is no longer active");
                            return;
                        }
                        long time1 = System.nanoTime();
                        // Looking up the existing encounters is expensive and it's the rarer case
                        // for the most part.  If we really want to, we'd be better off asking
                        // MobEncounterSystem... because if it still has one for this zone + type
                        // then it exists.
                        //def objectTypeInfo = ObjectTypeInfo.create("ButterflyEncounter");
                        //def existing = findRandomEncounters(zoneId, objectTypeInfo);
                        //long time2 = System.nanoTime();
                        //log.info("existing:" + existing);
                        //if( existing ) {
                        //    // Should be safe... and even if it's not, the worst case is that
                        //    // we miss some random critters somewhere.
                        //    log.info("Reusing existing butterfly encounter for:" + zoneId);
                        //    return;
                        //}

                        def butterflies = createEntity(
                            ObjectTypeInfo.create("ButterflyEncounter"),
                            new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd()),
                            new TemporaryObject()
                        );
                        long time3 = System.nanoTime();
                        log.info("" + zoneId + " created butterflies:" + butterflies + " at:" + butterflies[SpawnPosition]);
                        // Random encounters are their own triggers
                        butterflies << new EncounterTrigger(butterflies, zoneId, EncounterTrigger.TYPE_TEMPORARY);
                        long time4 = System.nanoTime();
                        butterflies.run("initializeEncounter", blocks);
                        long end = System.nanoTime();
                        log.info(String.format("butterflies total: %.03f ms, zone: %.03f ms, create: %.03f ms, trigger: %.03f ms, init: %.03f ms",
                                        (end - start)/1000000.0,
                                        (time1 - start)/1000000.0,
                                        (time3 - time1)/1000000.0,
                                        (time4 - time3)/1000000.0,
                                        (end - time4)/1000000.0
                                ));
                    }
                }
            }

            if( enableBirds ) {
                def leafTypes = worldStats.getBlockSet("leaves");
                List<Vec3i> blocks = worldStats.getSurfaceBlocks(colId, leafTypes);
                if( blocks.size() > 10 ) {
                    // Run this on some future frame
                    spoolAsWorld {
                        log.info("" + zoneId + " Checking birds zone");
                        long start = System.nanoTime();
                        if( !mobEncounterSystem.isActiveZone(zoneId) ) {
                            log.info("" + zoneId + " is no longer active");
                            return;
                        }
                        long time1 = System.nanoTime();
                        // Looking up the existing encounters is expensive and it's the rarer case
                        // for the most part.  If we really want to, we'd be better off asking
                        // MobEncounterSystem... because if it still has one for this zone + type
                        // then it exists.
                        //def objectTypeInfo = ObjectTypeInfo.create("BirdEncounter");
                        //def existing = findRandomEncounters(zoneId, objectTypeInfo);
                        //long time2 = System.nanoTime();
                        //log.info("existing:" + existing);
                        //if( existing ) {
                        //    // Should be safe... and even if it's not, the worst case is that
                        //    // we miss some random critters somewhere.
                        //    log.info("Reusing existing bird encounter for:" + zoneId);
                        //    return;
                        //}

                        def birds = createEntity(
                            ObjectTypeInfo.create("BirdEncounter"),
                            new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd()),
                            new TemporaryObject()
                        );
                        long time3 = System.nanoTime();
                        log.info("" + zoneId + " created birds:" + birds + " at:" + birds[SpawnPosition]);
                        // Random encounters are their own triggers
                        birds << new EncounterTrigger(birds, zoneId, EncounterTrigger.TYPE_TEMPORARY);
                        long time4 = System.nanoTime();
                        birds.run("initializeEncounter", blocks);
                        long end = System.nanoTime();
                        log.info(String.format("birds total: %.03f ms, zone: %.03f ms, create: %.03f ms, trigger: %.03f ms, init: %.03f ms",
                                        (end - start)/1000000.0,
                                        (time1 - start)/1000000.0,
                                        (time3 - time1)/1000000.0,
                                        (time4 - time3)/1000000.0,
                                        (end - time4)/1000000.0
                                ));
                    }
                }
            }
        }
    }

    public void deactivateZone( long zoneId ) {
        log.info("" + zoneId + ":deactivateZone()");
        if( disableEncounters ) {
            return;
        }
    }
});

//system(AgentActivationSystem.class).addActivationZoneListener(new ActivationZoneListener() {
//    public void activateZone( long zoneId ) {
//        log.info("" + zoneId + " activateZone()");
//
//        if( disableEncounters ) {
//            return;
//        }
//
//        def triggers = findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class);
//        log.info("" + zoneId + " existing triggers:" + triggers);
//        triggers.each { trigger ->
//            log.info("" + zoneId + " found existing:" + trigger + "  trigger:" + trigger[EncounterTrigger]);
//        }
//        if( !triggers ) {
//
//            // Look at the world data to see if there are any flowers.
//            // no flowers -> don't create it
//            def colId = new ColumnId(zoneId);
//            if( enableButterflies ) {
//                def flowerTypes = worldStats.getBlockSet("flowers");
//                //List<Vec3i> flowers = worldStats.getSurfaceBlocks(colId, flowerTypes);
//                boolean found = worldStats.hasSurfaceBlocks(colId, flowerTypes, 1);
//
//                // Figure out how many max we would want to spawn based on season, climate, etc..
//                // ...but for now spawn the encounter if there are any flowers at all.
//                if( found ) {
//                    def butterflies = createEntity(
//                        ObjectTypeInfo.create("ButterflyEncounter"),
//                        new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd())
//                    );
//                    log.info("" + zoneId + " created butterflies:" + butterflies + " at:" + butterflies[SpawnPosition]);
//                    def trigger = createEntity(
//                        new EncounterTrigger(butterflies, zoneId)
//                    );
//                }
//            }
//
//            if( enableBirds ) {
//                def leafTypes = worldStats.getBlockSet("leaves");
//                //List<Vec3i> leaves = worldStats.getSurfaceBlocks(colId, leafTypes);
//                boolean found = worldStats.hasSurfaceBlocks(colId, leafTypes, 10);
//                if( found ) {
//                    def birds = createEntity(
//                        ObjectTypeInfo.create("BirdEncounter"),
//                        new SpawnPosition(colId.getWorld(null).toVec3d(), new Quatd())
//                    );
//                    log.info("" + zoneId + " created birds:" + birds + " at:" + birds[SpawnPosition]);
//                    def trigger = createEntity(
//                        new EncounterTrigger(birds, zoneId)
//                    );
//                }
//            }
//        //} else {
//        //    log.info("Found existing:" + trigger);
//        }
//
//        log.info("kilroy butterflies:" + findEncounters(zoneId, ObjectTypeInfo.create("ButterflyEncounter")));
//        log.info("kilroy birds:" + findEncounters(zoneId, ObjectTypeInfo.create("BirdEncounter")));
//    }
//
//    public void deactivateZone( long zoneId ) {
//        log.info("" + zoneId + ":deactivateZone()");
//        findEntities(EncounterTrigger.zoneFilter(zoneId), EncounterTrigger.class).each { trigger ->
//            // Note: removing the trigger is always necessary... this is not necessarily
//            // removing the encounter itself.  That's up to what the encounter does when
//            // deactivated.  ie: trigger != encounter
//            log.info("" + zoneId + " removing:" + trigger + " tigger:" + trigger[EncounterTrigger]);
//            removeEntity(trigger);
//        }
//    }
//});
