

def triggerName = { EntityId trigger ->

log.info("triggerName(" + trigger + ")");

    def results = [];

    results.add(trigger.name);

log.info("  name:" + trigger.name);

    ObjectTypeInfo oti = trigger[ObjectTypeInfo];
    if( oti ) {
        results.add("type:" + oti.typeName);
log.info("  type:" + oti.typeName);
    }

    StructureInfo si = trigger[StructureInfo];
    if( si ) {
        results.add("struct:" + si.getFactory(entityData));
log.info("struct:" + si.getFactory(entityData));
    }

    ChildOf parent = trigger[ChildOf];
    if( parent ) {
        results.add(" childOf:" + parent.parent.name);
log.info(" ownedBy:" + parent.parent.name);
    }

    return results.join(",");
}

def journalPrefix = GameConstants.JOURNAL_PREFIX;
createType("PlayerJournal").with {
    addAction("getActiveQuests") { int page, int size ->
        //journalGetActiveQuests(session, self, page, size);
        log.info("getActiveQuests(" + page + ", " + size + ")");
        session.showPrompt(self, PromptType.Dialog, journalPrefix + "Active Quests");

        // Collect the active quests and send the options
        def options = [];

        def filter = Filters.fieldEquals(Quest.class, "quester", self);
        findEntities(filter, Quest.class).each { quest ->
            if( quest[Quest].status >= 0 ) {
                options.add(new Option(self, quest.name, "getQuestInfo", quest));
            }
        }

        session.showOptions(self, options);
    }

    addAction("getQuestInfo") { EntityId questEntity ->
        //journalGetQuestInfo(session, self, questEntity);
        log.info("getQuestInfo(" + questEntity + ")");
        Quest quest = questEntity[Quest];
        def questGiver = quest?.questGiver;
        def relativeLoc = questEntity[RelativeLocation]
        if( !relativeLoc ) {
            // This shouldn't be necessary anymore so we'll log a warning
            log.warn("Copying parent relative location to child:" + questEntity);
            relativeLoc = quest.parentQuest[RelativeLocation]
            questEntity << relativeLoc;
        }

        def sourceLoc = relativeLoc.source[SpawnPosition]?.location;
        def targetLoc = relativeLoc.target[SpawnPosition]?.location;

        def targetPoi = relativeLoc.target[StructureInfo];
        def factoryName = targetPoi.getFactory(entityData);
        def poiDesc = worldManager.textDb.getText("poi/test-poi-desciptions.txt#${factoryName}", null);
        poiDesc = poiDesc.readLines()[0];

        def angle = Math.atan2(targetLoc.z - sourceLoc.z, targetLoc.x - sourceLoc.x);
        def distance = targetLoc.distance(sourceLoc);
        angle = Math.toDegrees(angle);
        if( angle < 0 ) {
            angle += 360;
        }
        def angleDelta = 360 / 16; // n, nne, ne, ene, etc.
        def angleIndex = (int)((angle + (angleDelta * 0.5)) / angleDelta);
        def dir = directionNames[angleIndex];

        def firstName;
        log.info("relationship:" + resolveEntityId(session).relationship(questGiver));
        if( resolveEntityId(session).relationship(questGiver)?.level > 0 ) {
            firstName = questGiver.characterName;
        } else {
            firstName = questGiver[ObjectName]?.name;
            if( firstName ) {
                firstName = "A " + firstName.toLowerCase();
            } else {
                firstName = "Someone";
            }
        }

        def description = /## ${questEntity.name}
            ${firstName} has lost
            ${questGiver.pronouns.possessive} ${questEntity.questItem[ObjectName]?.name?.toLowerCase()}
            and has asked us to find it.

            ${questGiver.pronouns.personal.capitalize()} has indicated
            that it is ${findDistanceName(distance)} ${dir} of ${relativeLoc.source.name}
            in or near ${poiDesc}.
        /

        if( quest.status == 1 ) {
            description += "\n\nYou have found this item."
        } else if( quest.status == -1 ) {
            description += "\n\nYou have delivered the item."
        } else {
            description += "\n\n${questGiver.pronouns.personal.capitalize()} has sketched the rough area on our map."
        }

        session.showPrompt(self, PromptType.Dialog, journalPrefix + "Quest:" + description);
        def options = [];
        //options.add(new Option(self, "Who is ${questGiver?.name}?", "whoIs", questGiver));
        options.add(new Option(self, "cheat", questEntity));
        //options.add(new Option(self, "abandonQuest", questEntity));
        session.showOptions(self, options);
    }

    addAction("whoIs") { EntityId person ->
        //journalWhoIs(session, self, person);
        session.showPrompt(self, PromptType.Dialog, journalPrefix + "Description:I don't know, who is " + person.name + "?");
    }

    addAction("abandonQuest") { EntityId quest ->
        //journalAbandonQuest(session, self, quest);
        log.info("abandonQuest(" + quest + ")");
    }

    addAction("cheat") { EntityId questEntity ->
        //journalCheat(session, self, quest);
        log.info("cheat(" + questEntity + ")");
        // Cheat means different things depending on the quest
        Quest quest = questEntity[Quest];
        def item = questEntity.questItem;
        if( !item ) {
            session.echo("No quest item found for this quest.  What's up?!?");
            return;
        }
        if( quest.status == 0 ) {
            // Take us to the object
            physics.teleport(resolveEntityId(session), item[SpawnPosition].location, new Quatd());
        } else if( quest.status == 1 ) {
            // Take us to the NPC
            physics.teleport(resolveEntityId(session), quest.questGiver[SpawnPosition].location, new Quatd());
        }
    }
}

import mythruna.es.debug.*;


createType("PlayerCharacter").with {

    // For when the player is bare-handed, "we" are the tool
    supertypes("ObjectManipulator",
                // Temporary until there is a real journal object
                "PlayerJournal");

    addAction("moveToTrash") { ->
        self << ObjectTypeInfo.create("DeletedCharacter");
    }

    addAction("setSkinColor") { SkinColor skinColor ->
        log.info("setSkinColor:" + skinColor);
        self << skinColor;
    }

    addAction("setHairColor") { HairColor hairColor ->
        log.info("setHairColor:" + hairColor);
        self << hairColor;
    }

    addAction("deleteBlueprint") { EntityId blueprint ->
        log.info("Delete blueprint:" + blueprint);
        def bpInfo = blueprint[BlueprintInfo];
        if( bpInfo ) {
            log.info("bpInfo:" + bpInfo);
            // We should be able to "delete" it by setting the parent ID to null and
            // then it just won't appear anyway.  We'll still retain the blueprint
            // info for any object that might exist in the world.
            blueprint << bpInfo.changeParent(null);
            echo("Deleted blueprint:" + bpInfo.name);
            return;
        }
        def asmInfo = blueprint[AssemblyBlueprintInfo];
        if( asmInfo ) {
            log.info("asmInfo:" + asmInfo);
            blueprint << asmInfo.changeParent(null);
            echo("Deleted design:" + asmInfo.name);
            return;
        }
        log.warn("Error deleting unknown blueprint type for:" + blueprint);
        echo("Error deleting unknown blueprint type for:" + blueprint);
    }

    addAction("onApproachingTrigger") { EntityId trigger ->
        if( log.isTraceEnabled() ) {
            log.trace("Approaching " + triggerName(trigger));
        }
        //echo("player: Approaching " + triggerName(trigger));
        if( !trigger.runIfExists("onTriggerApproached", self) ) {
            //echo("Approaching " + triggerName(trigger));
        }
    }

    addAction("onEnteredTrigger") { EntityId trigger ->
        if( log.isTraceEnabled() ) {
            log.trace("Entered " + triggerName(trigger));
        }
        //echo("player: Entered " + triggerName(trigger));
        if( !trigger.runIfExists("onTriggerEntered", self) ) {
            //echo("Entered " + triggerName(trigger));
        }
    }

    addAction("onExitedTrigger") { EntityId trigger ->
        if( log.isTraceEnabled() ) {
            log.trace("Exited " + triggerName(trigger));
        }
        //echo("player: Exited " + triggerName(trigger));
        if( !trigger.runIfExists("onTriggerExited", self) ) {
            //echo("Exited " + triggerName(trigger));
        }
    }

    addAction("onLeavingTrigger") { EntityId trigger ->
        if( log.isTraceEnabled() ) {
            log.trace("Leaving " + triggerName(trigger));
        }
        //echo("player: Leaving " + triggerName(trigger));
        if( !trigger.runIfExists("onTriggerLeft", self) ) {
            //echo("Leaving " + triggerName(trigger));
        }
    }

    addAction("detailTest") { blockHit, objectHit ->
        log.info("objectHit:" + objectHit);
        if( objectHit || blockHit ) {
            def arrowEntity = env.getVar("debugArrowHit", null);
            if( arrowEntity == null ) {
                arrowEntity = createEntity();
                env.setVar("debugArrowHit", arrowEntity);
            }
            if( objectHit ) {
                arrowEntity << new SpawnPosition(objectHit.location, new Quatd());
                arrowEntity << new DebugShape(DebugShape.TYPE_ARROW, objectHit.normal, ColorRGBA.Green, 0);
            } else if( blockHit ) {
                arrowEntity << new SpawnPosition(blockHit.location, new Quatd());
                arrowEntity << new DebugShape(DebugShape.TYPE_ARROW, blockHit.normal, ColorRGBA.Blue, 0);
            }
            arrowEntity << Decay.seconds(60);
        }
    }

    addAction("showMarkDialog") { ->
        session.showPrompt(self, PromptType.Dialog,
                            "## Warp Locations\n" +
                            "Select a warp target:"
                        );

        run("showOptions", 0, 10);
    }

    addAction("showOptions") { page, pageSize ->

        def sorted = [:] as TreeMap;
        // Get our player-specific marks
        getMarks(self).each { mark ->
            def name = mark.name;
            sorted.put(name.capitalize(), mark);
        }
        // Get the world-level marks
        getMarks(worldManager.worldEntity).each { mark ->
            def name = mark.name;
            sorted.put(name.capitalize(), mark);
        }

        def opts = [];
        int index = 0;
        int first = page * pageSize;
        int last = (page + 1) * pageSize;
        sorted.entrySet().each { entry ->
            if( index >= first && index < last ) {
                def mark = entry.value;
                opts.add(new Option(self, "" + (index + 1) + ") " + mark.name, "warpOption", mark[SpawnPosition]));
            }
            index++;
        }
        if( page > 0 ) {
            opts.add(new Option(self, "<previous>", "showOptions", page-1, pageSize));
        }
        if( last < sorted.size() ) {
            opts.add(new Option(self, "<next>", "showOptions", page+1, pageSize));
        }

        session.showOptions(self, opts, new Option(self, "<cancel>", "cancel"));
    }

    addAction("warpOption") { pos ->
        // Close the dialog
        session.showPrompt(self, PromptType.Dialog, null);
        echo("Warping to:" + pos.location.floor());
        run("warpTo", pos);
    }

    addAction("warpTo") { pos ->
        log.info("Warping to:" + pos.location.floor());
        def phys = system(PhysicsSpace);
        phys.teleport(self, pos.location, pos.orientation);
        self << new VisionEffect(VisionEffect.FX_PORTAL_BLINDNESS);
    }

    addAction("cancel") { ->
    }

    addAction("clearPortalBlindness") { ->
        // TODO: check to see if portal blindness is the current effect
        self.remove(VisionEffect.class);
    }
}


