
findTempleGate = { temple ->
    def gate = null;
    findEntities(ChildOf.filter(temple), ChildOf.class).each { child ->
        // If it's a moon gate then it's our result
        if( child.type.name == "MoonGate" ) {
            gate = child;
        }
    }
    return gate;
}

getGatePortals = { gate ->
    def temple = gate[ChildOf]?.parent;
    return findEntities(PortalLink.outFilter(temple), PortalLink.class);
}

getLinkedPortal = { gate1, gate2 ->
    def temple1 = gate1[ChildOf]?.parent;
    def temple2 = gate2[ChildOf]?.parent;
    return findEntity(PortalLink.filter(temple1, temple2), PortalLink.class);
}

// We find the POIs instead of just the inserts so that we
// can do some additional connections as needed.
findNearbySpiritTemplePois = { temple, double maxRadius, boolean generateIfNeeded ->
    def loc = temple.location;
    def temples = [] as Set;

    findPointsOfInterest2(loc, (maxRadius * 1024), generateIfNeeded).each { poi ->
        if( "Spirit Henge" == poi.type ) {
            def skip = poi.children.find { it.id == temple.id };
            if( skip == null ) {
                temples.add(poi);
            }
        }
    }
    return temples;
}

//// Temple can be an EntityId or the InsertInfo for a temple
//findNearbySpiritTemples = { temple, double maxRadius, generateIfNeeded ->
//    def loc = temple.location;
//    def candidates = [];
//    findPointsOfInterest2(loc, (maxRadius * 1024), generateIfNeeded).each { poi ->
//        candidates.add(poi);
//    }
//
//    def moonGates = [] as Set;
//    candidates.each { poi ->
//        if( "Spirit Henge" == poi.type ) {
//            poi.children.each { insert ->
//                if( insert.id == temple.id ) {
//                    return;
//                }
//                moonGates.add(insert);
//            }
//        }
//    }
//    return moonGates;
//}

class PortalDirection {
    String name;
    Vec3d offset;
    Quatd orientation;
}

def orientations = [
    new Quatd().fromAngles(0, -Math.PI * 0.5, 0),
    new Quatd().fromAngles(0, -Math.PI * 0.75, 0),
    new Quatd().fromAngles(0, -Math.PI * 1.0, 0),
    new Quatd().fromAngles(0, -Math.PI * 1.25, 0),
    new Quatd().fromAngles(0, -Math.PI * 1.5, 0),
    new Quatd().fromAngles(0, -Math.PI * 1.75, 0),
    new Quatd(),
    new Quatd().fromAngles(0, -Math.PI * 0.25, 0)
]

def angleOffset = 1.2;
def straightOffset = 1.6;

portalDirections = [
    new PortalDirection(name:"East",
                        offset:new Vec3d(straightOffset, 0, 0),
                        orientation:orientations[0]),
    new PortalDirection(name:"Southeast",
                        offset:new Vec3d(angleOffset, 0, angleOffset),
                        orientation:orientations[1]),
    new PortalDirection(name:"South",
                        offset:new Vec3d(0, 0, straightOffset),
                        orientation:orientations[2]),
    new PortalDirection(name:"Southwest",
                        offset:new Vec3d(-angleOffset, 0, angleOffset),
                        orientation:orientations[3]),
    new PortalDirection(name:"West",
                        offset:new Vec3d(-straightOffset, 0, 0),
                        orientation:orientations[4]),
    new PortalDirection(name:"Northwest",
                        offset:new Vec3d(-angleOffset, 0, -angleOffset),
                        orientation:orientations[5]),
    new PortalDirection(name:"North",
                        offset:new Vec3d(0, 0, -straightOffset),
                        orientation:orientations[6]),
    new PortalDirection(name:"Northeast",
                        offset:new Vec3d(angleOffset, 0, -angleOffset),
                        orientation:orientations[7])
];

import mythruna.world.town.InsertInfo;

class PortalCandidate {
    EntityId source;
    EntityId target;
    InsertInfo targetInsert;
    boolean enabled;
    double length;
    PortalDirection dir;
}

/**
 *  Retrieves an array of PortalCandidate objects in the 8 cardinal
 *  directions.  A null represents no portal  in that direction.  All of the accumulated
 *  portal candidates will be added to the 'links' list.  If 'generateIfNeeded' is
 *  true then the sedectile generation will happen necessary to perform the radius
 *  query.  Also, if "generateIfNeeded' is true then all of the InsertInfo objects will
 *  be connected to their parent POI with a ChildOf component.
 *  'temple' can be an EntityId or an InsertInfo.
 */
getPortalCandidates = { def temple, List links, boolean generateIfNeeded ->
    PortalCandidate[] dirs = new PortalCandidate[8];

    def templeEntity;
    if( temple instanceof EntityId ) {
        templeEntity = temple;
    } else {
        templeEntity = new EntityId(temple.id);
    }

    def startLoc = temple.location;

    double angleDelta = (Math.PI * 2) / 8;

    //def maxRadius = 8;
    def maxRadius = 12;

    //findNearbySpiritTemples(temple, maxRadius, generateIfNeeded).each { n ->
    findNearbySpiritTemplePois(temple, maxRadius, generateIfNeeded).each { poi ->
        def poiEntity = new EntityId(poi.featureId.id);
        poi.children.each { n ->
            def loc = n.location.toVec3d();
            def target = new EntityId(n.id);

            // Make sure they are connected to their parent POI in case the target
            // entities have not really been generated yet.
            if( generateIfNeeded ) {
                if( target[ChildOf] == null ) {
                    log.info("Preconnecting " + target + " to parent:" + poiEntity);
                    target << new ChildOf(poiEntity);
                }
            }

            double angle = Math.atan2(loc.z - startLoc.z, loc.x - startLoc.x);
            angle += angleDelta * 0.5;
            if( angle < 0 ) {
                angle += Math.PI * 2;
            }
            int dir = (int)(angle / angleDelta);

            def candidate = new PortalCandidate(
                source:templeEntity,
                target:target,
                targetInsert:n,
                enabled:true,
                length:loc.distance(startLoc),
                dir:portalDirections[dir]
                );
            links.add(candidate);

            if( dirs[dir] == null ) {
                dirs[dir] = candidate;
            } else {
                if( dirs[dir].length > candidate.length ) {
                    // This one is better than the existing one
                    dirs[dir].enabled = false;
                    dirs[dir] = candidate;
                } else {
                    // Else this new one should be disabled as we won't use it
                    candidate.enabled = false;
                }
            }
        }
    }

    return dirs;
}

/**
 *  Creates the outbound portals for the 'temple' specified.  Created portals
 *  will be made a child of the specified 'gate'.
 */
createPortalLinks = { EntityId temple, EntityId gate, Vec3d center ->

    log.info("createPortalLinks(" + temple + ", " + gate + ", " + center + ")");

    def links = [];
    PortalCandidate[] dirs = getPortalCandidates(temple, links, true);

    // For each actual cardinal target, if it doesn't already have portal
    // links then temporarily calculate some candidates so we can make sure our outbound
    // links will still be enabled.
    for( PortalCandidate link : dirs ) {
        if( link == null ) {
            continue;
        }

        def targetPortals = findEntities(PortalLink.outFilter(link.target), PortalLink.class);
        if( targetPortals ) {
            // We'll already consider these below when we lookup backlinks
            continue;
        }

        // Calculate the hypothetical links
        def tempLinks = [];
        def targetCandidates = getPortalCandidates(link.targetInsert, tempLinks, false);

        // Find our loopback
        def loopBack = tempLinks.find { it.target == link.source && it.source == link.target };

        log.info("link:" + link + " loopback:" + loopBack);

        if( loopBack ==  null || !loopBack.enabled ) {
            log.info("Disabling link:" + link);
            link.enabled = false;
        }
    }

    def results = [];

    // Do the actual link creation/updates
    for( PortalCandidate link : links ) {
        def back = findEntity(PortalLink.filter(link.target, link.source), PortalLink.class);
        if( back != null ) {
            def backLink = back[PortalLink];
            if( !backLink.enabled ) {
                link.enabled = false;
            } else if( !link.enabled && backLink.enabled ) {
                // Then the back channel needs to be disabled, too
                back << backLink.changeEnabled(false);
            }
        }

        def dir = link.dir;
        def gateLoc = center.add(dir.offset);

        // Make an entity for the linkage
        def portal = type("Portal").newInstance(
            new ChildOf(gate),

            // Where this portal will take us (the temple)
            new PortalLink(link.source, link.target, link.enabled),

            // The actual position and orientation of the physical portal
            // when it appears.
            new SpawnPosition(gateLoc, dir.orientation)
        );
        log.info("Created:" + portal);

        results.add(portal);
    }

    return results;
}

/**
 *  Builds the portal links for the specified gate if they don't already exist.
 */
buildPortalLinks = { EntityId gate ->
    log.info("buildPortalLinks(" + gate + ")");

    // No need to add 0.5 because the moon gate is already
    // centered now.
    def center = gate.location;

    def temple = gate[ChildOf]?.parent;

    def portals = findEntities(PortalLink.outFilter(temple), PortalLink.class);
    if( !portals ) {
        log.info("creating portals");
        portals = createPortalLinks(temple, gate, center);
    } else {
        log.info("reusing portals");
    }

    return portals;
}

/**
 *  Adds the mob->gate tag with effect.  This is necessary for a mob to
 *  enter a particular portal for the gate.
 */
tagPortalToMob = { EntityId mob, EntityId gate ->

    def existing = getPortalMobTag(mob, gate);
    if( existing ) {
        log.info("tag already exists:" + existing + " for:" + mob.name);

        // But remove any decay it might have
        existing.remove(Decay);

        return;
    }

    log.info("tag:" + mob.name);

    double scale = 0.6;
    def effect = createEntity(
        ShapeInfo.create("/objects/moon-gate.j3o", scale),
        AttachedTo.create(mob, "com", new Vec3d(0, 0, 0).mult(scale), new Quatd()),
        ObjectTypeInfo.create("portalTag")
    );
    effect << EntityLink.create(mob, gate, "portalTag");

    return effect;
}

/**
 *  Returns the mob->gate tag for the specified 'mob' and 'gate'.
 */
getPortalMobTag = { EntityId mob, EntityId gate ->
    return findEntity(EntityLink.filter("portalTag", mob, gate), EntityLink);
}

/**
 *  Removes the mob->gate tag at the specified delay.
 */
untagPortalToMob = { EntityId mob, EntityId gate, double seconds ->
    log.info("untag:" + mob.name + " " + mob);
    findEntities(EntityLink.filter("portalTag", mob, gate), EntityLink).each {
        log.info("removing tag:" + it  + " from:" + mob);
        it << Decay.seconds(seconds);
    }
}

/**
 *  Debug method for resetting the portal positions to where the current
 *  algorithm thinks they should be.
 */
resetPortalPositions = { gate ->
    log.info("resetPortalPositions(" + gate + ") loc:" + gate.location);
    def center = gate.location;
    def temple = gate[ChildOf]?.parent;

    def links = [];
    def dirs = getPortalCandidates(temple, links, true);
    def portals = getGatePortals(gate);

    for( def candidate : dirs ) {
        if( candidate == null ) {
            continue;
        }

        def portal = portals.find {
            PortalLink link = it[PortalLink];
            return link.source == candidate.source && link.target == candidate.target;
        }

        log.info("dir:" + candidate + " found:" + portal);
        if( portal ) {
            def dir = candidate.dir;
            def gateLoc = center.add(dir.offset);
            portal << new SpawnPosition(gateLoc, dir.orientation);
        }
    }
}

/**
 *  Calculates where a temple's moon gate should be.
 */
calculateGateLocation = { temple ->
    log.info("calculateGateLocation(" + temple + ")");
    def info = temple[StructureInfo];
    def trigger = temple[TriggerInfo];
    log.info("info:" + info);
    log.info("trigger:" + trigger);

    if( info == null ) {
        return null;
    }

    def loc = temple.location;
    def gateLoc = loc.add((int)(trigger.size.x/2), 2, (int)(trigger.size.z/2)).floor().toVec3d();
    gateLoc.addLocal(0.5, 0, 0.5);

    log.info("loc:" + loc + "  gate:" + gateLoc);

    return gateLoc;
}

/**
 *  Returns the location of the temple's gate if it already exists or calculates
 *  it if it doesn't.
 */
getGateLocation = { source, temple ->
    def gate = findTempleGate(temple);
    if( gate ) {
        return gate.location;
    }
    // The moon gate itself isn't generated until we actually approach
    // the temple so it's quite common that it won't exist yet.  In those
    // cases, we'll figure out where it would be and use that as the teleport
    // target.
    return calculateGateLocation(temple);

    // If we don't find that then we'll be SOL anyway.  The caller
    // will lookup the Tile which should cause everything to generate.
}

/**
 *  Utility method for clearing out the enabled or partially enabled state
 *  of a portal.  This resets it to fully disabled, no trigger, no shape, no special
 *  lighting.
 */
clearPortal = { self ->
    def portalLoc = self.location;
    world.setWorldCell(portalLoc, 0);
    self.remove(ShapeInfo.class);
    self.remove(TriggerInfo.class);
}



