/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

log.info("Default type:" + type("Default"))

createType("BaseObject").with {

    addGetter("name") { ->
        //log.info("target:" + target);
        //log.info("delegate:" + delegate);
        def on = delegate[ObjectName];
        //log.info("name test:" + on);
        //log.info("delegate test:" + delegate[ObjectName]);
        if( on != null ) {
            return on.getName(entityData);
        } 
        def n = delegate[Name];
        if( n != null ) {
            return n.getName();
        }
        return "Entity@" + target.getId(); 
    }

    // For now, all objects will be equippable if they are in inventory
    addAction("Equip") { ->
        // This should be in the "can run" check
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            log.error("Activator:" + activator + " tried to equip foreign object:" + object);
            return;
        }
                
        // Right now there is only one hand "Holder"
        activator << Holding.create(object, object.type.manipulator?:"Default")
    }

    addAction("Look") { ->
        def lines = [];
        
        lines += "Examine: " + object.name;

        def creator = object[CreatedBy]?.creatorId?.name?:"Unknown";
        
        def desc = object.type.getTypeVar("description", null);
        if( desc != null ) {
            // Bound to be a better groovy way to do this but so far
            // it only gets more complicated the deeper I drill.
            def s = desc.replaceAll("%creator", creator);
            s = s.replaceAll("%id", id as String);
            lines += s;
        } else {  
            lines += "ID:" + id;
            if( creator != null ) {
                lines += "Created by:" + creator;
            }
        }
 
        //lines += "Volume:" + object[ObjectVolume];
 
        session.showPrompt(target, PromptType.Dialog, lines.join("\n"));
        session.showOptions(target, []);
        
        return true;
    }
}

type("Default").with {
    supertypes("BaseObject");
}

createType("BuildWand").with {
    supertypes("BaseObject");
    
    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
              "The Build Wand is used for placing and removing blocks.",
              "",
              "The right mouse button will place the current block type", 
              "and the left mouse button will remove the block under the", 
              "cursor (highlighted yellow).", 
              "", 
              "Press 'e' to open the block type selector menu.", 
              "Press 'c' to select the block under the cursor as the", 
              "current type.", 
              "", 
              "The selected block will appear in the bottom center of the",
              "screen.  If the block supports multiple rotations then the",
              "mouse wheel will change the block rotation."
            ); 
    setTypeVar("handleLocation", new Vec3d(1.5, 1.5, 3));
    
    
    addAction("startDigging") { ->
        log.info("" + context + ".startDigging()");
    }

    addAction("stopDigging") { ->
        log.info("" + context + ".stopDigging()");
        def hits = activator.pick();
        if( hits.hasNext() ) {
            def intersect = hits.next();
            world.setWorldCell(intersect.getBlock().toVec3d(), 0);
        }
    }
    
    addAction("startFilling") { ->
        log.info("" + context + ".startFilling()");
    }

    addAction("stopFilling") { ->
        log.info("" + context + ".stopFilling()");
        def selected = env.getVar("selectedType", 1);
        def hits = activator.pick();
        if( hits.hasNext() ) {
            def intersect = hits.next();
            def pos = intersect.getBlock().toVec3d();
            pos.addLocal(intersect.getNormal());
            world.setWorldCell(pos, selected);
        }
    }

    addAction("selectType") { Integer index ->
        log.info("" + context + ".selectType(" + index + ")");
    
        env.setVar("selectedType", index);
        context << ObjectFocus.blockType(index); 
    }

    addAction("rotate") { Integer index ->
        log.info("" + context + ".rotate(" + index + ")");
        // Handled by the client right now
    }
} 

createType("ObjectTool").with {
    supertypes("BaseObject");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0)); 
    setTypeVar("description", 
               "The Object Tool is used for placing and updating objects.",
               "",
               "Clicking the right mouse button in an open location will",
               "place the current object type centered over that location.",
               "",
               "The mouse wheel will switch the object type.  The current",
               "object type is displayed in the bottom middle of the screen.",
               "",
               "Clicking the right mouse button on an existing object will",
               "open its action menu.",
               "",
               "Press the right mouse button down on an object and drag to",
               "move it.",
               "While dragging an object, the mouse wheel will rotate the",
               "object.",
               "",
               "New object types can be created in the blueprint editor",
               "accessible from the player menu (tab).",
             ); 

    addAction("Equip") { ->
        log.info("Subclass equip");
        
        // Need to run the real equip
        run(type("BaseObject"), "Equip");
 
        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);
        
        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;        
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }
        
        // See if the one we have is valid
        def focus = target[ObjectFocus]?.entityId;
        if( focus && blueprints.contains(focus) ) {
            // We're done... the current focus is valid. 
        } else {
            // Just use the first item
            focus = blueprints.find();
            target << ObjectFocus.entity(focus); 
        }
        
        env.setVar("selectedBlueprint", focus);
        
        return true;                         
    }

    addAction("place") { Vec3d loc -> 
        log.info("place:" + loc);
        
        def bp = env.getVar("selectedBlueprint", null);
        if( bp == null ) {
            // Not our job to set it
            return false;
        }        
        log.info("" + activator + " creating object at:" + loc + " with blueprint:" + bp);    

        def shape = bp[ShapeInfo];
        
        // Need to get the size so we can find its base's center.
        // We probably should keep this as part of the object type or even a separate
        // component.
        def cells = loadCells(shape);
        def size = cells.size;
        def offset = size * 0.5 * shape.scale;
        offset.y = 0;
        log.info("cells:" + cells + "  offset:" + offset);
        
        loc -= offset;

        // FIXME: make a createObject() method that encapsulates some of
        // this, calls a default object init action, etc..
        def pos = new SpawnPosition(loc, new Quatd());
        def name = bp[ObjectName];
        def placed = createEntity(
                pos, shape, name,
                new BlueprintSource(bp),
                new CreatedBy(activator.id)
            );
 
        return true;       
    }

    addAction("dragStart") { dragged, loc, facing, click ->
        log.info("dragStart:" + dragged + ", " + loc + ", " + facing + ", " + click);
        
        env.setVar("draggedObject", dragged);
        
        def shape = dragged[ShapeInfo];
        def cells = loadCells(shape);
        def size = cells.size;
        def offset = size * 0.5 * shape.scale;
        offset.y = 0;
        env.setVar("draggedOffset", offset);
    }

    addAction("drag") { dragged, loc, facing, click ->
        //log.info("drag:" + dragged + ", " + loc + ", " + facing + ", " + click);
 
        // Figure out the proper y-up facing       
        def xAxis = facing.mult(new Vec3d(1, 0, 0));
        double rads = Math.atan2(-xAxis.z, xAxis.x);    
        facing = new Quatd().fromAngles(0, rads, 0);
 
        // Figure out where the bottom center is so that
        // we can project down for ground       
        def center = loc;
        def offset = env.getVar("draggedOffset", null);
        if( offset ) {
            def localOffset = facing.mult(offset);
            center += localOffset;
        }
  
        // Check the ground
        def ray = new Rayd(center.add(0, 1, 0), new Vec3d(0, -1, 0));
        def hits = activator.pick(ray, 1);
        if( hits.hasNext() ) {
            def hit = hits.next().point.y;            
            if( loc.y < hit ) {
                loc.y = hit;
            }
        }
        
        dragged << new SpawnPosition(loc, facing);
    }
 
    addAction("dragEnd") { dragged ->
        log.info("dragEnd:" + dragged);
        
        // just clear the working env vars
        env.setVar("draggedObject", null);
        env.setVar("draggedOffset", null);        
    } 

    addAction("contextAction") { clicked, pos ->
        log.info("contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";
        
        // Here we want to let the player run some actions on the object
                   
        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        log.info("Player:" + activator.id + " creator:" + creator);
        
        def options = [];
        options.add(new Option("Look"));
        options.add(new Option(target, "copyBlueprint", clicked));
                    
        if( activator.isSameEntity(creator) ) {
            // Options to call us back as the manipulator
            options.add(new Option(target, "deleteObject", clicked));
        }
        session.showPrompt(clicked, PromptType.List, name); 
        session.showOptions(clicked, options);
    }

    // Targetted context operations
    //-----------------------------------------------
    addAction("copyBlueprint") { clicked ->        
        log.info("copyBlueprint:" + clicked);
    
        def bpSource = clicked[BlueprintSource];        
        def name = clicked.name;
        if( name == null ) {
            if( bpSource ) {
                name = bpSource.blueprintId.name?:"Unknown";
            } else {
                name = "Unknown";
            } 
        }
        
        def shape = clicked[ShapeInfo];        
        def cells = loadCells(shape);
 
        // Load the cells just so we can get a proper offset               
        def offset = cells.size * 0.5;
        offset.x = 5 - offset.x;
        offset.y = 0;
        offset.z = 5 - offset.z;
        offset = offset.floor(); // back to vec3i 
                 
        log.info("name:" + name + "  shape:" + shape + "  offset:" + offset);
 
        def copy = createEntity(
                    new BlueprintInfo(activator.id, name + " Copy", offset),
                    ObjectName.create(name),
                    shape,
                    // We'll make this player the creator but also
                    // set the blueprint source so we can still track the
                    // original author.  This player is likely to make changes
                    // the blueprint and we won't change the creator when that
                    // happens... so we need to change it now.
                    new CreatedBy(activator.id)
                    );
        if( bpSource ) {
            copy << bpSource;
        }        
    }

    addAction("deleteObject") { clicked ->        
        log.info("deleteObject:" + clicked);        
        // We should probably check to see if we can in case we hacked 
        // in around the context menu        
        removeEntity(clicked);        
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);
 
        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);
        
        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;        
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }
 
        // See what the current index is
        def selected = env.getVar("selectedBlueprint", null);
        int index = blueprints.findIndexOf { it == selected };
        index = (index + delta) % blueprints.size();
        if( index < 0 ) {
            index += blueprints.size(); 
        }
        selected = blueprints[index];
        target << ObjectFocus.entity(selected);
        env.setVar("selectedBlueprint", selected);        
    }
    
}

createType("Backpack").with {
    // Right now, no super type because we can't pick up, put down, etc.
    // a backpack... but eventually.
    //supertypes("BaseObject");
    
    addAction("drop") { dropped, xSlot, ySlot ->
        log.info("drop:" + dropped + " at:" + xSlot + ", " + ySlot);
        
        // Find out the root of this container object
        def root = object[ContainedIn].root;
        dropped << new ContainedIn(root, target, xSlot, ySlot);
    }
    
    addAction("itemActions") { clicked ->    
        log.info("itemActions:" + clicked);
        
        // Confirm that it really is in this container just in case
        if( !object.isSameEntity(clicked[ContainedIn].container) ) {
            log.warn("Running actions from container:" + target + " on object not in the container:" + clicked);
            return false;
        }
        
        // Assemble the action list
        def options = [];
        def clickedContext = clicked.context;
        for( def action : clicked.type.contextActions ) {
            if( !action.canRun(env, clickedContext) ) {
                log.info("Skipping:" + action);
                continue;
            }
            options.add(new Option(action.getName()));
        }
        session.showPrompt(clicked, PromptType.List, clicked.name);
        session.showOptions(clicked, options);
        
        return true; 
    }
}



createType("Axe").with {
    supertypes("BaseObject");
    
    setTypeVar("description",
        "This was axe was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(6, 0, 5));  
} 

createType("Sword").with {
    supertypes("BaseObject");
    
    setTypeVar("description",
        "This was axe was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(3, 0, 4)); 
} 

createType("Book").with {
    supertypes("BaseObject");
    
    setTypeVar("description",
        "This was book was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(0, 4, 1)); 
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, Math.PI, 0));     
} 
 
 
createType("TestObject").with {
    supertypes("BaseObject");

    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
        "This was book was created by %creator with ID %id."); 
    
    addAction("Test") { ->
        //log.info("" + this + "." + getName() + "(" + Arrays.asList(args) + ")");                        
        log.info("" + this + "." + "Test" + "(" + ")");
        log.info("owner:" + owner + " delegate:" + delegate);                        
        log.info("Activator:" + activator);
        log.info("Activator:" + getActivator());
        //log.info("variables:" + getVariables()); // from 'delegate'
        log.info("activator.vars:" + activator.vars); // from 'delegate'
        log.info("target:" + getTarget()); // from 'this'
    
        // This goes to a global binding
        testProperty = 123;
    
        //log.info("variables:" + getVariables()); // from 'delegate'
    
        //activator.target = "Test Target";
        //log.info("activator.vars:" + activator.vars); // from 'delegate'
        //log.info("target:" + getTarget()); // from 'this'?  Yep.
            
        //log.info("vars:" + activator.vars);
        //log.info("test:" + activator.vars.get("selectedType"));
        //log.info("test1:" + activator.getProperty("selectedType"));
        //log.info("test2:" + activator.selectedType);
        //activator.foo = 123;
        //log.info("test3:" + activator.foo);
        return true;
    }
    
    addAction("Test2") { String name ->
        log.info("Test2:" + name);
        return true;
    }        

    addAction("Test3") { ->
        log.info("Calling delegate version of run()");
        // This one calls it directly on ActionContext which is our 'this'
        object.run(env, "Test2", "This is a test")
        log.info("Calling non-delegate version of run()");
        run("Test2", "This is a different test")
        //run("BaseTest")
        
        def manipulator = object.getType().getTypeVar("manipulator", "Default");
        
        log.info("Manipulator:" + manipulator);
        log.info("test:" + (object.manipulator?:"Default"));
        
        log.info("holding1:" + Holding.create(object.getTarget(), manipulator, entityData)); 
        log.info("holding2:" + Holding.create(object.getTarget(), manipulator)); 
        log.info("holding3:" + Holding.create(object, manipulator)); 
        log.info("holding4:" + Holding.create(activator, manipulator)); 

        log.info("contains1:" + activator.contains(object)); 
 
        // What about running an action from a different type or one of
        // our super types?
        log.info("Running it the hard way--------------------");
        type("BaseObject").findAction("Look").run(env, context);

        log.info("Running it the easy way--------------------");
        run(type("BaseObject"), "Look");
        
        return true;
    }
    
    addGetter("test") { -> //target ->
        log.info("getter-test");
        log.info("  getter-ed:" + system(EntityData));
        log.info("  getter-target:" + target);
        log.info("  getter-activator:" + activator);
        log.info("  owner:" + owner);
        log.info("  delegate:" + delegate);
        log.info("  this:" + this);
        return "foo-123"
    }
 
    //addGetter("name") { ->
    //    log.info("target:" + target);
    //    log.info("delegate:" + delegate);
    //    def on = delegate[ObjectName];
    //    log.info("name test:" + on);
    //    log.info("delegate test:" + delegate[ObjectName]);
    //    if( on != null ) {
    //        return on.getName(entityData);
    //    } 
    //    def n = delegate[Name];
    //    if( n != null ) {
    //        return n.getName();
    //    }
    //    return "Entity@" + target.getId(); 
    //}
    
}


