/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

log.info("Default type:" + type("Default"))

createType("BaseObject").with {

    addGetter("name") { ->
        //log.info("target:" + target);
        //log.info("delegate:" + delegate);
        def on = delegate[ObjectName];
        //log.info("name test:" + on);
        //log.info("delegate test:" + delegate[ObjectName]);
        if( on != null ) {
            return on.getName(entityData);
        } 
        def n = delegate[Name];
        if( n != null ) {
            return n.getName();
        }
        return "Entity@" + target.getId(); 
    }

    addAction("Look") { ->
        def lines = [];
        
        lines += "Examine: " + object.name;

        def creator = object[CreatedBy]?.creatorId?.name?:"Unknown";
        
        def desc = object.type.getTypeVar("description", null);
        if( desc != null ) {
            // Bound to be a better groovy way to do this but so far
            // it only gets more complicated the deeper I drill.
            def s = desc.replaceAll("%creator", creator);
            s = s.replaceAll("%id", id as String);
            lines += s;
        } else {  
            lines += "ID:" + id;
            if( creator != null ) {
                lines += "Created by:" + creator;
            }
        }
 
        //lines += "Volume:" + object[ObjectVolume];
 
        session.showPrompt(target, PromptType.Dialog, lines.join("\n"));
        session.showOptions(target, []);
        
        return true;
    }
    
    addAction("move") { loc, facing ->
        //log.info("move:" + loc + ", " + facing + "  by:" + activator);
        // By default, we'll just let everything go wherever
        object << new SpawnPosition(loc, facing);
    }
}

// An object that can defer to the default hand-based manpulation
// drag, drop, activate, etc..
createType("ObjectManipulator").with {
    supertypes("BaseObject");
    
    // For now, we'll use the creator manipulator type but probably
    // there should be a different one that the creator manipulator extends.
    setTypeVar("manipulator", "Hand");
    
    addAction("dragStart") { dragged, loc, facing, click ->
        log.info("dragStart:" + dragged + ", " + loc + ", " + facing + ", " + click);
        
        env.setVar("draggedObject", dragged);
 
        // See if we are allowed to move this object
        def objPos = dragged[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canMoveObject() ) {
            env.setVar("objectMoveAllowed", false);
            echo("You do not have permission to move objects here.");
            return;
        }
        env.setVar("objectMoveAllowed", true);
 
        def offset;       
        def shape = dragged[ShapeInfo];
        if( shape != null ) {
            def cells = loadCells(shape);
            def size = cells.size;
            offset = size * 0.5 * shape.scale;
            offset.y = 0;
        } else {
            // FIXME: lookup the alternate shape.
            // We don't use the offset for very much so right now in the the
            // limited use-cases where we don't have a shape (ClaimMarkers) 
            // 0 is ok.
            offset = new Vec3d();
        }
        env.setVar("draggedOffset", offset);
    }

    addAction("drag") { dragged, loc, facing, click ->
        //log.info("drag:" + dragged + ", " + loc + ", " + facing + ", " + click);
  
        if( !env.getVar("objectMoveAllowed", false) ) {
            // Drag all you want, nothing is going to happen            
            return;
        }
 
        // Figure out the proper y-up facing       
        def xAxis = facing.mult(new Vec3d(1, 0, 0));
        double rads = Math.atan2(-xAxis.z, xAxis.x);    
        facing = new Quatd().fromAngles(0, rads, 0);
 
        // Figure out where the bottom center is so that
        // we can project down for ground       
        def center = loc;
        def offset = env.getVar("draggedOffset", null);
        if( offset ) {
            def localOffset = facing.mult(offset);
            center += localOffset;
        }
  
        // Check the ground
        def ray = new Rayd(center.add(0, 1, 0), new Vec3d(0, -1, 0));
        def hits = activator.pick(ray, 1);
        if( hits.hasNext() ) {
            def hit = hits.next().point.y;            
            if( loc.y < hit ) {
                loc.y = hit;
            }
        }
 
        // Now that we know where we'll put it, we need to check to see
        // if it was dragged into an area that isn't allowed.
        if( !getPerms(activator, loc).canMoveObject() ) {
            echo("You do not have permission to move objects into that area.");
            return;
        }
 
        // Base object has a move... so everything should have a move
        // that we will be trying to drag.       
        dragged.run(env, "move", loc, facing);        
        //dragged << new SpawnPosition(loc, facing);
    }
 
    addAction("dragEnd") { dragged ->
        log.info("dragEnd:" + dragged);
        
        // just clear the working env vars
        env.setVar("draggedObject", null);
        env.setVar("draggedOffset", null);
        env.setVar("objectMoveAllowed", null);        
    } 

    addAction("contextAction") { clicked, pos ->
        log.info("contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";
        
        // Here we want to let the player run some actions on the object
                   
        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        log.info("Player:" + activator.id + " creator:" + creator);
        
        def options = [];
        options.add(new Option("Look"));
        session.showPrompt(clicked, PromptType.List, name); 
        session.showOptions(clicked, options);
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);
    }

    addAction("altPressGround") { Vec3d loc -> 
        log.info("altPressGround:" + loc);
    }

    addAction("altClickGround") { Vec3d loc -> 
        log.info("altClickGround:" + loc);
    }
    
}

// For things that can be in inventory and equipped and stuff
createType("BaseItem").with {
    
    // Let all equippable items default to the hand manipulator with
    // default actions.  This will automatically let the user continue
    // to perform standard manipulation actions if the object doesn't
    // define its own.
    supertypes("ObjectManipulator");

    // For now, all objects will be equippable if they are in inventory
    addAction("Equip") { ->
        // This should be in the "can run" check
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            log.error("Activator:" + activator + " tried to equip foreign object:" + object);
            return;
        }
                
        // Right now there is only one hand "Holder"
        activator << Holding.create(object, object.type.manipulator?:"Default")
    }

    addAction("Unequip") { ->
        // This should be in the "can run" check
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            log.error("Activator:" + activator + " tried to unequip foreign object:" + object);
            return;
        }
 
        // See if the current holder is holding us or not
        def holding = activator[Holding];
        log.info("Unequip holding:" + holding);
        if( !object.isSameEntity(holding?.target) ) {
            log.warn("Activator:" + activator + " trying to unequip item that is not equipped:" + object);
        }

        log.info("Unequpping:" + object);
        activator.remove(Holding);                
    }    
}

type("Default").with {
    // By default, objects will have manipulator actions but not be equippable.
    // This is mostly because we let the empty hand be its own target when
    // it is not holding things... and equipping/unequipping a hand doesn't
    // seem logical.
    supertypes("ObjectManipulator");
}

createType("BuildWand").with {
    supertypes("BaseItem");
    
    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
              "The Build Wand is used for placing and removing blocks.",
              "",
              "The right mouse button will place the current block type", 
              "and the left mouse button will remove the block under the", 
              "cursor (highlighted yellow).", 
              "", 
              "Press 'e' to open the block type selector menu.", 
              "Press 'c' to select the block under the cursor as the", 
              "current type.", 
              "", 
              "The selected block will appear in the bottom center of the",
              "screen.  If the block supports multiple rotations then the",
              "mouse wheel will change the block rotation."
            ); 
    setTypeVar("handleLocation", new Vec3d(1.5, 1.5, 3));
    
    
    addAction("startDigging") { ->
        log.info("" + context + ".startDigging()");
    }

    addAction("stopDigging") { ->
        log.info("" + context + ".stopDigging()");
                
        def hits = activator.pick();

        if( hits.hasNext() ) {
            def intersect = hits.next();
            if( !getPerms(activator, intersect.getBlock()).canRemoveBlock() ) {
                echo("You do not have permission to remove blocks here.");
                return;
            }
            world.setWorldCell(intersect.getBlock().toVec3d(), 0);
        }
    }
    
    addAction("startFilling") { ->
        log.info("" + context + ".startFilling()");
    }

    addAction("stopFilling") { ->
        log.info("" + context + ".stopFilling()");
        def selected = env.getVar("selectedType", 1);
        def hits = activator.pick();
        if( hits.hasNext() ) {
            def intersect = hits.next();
            def pos = intersect.getBlock().toVec3d();
            pos.addLocal(intersect.getNormal());
            if( !getPerms(activator, pos).canAddBlock() ) {
                echo("You do not have permission to place blocks here.");
                return;
            }
            log.info("claim:" + getClaim(activator, pos));
            world.setWorldCell(pos, selected);
        }
    }

    addAction("selectType") { Integer index ->
        log.info("" + context + ".selectType(" + index + ")");
    
        env.setVar("selectedType", index);
        context << ObjectFocus.blockType(index); 
    }

    addAction("rotate") { Integer index ->
        log.info("" + context + ".rotate(" + index + ")");
        // Handled by the client right now
    }
} 

createType("ObjectTool").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0)); 
    setTypeVar("description", 
               "The Object Tool is used for placing and updating objects.",
               "",
               "Clicking the right mouse button in an open location will",
               "place the current object type centered over that location.",
               "",
               "The mouse wheel will switch the object type.  The current",
               "object type is displayed in the bottom middle of the screen.",
               "",
               "Clicking the right mouse button on an existing object will",
               "open its action menu.",
               "",
               "Press the right mouse button down on an object and drag to",
               "move it.",
               "While dragging an object, the mouse wheel will rotate the",
               "object.",
               "",
               "New object types can be created in the blueprint editor",
               "accessible from the player menu (tab).",
             ); 

    addAction("Equip") { ->
        log.info("Subclass equip");
        
        // Need to run the real equip
        run(type("BaseItem"), "Equip");
 
        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);
        
        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;        
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }
        
        // See if the one we have is valid
        def focus = target[ObjectFocus]?.entityId;
        if( focus && blueprints.contains(focus) ) {
            // We're done... the current focus is valid. 
        } else {
            // Just use the first item
            focus = blueprints.find();
            target << ObjectFocus.entity(focus); 
        }
        
        env.setVar("selectedBlueprint", focus);
        
        return true;                         
    }

    addAction("place") { Vec3d loc -> 
        log.info("place:" + loc);
 
        if( !getPerms(activator, loc).canAddObject() ) {
            echo("You do not have permission to place objects here.");
            return;
        }
        
        def bp = env.getVar("selectedBlueprint", null);
        if( bp == null ) {
            // Not our job to set it
            return false;
        }        
        log.info("" + activator + " creating object at:" + loc + " with blueprint:" + bp);    

        def shape = bp[ShapeInfo];
        
        // Need to get the size so we can find its base's center.
        // We probably should keep this as part of the object type or even a separate
        // component.
        def cells = loadCells(shape);
        def size = cells.size;
        def offset = size * 0.5 * shape.scale;
        offset.y = 0;
        log.info("cells:" + cells + "  offset:" + offset);
        
        loc -= offset;

        // FIXME: make a createObject() method that encapsulates some of
        // this, calls a default object init action, etc..
        def pos = new SpawnPosition(loc, new Quatd());
        def name = bp[ObjectName];
        def placed = createEntity(
                pos, shape, name,
                new BlueprintSource(bp),
                new CreatedBy(activator.id)
            );
 
        return true;       
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);
 
        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);
        
        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;        
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }
 
        // See what the current index is
        def selected = env.getVar("selectedBlueprint", null);
        int index = blueprints.findIndexOf { it == selected };
        index = (index + delta) % blueprints.size();
        if( index < 0 ) {
            index += blueprints.size(); 
        }
        selected = blueprints[index];
        target << ObjectFocus.entity(selected);
        env.setVar("selectedBlueprint", selected);        
    }

    addAction("contextAction") { clicked, pos ->
        log.info("contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";
        
        // Here we want to let the player run some actions on the object
                   
        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        log.info("Player:" + activator.id + " creator:" + creator);
        
        def options = [];
        options.add(new Option("Look"));
        
        // Can't copy the blueprint unless it has a shape
        boolean hasShape = clicked[ShapeInfo] != null; 
        if( hasShape ) {
            // For the blueprint related actions, we are the thing upon
            // which the action is run and the object is the parameter.
            options.add(new Option(target, "copyBlueprint", clicked));
        }
 
        // Instead of hasShape... we should probably base this on
        // the actions that the object has.  That way we can have non-removable
        // objects that have shapes.  And then move "deleteObject" to the objects
        // that support delete... the others can echo a message that they can't
        // be removed.
        // FIXME: use the runnable action list when we have it... ie: not all 
        // actions, just the ones that canRun().       
        def objPos = clicked[SpawnPosition]?.location;
        if( hasShape && getPerms(activator, objPos).canRemoveObject() ) {
            // Options to call us back as the manipulator
            options.add(new Option(target, "deleteObject", clicked));
        }
        session.showPrompt(clicked, PromptType.List, name); 
        session.showOptions(clicked, options);
    }

    // Targetted context operations
    //-----------------------------------------------
    addAction("copyBlueprint") { clicked ->        
        log.info("copyBlueprint:" + clicked);
    
        def bpSource = clicked[BlueprintSource];        
        def name = clicked.name;
        if( name == null ) {
            if( bpSource ) {
                name = bpSource.blueprintId.name?:"Unknown";
            } else {
                name = "Unknown";
            } 
        }
        
        def shape = clicked[ShapeInfo];        
        def cells = loadCells(shape);
 
        // Load the cells just so we can get a proper offset               
        def offset = cells.size * 0.5;
        offset.x = 5 - offset.x;
        offset.y = 0;
        offset.z = 5 - offset.z;
        offset = offset.floor(); // back to vec3i 
                 
        log.info("name:" + name + "  shape:" + shape + "  offset:" + offset);
 
        def copy = createEntity(
                    new BlueprintInfo(activator.id, name + " Copy", offset),
                    ObjectName.create(name),
                    shape,
                    // We'll make this player the creator but also
                    // set the blueprint source so we can still track the
                    // original author.  This player is likely to make changes
                    // the blueprint and we won't change the creator when that
                    // happens... so we need to change it now.
                    new CreatedBy(activator.id)
                    );
        if( bpSource ) {
            copy << bpSource;
        }        
    }

    addAction("deleteObject") { clicked ->        
        log.info("deleteObject:" + clicked);
        
        def objPos = clicked[SpawnPosition]?.location;        
        if( !getPerms(activator, objPos).canRemoveObject() ) {
            echo("You do not have permission to remove objects here.");
            return;
        }
        // We should probably check to see if we can in case we hacked 
        // in around the context menu        
        removeEntity(clicked);        
    }    
}

createType("Backpack").with {
    // Right now, just extend base object since we cannot pickup/put down
    // a backpack yet and therefore there is also no equip/unequip... but
    // we might want to look at it or get its name.
    supertypes("BaseObject");
    
    addAction("drop") { dropped, xSlot, ySlot ->
        log.info("drop:" + dropped + " at:" + xSlot + ", " + ySlot);
        
        // Find out the root of this container object
        def root = object[ContainedIn].root;
        dropped << new ContainedIn(root, target, xSlot, ySlot);
    }
    
    addAction("itemActions") { clicked ->    
        log.info("itemActions:" + clicked);
        
        // Confirm that it really is in this container just in case
        if( !object.isSameEntity(clicked[ContainedIn].container) ) {
            log.warn("Running actions from container:" + target + " on object not in the container:" + clicked);
            return false;
        }
        
        // Assemble the action list
        def options = [];
        def clickedContext = clicked.context;
        for( def action : clicked.type.contextActions ) {
            if( !action.canRun(env, clickedContext) ) {
                log.info("Skipping:" + action);
                continue;
            }
            options.add(new Option(action.getName()));
        }
        session.showPrompt(clicked, PromptType.List, clicked.name);
        session.showOptions(clicked, options);
        
        return true; 
    }
}



createType("Axe").with {
    supertypes("BaseItem");
    
    setTypeVar("description",
        "This was axe was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(6, 0, 5));  
} 

createType("Sword").with {
    supertypes("BaseItem");
    
    setTypeVar("description",
        "This was axe was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(3, 1, 4));
    setTypeVar("handleRotation", new Quatd().fromAngles(0, 0, -Math.PI * 0.5));      
} 

createType("Book").with {
    supertypes("BaseItem");
    
    setTypeVar("description",
        "This was book was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(0, 4, 1)); 
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, Math.PI, 0));     
} 
 
 
createType("TestObject").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
        "This was book was created by %creator with ID %id."); 
    
    addAction("Test") { ->
        //log.info("" + this + "." + getName() + "(" + Arrays.asList(args) + ")");                        
        log.info("" + this + "." + "Test" + "(" + ")");
        log.info("owner:" + owner + " delegate:" + delegate);                        
        log.info("Activator:" + activator);
        log.info("Activator:" + getActivator());
        //log.info("variables:" + getVariables()); // from 'delegate'
        log.info("activator.vars:" + activator.vars); // from 'delegate'
        log.info("target:" + getTarget()); // from 'this'
    
        // This goes to a global binding
        testProperty = 123;
    
        //log.info("variables:" + getVariables()); // from 'delegate'
    
        //activator.target = "Test Target";
        //log.info("activator.vars:" + activator.vars); // from 'delegate'
        //log.info("target:" + getTarget()); // from 'this'?  Yep.
            
        //log.info("vars:" + activator.vars);
        //log.info("test:" + activator.vars.get("selectedType"));
        //log.info("test1:" + activator.getProperty("selectedType"));
        //log.info("test2:" + activator.selectedType);
        //activator.foo = 123;
        //log.info("test3:" + activator.foo);
        return true;
    }
    
    addAction("Test2") { String name ->
        log.info("Test2:" + name);
        return true;
    }        

    addAction("Test3") { ->
        log.info("Calling delegate version of run()");
        // This one calls it directly on ActionContext which is our 'this'
        object.run(env, "Test2", "This is a test")
        log.info("Calling non-delegate version of run()");
        run("Test2", "This is a different test")
        //run("BaseTest")
        
        def manipulator = object.getType().getTypeVar("manipulator", "Default");
        
        log.info("Manipulator:" + manipulator);
        log.info("test:" + (object.manipulator?:"Default"));
        
        log.info("holding1:" + Holding.create(object.getTarget(), manipulator, entityData)); 
        log.info("holding2:" + Holding.create(object.getTarget(), manipulator)); 
        log.info("holding3:" + Holding.create(object, manipulator)); 
        log.info("holding4:" + Holding.create(activator, manipulator)); 

        log.info("contains1:" + activator.contains(object)); 
 
        // What about running an action from a different type or one of
        // our super types?
        log.info("Running it the hard way--------------------");
        type("BaseObject").findAction("Look").run(env, context);

        log.info("Running it the easy way--------------------");
        run(type("BaseObject"), "Look");
        
        return true;
    }
    
    addGetter("test") { -> //target ->
        log.info("getter-test");
        log.info("  getter-ed:" + system(EntityData));
        log.info("  getter-target:" + target);
        log.info("  getter-activator:" + activator);
        log.info("  owner:" + owner);
        log.info("  delegate:" + delegate);
        log.info("  this:" + this);
        return "foo-123"
    }
 
    //addGetter("name") { ->
    //    log.info("target:" + target);
    //    log.info("delegate:" + delegate);
    //    def on = delegate[ObjectName];
    //    log.info("name test:" + on);
    //    log.info("delegate test:" + delegate[ObjectName]);
    //    if( on != null ) {
    //        return on.getName(entityData);
    //    } 
    //    def n = delegate[Name];
    //    if( n != null ) {
    //        return n.getName();
    //    }
    //    return "Entity@" + target.getId(); 
    //}
    
}


