#import "Common/ShaderLib/GLSLCompat.glsllib"

#if defined(DISCARD_ALPHA)
    uniform float m_AlphaDiscardThreshold;
#endif

uniform vec4 m_Color;
uniform sampler2D m_TerrainMap;
uniform sampler2D m_FluidMap;

varying vec2 texCoord1;

const float TEXTURE_SIZE = 1024.0;
const float RESOLUTION = 1.0/TEXTURE_SIZE;
const float TERRAIN_MULT = 1.0;

const float GRID_BORDER = 1.5;

#define SAMPLE_RADIUS 9
#define KERNEL_SIZE   (SAMPLE_RADIUS * 2 + 1)
//#define SAMPLE_SIZE   0.001
#define SAMPLE_SIZE   0.003
// Need to make sure that SAMPLE_SIZE * SAMPLE_RADIUS * 1024 does not exceed half a tile

// Pulls the elevation out of the texture value
float texToElevation( vec4 color ) {
    // Height is in 0 = bedrock space
    return (color.b * 255.0 * 255.0 + color.a * 255.0) * TERRAIN_MULT;
}

void main(){
    vec4 color = vec4(1.0);

    vec4 terrain = texture2D(m_TerrainMap, texCoord1);
    vec4 fluid = texture2D(m_FluidMap, texCoord1);
    float height = texToElevation(terrain);
    float dist = abs(height - 128.0);

    #ifdef ACCURATE_FLUID
        float fluidHeight = texToElevation(fluid);
        if( fluidHeight > 128.0 ) {
            dist = abs(height - fluidHeight);
        }
    #endif

    float ground = 0.0;
    float sea = 0.0;
    float samples = 0.0;
    for( int i = -SAMPLE_RADIUS; i <= SAMPLE_RADIUS; i++ ) {
        for( int j = -SAMPLE_RADIUS; j <= SAMPLE_RADIUS; j++ ) {
            float x = float(i);
            float z = float(j);
            vec2 uv = texCoord1 + vec2(x * SAMPLE_SIZE, z * SAMPLE_SIZE);
            float y = texToElevation(texture2D(m_TerrainMap, uv));
            float yFluid = texToElevation(texture2D(m_FluidMap, uv));
        #ifdef ACCURATE_FLUID
            if( y <= yFluid ) {
        #else
            if( y < 128.0 ) {
        #endif
                sea += 1.0;
            } else {
                ground += 1.0;
            }
            samples += 1.0;
        }
    }
    vec4 seaColor = vec4(0.0, 0.3, 1.0, 1.0 - (sea/samples));
    vec4 groundColor = vec4(0.2, 0.7, 0.2, 1.0 - (ground/samples));
    vec4 base;
#ifdef ACCURATE_FLUID
    if( height <= fluidHeight ) {
#else
    if( height < 128.0 ) {
#endif
        base = seaColor;
    } else {
        base = groundColor;
    }
    color = mix(vec4(0.0, 0.0, 0.0, 1.0), base, min(1.0, dist * 0.1));

    float mountains = smoothstep(200.0, 640.0, height);
    color = mix(color, vec4(89.0/255.0, 64.0/255.0, 39.0/255.0, 1.0), mountains);

    vec2 xy = texCoord1 * 1024.0;
    vec2 grid = round(xy / 64.0) * 64.0;
    float xDist = min(GRID_BORDER, abs(grid.x - xy.x)) / GRID_BORDER;
    float yDist = min(GRID_BORDER, abs(grid.y - xy.y)) / GRID_BORDER;
    float gridFactor = min(1.0, xDist * yDist * 1.0);
    color = mix(vec4(0.0, 0.0, 0.0, 0.5), color, gridFactor);

    #ifdef HAS_COLOR
        color *= m_Color;
    #endif

    #if defined(DISCARD_ALPHA)
        if(color.a < m_AlphaDiscardThreshold){
           discard;
        }
    #endif

    gl_FragColor = color;
}
