
def adjustPosition = { target, x, z ->
    def pos = target[SpawnPosition];
    def loc = pos.location;
    if( log.isDebugEnabled() ) {
        log.debug("Moving:" + target + "  from:" + loc + "  to:" + x + ", " + z);
    }
    loc.x = x;
    loc.z = z;
    target << new SpawnPosition(GameConstants.PHYSICS_GRID, loc); 
}

createType("Claim").with {
    // We can move claims around by themselves since they are their
    // own marker... thus we need to be a base object at minimum
    //supertypes("BaseObject");
    // And since I'm playing with equipping claims directly for placement
    // rather than managing a stand-in temporary entity... let's make them
    // extend the standard manipulator
    supertypes("ObjectManipulator");

    setTypeVar("handleLocation", new Vec3d(2.5, 5, 0.5)); 
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
         
    setTypeVar("description",
        "Property marker for a protected world area.");
 
    addAction("updateArea") { corner, area ->
        if( log.isDebugEnabled() ) {
            log.debug("Update area:" + object + "  from:" + corner + "  area:" + area);
        }
        
        // public Set<EntityId> findEntities( ComponentFilter filter, Class... types );
 
        object << area;
        
        def markers = findEntities(Filters.fieldEquals(ClaimMarker, "claim", target), SpawnPosition, ClaimMarker);
        markers = new ArrayList(markers).sort();
        
        // Remove the center marker from the list
        markers.remove(target);

        if( log.isDebugEnabled() ) {
            log.debug("markers:" + markers);
        }
        
        // It doesn't matter much which corner we move where except that players
        // might be dragging the corners around... so we at least don't want to move
        // the corner passed in.  We'll try to be the least destructive possible
        // so we'll sort the markers and update them in the same order we created them.
 
        def min = area.min;
        def max = area.max;
 
        // Add 0.5 to move them to the center of their cells       
        adjustPosition(markers[0], min.x + 0.5, min.z + 0.5);
        adjustPosition(markers[1], max.x + 0.5, max.z + 0.5);
        adjustPosition(markers[2], min.x + 0.5, max.z + 0.5);
        adjustPosition(markers[3], max.x + 0.5, min.z + 0.5);
    }
    
    addAction("retrieve") { ->
        if( object[ClaimArea] == null ) {
            echo("Can't retrieve a property that is not placed in the world.");
            return;
        }
 
        // Remove its world presence       
        object.remove(ClaimArea);
        object.remove(SpawnPosition);
        object.remove(LargeGridCell);
        
        // And remove any of the corner markers
        findEntities(Filters.fieldEquals(ClaimMarker, "claim", target), ClaimMarker).each { marker ->
            if( marker.isSameEntity(object) ) {
                return;
            }
            log.info("Removing marker:" + marker);
            removeEntity(marker); 
        }
    }
    
    addAction("prepareToPlace") { ->
        log.info("activator:" + activator + " wants to place this property:" + object);
        
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying on our person
            log.error("Activator:" + activator + " tried to equip foreign property object:" + object);
            return;
        }
 
        // Remove the claim marker while we have a temporary shape info.
        // This avoids having the claim view state think there is a claim to render
        // while we are holding it in our hand.
        object.remove(ClaimMarker);  
 
        // Temporarily set a model so that we can be equipped and seen
        double itemScale = 0.25 * 0.25; // really?  Seems big.
        //itemScale *= 0.5;
        //itemScale *= 0.5;
        object << ShapeInfo.create("/Models/symbols/property-marker.blocks", itemScale, entityData)
        
        // Set this object's focus to itself
        object << ObjectFocus.entity(object.target);        
                
        // Right now there is only one hand "Holder"
        activator << Holding.create(object, object.type.manipulator?:"Default")
    }
 
    // To do the actual placement if equipped   
    addAction("altPressGround") { Vec3d loc ->
        // We should put back the item that was previously equipped
        
        // See if we can place an area here or not
        // ...in the short term, we'll just make the claim area really small
        // and let the user expand it.  But we'll still need to check for claim
        // intersection.
        def v1 = loc.floor().subtractLocal(1, 0, 1);
        def v2 = v1.add(2, 0, 2);
        def v3 = v1.add(0, 0, 2);
        def v4 = v1.add(2, 0, 0);
        def area = new ClaimArea(v1, v2);
 
        for( def found : getIntersections(area) ) {        
            // Right now we don't yet support subplots so it's ok
            // to just abort if we find anything
            echo("The property cannot be placed here; It intersects another property.");
            return;
        }
 
        // Move the markers to the center of their cells
        def offset = new Vec3d(0.5, 0, 0.5); 
        
        // So we should be ok to place the claim and create the corner markers
        try {
            def claim = object.target;
            object << [
                    new SpawnPosition(GameConstants.PHYSICS_GRID, loc),
                    new LargeGridCell(GameConstants.LARGE_OBJECT_GRID.worldToId(loc)),
                    area,
                    ClaimMarker.create(claim, "/Models/symbols/property-marker.blocks", entityData)                   
                ]
                
            // object.name will look for the ObjectName first and we only care
            // about Name
            def name = object[Name]?.name?:"Unknown";
            createEntity(
                new Name(name + " Corner Marker"),
                ObjectTypeInfo.create("CornerMarker", entityData),
                ClaimMarker.create(claim, "/Models/symbols/property-corner.blocks", entityData),
                new SpawnPosition(GameConstants.PHYSICS_GRID, v1 + offset)
                );        
            createEntity(
                new Name(name + " Corner Marker"),
                ObjectTypeInfo.create("CornerMarker", entityData),
                ClaimMarker.create(claim, "/Models/symbols/property-corner.blocks", entityData),
                new SpawnPosition(GameConstants.PHYSICS_GRID, v2 + offset)
                );        
            createEntity(
                new Name(name + " Corner Marker"),
                ObjectTypeInfo.create("CornerMarker", entityData),
                ClaimMarker.create(claim, "/Models/symbols/property-corner.blocks", entityData),
                new SpawnPosition(GameConstants.PHYSICS_GRID, v3 + offset)
                );        
            createEntity(
                new Name(name + " Corner Marker"),
                ObjectTypeInfo.create("CornerMarker", entityData),
                ClaimMarker.create(claim, "/Models/symbols/property-corner.blocks", entityData),
                new SpawnPosition(GameConstants.PHYSICS_GRID, v4 + offset)
                );                    
        } finally {
        
            // Clear the things we temporarily added
            object.remove(ShapeInfo); // would make the object show up as a regular model
            object.remove(ObjectFocus); // wastes space when we don't need it
    
            // See if the current holder is holding us or not
            def holding = activator[Holding];
            log.info("Unequip holding:" + holding);
            if( !object.isSameEntity(holding?.target) ) {
                log.warn("Activator:" + activator + " trying to unequip item that is not equipped:" + object);
            }
    
            log.info("Unequpping:" + object);
            activator.remove(Holding);
        }
    }

    // When we move the claim marker we also need to keep track of the
    // LOB grid because that's what we use to reduce searches.     
    addAction("move") { loc, facing ->
    
        // Limit the marker to be within the claim area
        def area = object[ClaimArea];
        if( !area.contains(loc) ) {
            echo("Marker must remain in claim area.");
            return;
        }
    
        object << [
                new SpawnPosition(loc, facing),
                new LargeGridCell(GameConstants.LARGE_OBJECT_GRID.worldToId(loc))
            ];
    }


}


createType("CornerMarker").with {
    supertypes("BaseObject");
    
    setTypeVar("description",
        "Used to adjust property boundaries.");
        
    addAction("move") { loc, facing ->
        //log.info("move:" + loc + ", " + facing + "  by:" + activator);
        def marker = object[ClaimMarker];
        def pos = object[SpawnPosition];
        if( marker == null || pos == null ) {
            log.warn("Claim is missing things, marker:" + marker + ", pos:" + pos);
            return;
        }
        
        // Can we just move on cell changes?
        def from = pos.location.floor();
        def to = loc.floor();
        if( from == to ) {
            //log.info(".....not a real move");
            return;
        }
 
        def existing = marker.claim[ClaimArea];
        def newArea = existing.moveCorner(from, to);
 
        // See if the area size is valid
        if( newArea.sizeX < 2 || newArea.sizeZ < 2 ) {
            return;
        }
        def type = marker.claim[ClaimType];
        if( newArea.area > type.maxArea ) {
            echo("Property area:" + newArea.area + " sq. meters would exceed maximum area:" + type.maxArea + " sq. meters.");
            return;
        }
 
        // See if the new area is even allowed
        for( def found : getIntersections(newArea) ) {
            // Right now we don't yet support subplots so it's ok
            // to just abort if we find anything other than ourselves
            if( found == marker.claim ) {
                continue;
            }
            // Right now we don't yet support subplots so it's ok
            // to just abort if we find anything
            echo("The property cannot be moved here; It intersects another property.");
            return;
        }        

        // Update the marker itself... we do this so that it picks up the
        // latest adjusted y value.
        object << new SpawnPosition(to.toVec3d(), facing);
        
        // Let the claim know it needs to update its area              
        marker?.claim?.run(env, "updateArea", object, newArea);
    }
            
} 


