/*
 * $Id$
 * 
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */
import mythruna.shell.*;
import mythruna.net.server.AccountHostedService;
import mythruna.net.server.GameSessionHostedService;

addShellCommand = { CommandShell target, String name, List help, Closure doIt ->
    ShellCommand cmd = new AbstractShellCommand(help as String[]) {
                public Object execute( CommandShell shell, String args ) {
                    // rehydrate(Object delegate, Object owner, Object thisObject) 
                    Closure c = doIt.rehydrate(shell, doIt.owner, doIt);
                    int length = doIt.getParameterTypes().length;
                    if( length == 2 ) {
                        c(shell, args);
                    } else { 
                        c(args);
                    }                    
                    return null;
                }        
            }; 
    target.addCommand(name, cmd); 
    return cmd;           
} 

adminCommand = { String name, List help, Closure doIt ->
    addShellCommand(server.adminShell, name, help, doIt);
    //ShellCommand cmd = new AbstractShellCommand(help as String[]) {
    //            public Object execute( CommandShell shell, String args ) {
    //                int length = doIt.getParameterTypes().length;
    //                if( length == 2 ) {
    //                    doIt(shell, args);
    //                } else { 
    //                    doIt(args);
    //                }                    
    //                return null;
    //            }        
    //        }; 
    //server.adminShell.addCommand(name, cmd); 
    //return cmd;           
} 

// Add some standard shell commands
def help = [
    "<connection#> [message] - disconnects the specified connection.",
    "Where:",
    "  <connection#> - is the integer connection number.",
    "  [message] - is an optional message to send with the disconnect."
    ];
adminCommand("kick", help) { args ->
    def id;
    def message;
    int split = args.indexOf(' ');
    if( split < 0 ) {
        id = args as int;
        message = "The admin has closed your connection."
    } else {
        id = args.substring(0, split) as int;
        message = args.substring(split + 1);
    }
    def conn = server.server.getConnection(id);
    if( conn == null ) {
        throw new IllegalArgumentException("Invalid connection ID:" + id);
    }
    log.info("Disconnecting:" + conn);
    conn.close(message);
}    

getAccount = { String id ->
    if( id.isInteger() ) {
        def conn = server.server.getConnection(id as int);
        if( conn == null ) {
            throw new IllegalArgumentException("ID is not an active connection:" + id);
        }
        def account = AccountHostedService.getAccount(conn);
        if( account == null ) {
            throw new IllegalArgumentException("Connection is not logged in yet:" + id);
        }
        return account;      
    } else {
        def account = accountManager.getAccount(id);
        if( account == null ) {
            throw new IllegalArgumentException("User ID is not a valid user:" + id)
        }
        return account; 
    }
}

help = [
    "<connection#|userId> - bans the specified user from future logins.",
    "Where:",
    "  <connection#|userId> - is the connection number of the user to",
    "       ban or their user ID."
    ];
adminCommand("ban", help) { shell, args ->
    def account = getAccount(args);
    account.setProperty("banned", true);
    account.setProperty("bannedUpdateTime", System.currentTimeMillis());
    account.save();
    shell.echo("Banned:" + account.getUserId());    
}

help = [
    "<connection#|userId> - unbans a player that was previously banned.",
    "Where:",
    "  <connection#|userId> - is the connection number of the user to",
    "       unban or their user ID."
    ];
adminCommand("unban", help) { shell, args ->
    def account = getAccount(args);
    account.setProperty("banned", false);
    account.setProperty("bannedUpdateTime", System.currentTimeMillis());
    account.save();
    shell.echo("Unbanned:" + account.getUserId());    
}

help = [
    "- Shows network stats about the connected players"
]
adminCommand("stats", help) { args ->
    if( server.server.getConnections().isEmpty() ) {
        echo("No connections")
        return;
    } 
    def host = server.server.getServices().getService(com.simsilica.ethereal.EtherealHost.class);
    for( def conn : server.server.getConnections() ) {
        echo("Client[" + conn.getId() + "] address:" + conn.getAddress());
        def listener = host.getStateListener(conn);
        if( listener == null ) {
            echo("    [" + conn.getId() + "] No stats");
            continue;
        }
        echo("    [" + conn.getId() + "] Ping time: " + (listener.getConnectionStats().getAveragePingTime() / 1000000.0) + " ms");
        String miss = String.format("%.02f", listener.getConnectionStats().getAckMissPercent());
        echo("    [" + conn.getId() + "] Ack miss: " + miss + "%");
        echo("    [" + conn.getId() + "] Average msg size: " + listener.getConnectionStats().getAverageMessageSize() + " bytes");
    }
} 


on( playerEntityJoined ) { event ->

    def account = AccountHostedService.getAccount(event.connection);
    if( account ) {
        log.info("Entity joined for account:" + account);
    
        def perms = account.getProperty("permissions", List.class);
        if( perms?.contains("admin") ) {
            log.info("Need to add admin commands to player:" + account.getName());
        
            def shell = GameSessionHostedService.getShell(event.connection);
            server.adminShell.getCommands().entrySet().each {
                log.info("Adding command:" + it);
                shell.addCommand(it.key, it.value);
            }
        }
    }
    
    // And add any standard player server-side commands
    def shell = GameSessionHostedService.getShell(event.connection);

    def cmdHelp = [
            "- Shows stats about this client connection."
        ];
    addShellCommand(shell, "ping", cmdHelp) {
        def conn = event.connection;
        def host = server.server.getServices().getService(com.simsilica.ethereal.EtherealHost.class);
        def listener = host?.getStateListener(conn);
        if( listener == null ) {
            echo("No stats");
            return;
        }
        echo("Client[" + conn.getId() + "] address:" + conn.getAddress());
        echo("    Ping time: " + (listener.getConnectionStats().getAveragePingTime() / 1000000.0) + " ms");
        String miss = String.format("%.02f", listener.getConnectionStats().getAckMissPercent());
        echo("    Ack miss: " + miss + "%");
        echo("    Average msg size: " + listener.getConnectionStats().getAverageMessageSize() + " bytes");
    }
    
}
