/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

import org.slf4j.*;
//import org.apache.log4j.Category;
//import org.apache.log4j.NDC;
// I don't know why those fail.
import com.simsilica.action.*;
import com.simsilica.mod.*;
import com.simsilica.mworld.World;
import com.simsilica.ext.mblock.*;
import mythruna.GameConstants;
import mythruna.sim.Activator;
import mythruna.world.WorldManager;
import mythruna.net.server.GameSessionHostedService.ActivatorImpl;

objectTypes = system(ObjectTypeRegistry.class);

// This lets the "real classes" access the global bindings and stuff.
class GlobalAccess {
    static Object owner;    
}
GlobalAccess.owner = this;


// Resolve properties in a standard way where the property could be
// a closure.
Object resolveProperty( ObjectType objectType, EntityId entity, String name ) {
    Object val = objectType.getTypeVar(name, null);
    //log.info("  found:" + val);
    if( val == null ) {
        // throw new MissingPropertyException(name, getClass());
        // We don't provide callers any other way to find misses and
        // we can't report our custom types with a real groovy exception
        // so best to wait for some use-cases.
        return null;
    }
    if( val instanceof Closure ) {
        // rehydrate(Object delegate, Object owner, Object thisObject)
        Closure c = val.rehydrate(entity, val.owner, val);
        //MDC.pushByKey("stack", objectType.name + "." + name);
        def stack = MDC.get("stack");
        MDC.put("stack", objectType.name + "." + name);
        //NDC.push(objectType.name + "." + name); 
        try {
            return c();
        } finally {
            //MDC.popByKey("stack");
            MDC.put("stack", stack);
            //NDC.pop();
        }
    }
    return val;
} 

// We try to make ActionContext look like EntityId so that
// they are interchangeable in scripts.  So any meta-method we add
// to EntityId should be added here.
ActionContext.metaClass {
    getAt() { Class c ->
        //log.info("ActionContext.getAt(" + c + ") delegate:" + delegate);
        return entityData.getComponent(delegate.getTarget(), c);
    }

    get() { String name ->
        //log.info("ActionContext.get(" + name + ") delegate:" + delegate);
        def objectType = delegate.getType();
        return resolveProperty(objectType, delegate.target, name);
    }
    
    isSameEntity() { Object other ->
        return resolveEntityId(other)?.id == delegate.target.id; 
    }
    
    leftShift() { EntityComponent component ->
        entityData.setComponent(delegate.target, component);
    }

    leftShift() { List components ->
        entityData.setComponents(delegate.target, components as EntityComponent[]);
    }
 
    remove() { Class type ->
        entityData.removeComponent(delegate.target, type);
    }
    
    contains() { Object o ->
        if( log.isDebugEnabled() ) {
            log.debug("ActionContext.contains(" + o + ")");
        }
        delegate.target.contains(o);
    }
    
    run() { ActionEnvironment env, ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return objectTypes.getContext(delegate.target).run(env, *vargs);
    }

    canRun() { ActionEnvironment env, ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("canRun:" + vargs);
        }
        return objectTypes.getContext(delegate.target).canRun(env, *vargs);
    }

    runIfExists() { ActionEnvironment env, ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("runIfExists:" + vargs);
        }
        return objectTypes.getContext(delegate.target).runIfExists(env, *vargs);
    }

    hasAction() { ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("hasAction:" + vargs);
        }
        return objectTypes.getContext(delegate.target).hasAction(*vargs);
    }
}

// If you include this one in the closure then it causes an infinite recursion
// but if I don't then I can't even get it to be called, Object or no.    
//ActionContext.metaClass.isCase = { EntityId o ->
//    log.info("ActionContext.isCase(" + o + ")");
//    return delegate.target.isCase(o);
//}


ActivatorImpl.metaClass {
    getAt() { Class c ->
        return entityData.getComponent(delegate.getId(), c);
    }

    get() { String name ->
        //log.info("ActivatorImpl.get(" + name + ") delegate:" + delegate);
        def objectType = objectTypes.getType(delegate.getId());
        return resolveProperty(objectType, delegate.getId(), name);
    }
    
    isSameEntity() { Object other ->
        return resolveEntityId(other)?.id == delegate.id.id; 
    }
    
    leftShift() { EntityComponent component ->
        entityData.setComponent(delegate.id, component);
    }

    leftShift() { List components ->
        entityData.setComponents(delegate.id, components as EntityComponent[]);
    }

    remove() { Class type ->
        entityData.removeComponent(delegate.id, type);
    }
    
    contains() { Object o ->
        if( log.isDebugEnabled() ) {
            log.debug("ActivatorImpl.contains(" + o + ")");
        }
        delegate.id.contains(o);
    }
}

EntityId.metaClass {    
    getAt() { Class c ->
        return entityData.getComponent(delegate, c);
    }
    
    get() { String name ->
        //log.info("EntityId.get(" + name + ") delegate:" + delegate);
        // Look up the object type
        def objectType = objectTypes.getType(delegate);
        return resolveProperty(objectType, delegate, name);
    }

    isSameEntity() { Object other ->
        return resolveEntityId(other)?.id == delegate.id; 
    }
    
    leftShift() { EntityComponent component ->
        entityData.setComponent(delegate, component);
    }

    leftShift() { List components ->
        entityData.setComponents(delegate, components as EntityComponent[]);
    }
    
    remove() { Class type ->
        entityData.removeComponent(delegate, type);
    }
    
    contains() { Object o ->
        log.info("EntityId.contains(" + o + ")");
        def other = resolveEntityId(o);
        def contained = other[ContainedIn];
        log.info("Other contained in:" + contained);
        if( contained == null ) {
            log.info("No container");
            // It's not contained in anything
            return false;
        }
        // Are we its root?
        if( delegate == contained.root ) {
            log.info("We are the root");
            // We are their root so contain them by definition
            return true;
        }
        
        // Check the hiearchy starting here
        while( contained != null ) {
            log.info("Checking:" + contained.container);
            // Are we the container?
            if( delegate == contained.container ) {
                return true;
            }
            // Try the parent
            contained = contained.container[ContainedIn];
        }                
        return false;  
    }
    
    getType() { ->
        return objectTypes.getType(delegate);
    }
    
    getContext() { ->
        return objectTypes.getContext(delegate);
    }        

    run() { ActionEnvironment env, ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return objectTypes.getContext(delegate).run(env, *vargs);
    }
    
    canRun() { ActionEnvironment env, ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return objectTypes.getContext(delegate).canRun(env, *vargs);
    }
    
    runIfExists() { ActionEnvironment env, ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return objectTypes.getContext(delegate).runIfExists(env, *vargs);
    }
    
    hasAction() { ...vargs ->
        if( log.isDebugEnabled() ) {
            log.debug("hasAction:" + vargs);
        }
        return objectTypes.getContext(delegate).hasAction(*vargs);
    }
}
//EntityId.metaClass.isCase = { Object o ->
//    log.info("EntityId.isCase(" + o + ")");
//    return false;
//}

ObjectType.metaClass.get = { String name ->
    if( log.isDebugEnabled() ) {
        log.debug("ObjectType.get(" + name + ")");
    }
    return delegate.getTypeVar(name, null);
}

ObjectName.metaClass.static.create = { String name ->
    return ObjectName.create(name, entityData);
}

// Stuff like this should probably be a separate groovy file
Holding.metaClass.static.create << { EntityId id, String manipulator ->
    Holding.create(id, manipulator, entityData);
}
Holding.metaClass.static.create << { Activator activator, String manipulator ->
    Holding.create(activator.id, manipulator, entityData);
}
Holding.metaClass.static.create << { ActionContext context, String manipulator ->
    Holding.create(context.target, manipulator, entityData);
}

AttachedTo.metaClass.static.create << { EntityId target, String attachPoint, Vec3d offset, Quatd rotation ->
    AttachedTo.create(target, attachPoint, offset, rotation, entityData); 
}

ObjectName.metaClass.static.create << { String name ->
    ObjectName.create(name, entityData);
}

SpawnPosition.metaClass.constructor << { Vec3d loc, Quatd rotation ->
    return new SpawnPosition(GameConstants.PHYSICS_GRID, loc, rotation);
}

ShapeInfo.metaClass.getShapeName << { ->
    return delegate.getShapeName(entityData);
}

ShapeInfo.metaClass.static.create = { name, scale ->
    ShapeInfo.create(name, scale, entityData);
}

ShapeInfo.metaClass.loadCells << { ->
    return loadCellsForName(delegate.getShapeName(entityData));
}

ShapeInfo.metaClass.getVolume << { ->
    def size = loadCellsForName(delegate.getShapeName(entityData)).size.toVec3d();
    size *= delegate.scale;  
    return size;
}

MapMarker.metaClass.static.create = { String type, int size, Vec3d pos ->    
    return MapMarker.create(type, size, pos, entityData);
}

MapMarker.metaClass.static.create = { EntityId owner, String type, int size, Vec3d pos ->    
    return MapMarker.create(owner, type, size, pos, entityData);
}

ObjectTypeInfo.metaClass.static.create = { String typeName ->
    return ObjectTypeInfo.create(typeName, entityData);
}

EntityLink.metaClass.static.create = { def source, def target, String typeName ->
    return EntityLink.create(resolveEntityId(source), resolveEntityId(target), typeName, entityData);
}

/**
 *  Class to better control what is "in scope" for the action
 *  closures since trying to let groovy sort out our delegation
 *  policy with respect to naked properties, getters, global bindings,
 *  etc. gets messy. 
 */
class ActionDelegate {
    static Logger log = LoggerFactory.getLogger(ActionDelegate.class);

    private ObjectAction action;
    private ActionEnvironment env;
    private ActionContext context;

    public boolean run( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return context.run(env, *vargs);
    }

    public boolean runIfExists( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("run:" + vargs);
        }
        return context.runIfExists(env, *vargs);
    }

    public boolean canRun( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("canRun:" + vargs);
        }
        return context.canRun(env, *vargs);
    }

    public boolean hasAction( ...vargs ) {
        if( log.isDebugEnabled() ) {
            log.debug("hasAction:" + vargs);
        }
        return context.hasAction(*vargs); 
    }

    public ObjectType type( String name ) {
        return GlobalAccess.owner.objectTypes.getType(name);
    }
    
    public ActionEnvironment getEnv() {
        return env;
    }
    
    public ActionEnvironment getSession() {
        return env; 
    }
    
    public ActionContext getContext() {
        return context;
    }

    public Object getActivator() {
        return env.getActivator();
    }

    public long getId() {
        return context.getTarget().getId();
    }

    public EntityId getTarget() {
        return context.getTarget();
    }

    public ActionContext getObject() {
        return context;
    }
    
    public void echo( String s ) {
        env.showPrompt(context.target, PromptType.Message, s);
    }    
}

//class ActionOwner {
//    public Object get( String name ) {
//        log.info("Want to get:" + name);
//        Object result = GlobalAccess.owner.modManager.getGlobalBinding(name);
//        log.info("  global:" + result);
//        if( result != null  ) {
//            return result;
//        }
//        result = context.get(name);
//        log.info("  context:" + result);
//        if( result != null ) {
//            return result;
//        }        
//        //result = env.getProperty(name);
//        //log.info("  env:" + result);
//        //if( result != null ) {
//        //    return result;
//        //}        
//        // If we don't throw a missing property exception then
//        // the property resolution will think that we have this property
//        // and stop.  Since we are the owner that means it won't go
//        // to the delegate and we end up hiding environment properties
//        // like activator.
//        throw new MissingPropertyException(name, getClass());
//    }
//}

class ClosureAction extends AbstractObjectAction {
    static Logger log = LoggerFactory.getLogger(ClosureAction.class);
    Closure run;
    Closure canRun;
 
    public ClosureAction( String name, Class[] types ) {
        super(name, types);
    }
    
    @Override
    public boolean canRun( ActionEnvironment env, ActionContext context, Object... args ) {
        if( run == null ) {
            return false;
        }
        if( canRun == null ) {
            // No special condition
            return true;
        }
        def actionDelegate = new ActionDelegate(action:this, env:env, context:context);
        Closure c = canRun.rehydrate(actionDelegate, canRun.owner, canRun);
 
        //MDC.pushByKey("stack", context.type.name + "." + name);
        def stack = MDC.get("stack");
        MDC.put("stack", context.type.name + "." + name);
        //NDC.push(context.type.name + "." + name); 
        try {           
            Object result;
            if( args.size() != 0 ) {
                result = c(*args);
            } else {
                result = c();
            }
            // Not as dumb as it looks because 'result' can be all kinds of
            // things and groovy's if() handles more than just primitive boolean.
            if( result ) {
                //log.info("Result is true:" + result);
                return true;
            } else {
                //log.info("Result is NOT true:" + result);
                return false;
            }
            return true;
        } finally {
            //MDC.popByKey("stack");
            MDC.put("stack", stack);
            //NDC.pop();
        }
    }
    
    @Override
    public boolean run( ActionEnvironment env, ActionContext context, Object... args ) {
        //log.info("ClosureAction.run(" + env + ", " + context + ", " + args + ")");        
        if( run == null ) {
            return false;
        }
        def actionDelegate = new ActionDelegate(action:this, env:env, context:context);
        Closure c = run.rehydrate(actionDelegate, run.owner, run);
 
        //MDC.pushByKey("stack", context.type.name + "." + name);
        def stack = MDC.get("stack");
        MDC.put("stack", context.type.name + "." + name);
        //NDC.push(context.type.name + "." + name); 
        try {           
            Object result;
            if( args.size() != 0 ) {
                result = c(*args);
            } else {
                result = c();
            }
            // Not as dumb as it looks because 'result' can be all kinds of
            // things and groovy's if() handles more than just primitive boolean.
            if( result ) {
                //log.info("Result is true:" + result);
                return true;
            } else {
                //log.info("Result is NOT true:" + result);
                return false;
            }
            return true;
        } finally {
            //MDC.popByKey("stack");
            MDC.put("stack", stack);
            //NDC.pop();
        }
    }
    
    public ClosureAction onlyIf( Closure canRun ) {
        this.canRun = canRun;
        return this;
    }
}


class ObjectTypeWrapper {
    //static Logger log = LoggerFactory.getLogger(ObjectTypeWrapper.class);
    // This will hide the global 'log' from added actions and getters when
    // "with" is used because this class will be the owner.
    // If we want logging here then we should name it something else or
    // adjust the owner of the closure.  Future bridge to cross.
    
    def type;
    
    public ClosureAction addAction( String name, Closure exec ) {
        ClosureAction a = new ClosureAction(name, exec.getParameterTypes());
        a.run = exec;
        type.addAction(a);
        return a;
    }
    
    public void addGetter( String name, Closure exec ) {
        // Straight forward
        type.setTypeVar(name, exec);
    }
    
    public void setTypeVar( String name, Object value ) {
        type.setTypeVar(name, value);
    }

    public void setTypeVar( String name, String... lines ) {
        type.setTypeVar(name, lines.join("\n"));
    }

    public Object getTypeVar( String name ) {
        return getTypeVar(name, null);
    }
    
    public Object getTypeVar( String name, Object defaultValue ) {
        return type.getTypeVar(name, defaultValue);
    }
 
    public void supertypes( String... types ) {
        def objectTypes = GlobalAccess.owner.objectTypes;
        for( String t : types ) {
            if( !objectTypes.exists(t) ) {
                throw new IllegalArgumentException("Unknown type:" + t);
            }
            ObjectType parent = objectTypes.getType(t);
            type.addSuperType(parent);            
        }
    }
}

// We so it this way so that it's easier to access entityData and other
// globals
ObjectTypeWrapper.metaClass.getInfo = { ->
    return ObjectTypeInfo.create(delegate.type.name, entityData);
}

// Note: ObjectTypeWrapper is part of the delegate chain for executed actions
// and so anything we add to the metaClass will show up as a regular method.
// In this case, this closure was formerly called "createInstance" and was
// thus overriding any other regular createInstance() and not just type.createInstance().
// Problem showed when leaving off ObjectTypeInfo in createInstance() and suddenly
// seeing all kinds of ObjectTools around the world. 
// This was added only 2 weeks ago so I don't think it's poisoned very many
// things in my development worlds.
// ObjectTypeWrapper is in scope because we want it to be for a lot of normal
// "just outside" action closures stuff and I don't think we can easily truncate it
// without losing the rest of the delegation chain.  So we'll just have to be careful
// with the type-specific methods we add. 
ObjectTypeWrapper.metaClass.newInstance = { EntityComponent... init ->
    def result = entityData.createEntity();
    entityData.setComponents(result, delegate.info);
    entityData.setComponents(result, init);
    return result;
}

type = { String name ->
    if( !objectTypes.exists(name) ) {
        throw new IllegalArgumentException("Unknown type:" + name);
    }
    return new ObjectTypeWrapper(type:objectTypes.getType(name));
}

createType = { String name ->
    if( objectTypes.exists(name) ) {
        throw new IllegalArgumentException("Type already exists:" + name);
    }
    // Else create an empty type in lieu of a builder pattern
    def type = new ObjectType(name);
    objectTypes.registerType(type);
    return new ObjectTypeWrapper(type:type);
}


