/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

log.info("Default type:" + type("Default"))

import com.simsilica.crig.es.*;

// 1 block in item scale is 3.125 cm.
standardItemScale = 0.25 * 0.25 * 0.5;

createType("BaseObject").with {

    addGetter("name") { ->
        //log.info("target:" + target);
        //log.info("delegate:" + delegate);
        def on = delegate[ObjectName];
        //log.info("name test:" + on);
        //log.info("delegate test:" + delegate[ObjectName]);
        if( on != null ) {
            return on.getName(entityData);
        }
        def n = delegate[Name];
        if( n != null ) {
            return n.getName();
        }
        return "Entity@" + delegate?.getId();
    }

    addAction("Look") { ->
        def lines = [];

        def name = object.name;
        lines += "Examine: " + name;

        def creator = object[CreatedBy]?.creatorId?.name?:"Unknown";

        def desc = object.type.getTypeVar("description", null);
        if( desc != null ) {
            // Bound to be a better groovy way to do this but so far
            // it only gets more complicated the deeper I drill.
            def s = desc.replaceAll("%creator", creator);
            s = s.replaceAll("%id", id as String);
            s = s.replaceAll("%name", name);
            lines += s;
        } else {
            lines += "ID:" + id;
            if( creator != null ) {
                lines += "Created by:" + creator;
            }
        }

        //lines += "Volume:" + object[ObjectVolume];

        session.showPrompt(target, PromptType.Dialog, lines.join("\n"));
        session.showOptions(target, []);

        return true;
    }

    addAction("move") { loc, facing ->
        //log.info("move:" + loc + ", " + facing + "  by:" + activator);
        // By default, we'll just let everything go wherever
        object << new SpawnPosition(loc, facing);
    }
}

// An object that can defer to the default hand-based manpulation
// drag, drop, activate, etc..
createType("ObjectManipulator").with {
    supertypes("BaseObject");

    // For now, we'll use the creator manipulator type but probably
    // there should be a different one that the creator manipulator extends.
    setTypeVar("manipulator", "Hand");
    
    setTypeVar("moveLimit", 0.75);

    addAction("dragStart") { dragged, loc, facing, click ->
        log.info("dragStart:" + dragged + ", " + loc + ", " + facing + ", " + click);

        env.setVar("draggedObject", dragged);

        // See if we are allowed to move this object
        def objPos = dragged[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canMoveObject() ) {
            env.setVar("objectMoveAllowed", false);
            echo("You do not have permission to move objects here.");
            return;
        }
        env.setVar("objectMoveAllowed", true);

        def offset;
        def shape = dragged[ShapeInfo];
        if( shape != null ) {
            def cells = shape.loadCells();
            if( cells ) {
                def size = cells.size;
                offset = size * 0.5 * shape.scale;
                offset.y = size.y * shape.scale; // to spare us a separate lookup later
            } else {
                offset = new Vec3d();
                offset.y = 1.0;
            }
        } else {
            // FIXME: lookup the alternate shape.
            // We don't use the offset for very much so right now in the the
            // limited use-cases where we don't have a shape (ClaimMarkers)
            // 0 is ok.
            offset = new Vec3d();
            offset.y = 1.0;
        }
        env.setVar("draggedOffset", offset);
    }

    addAction("drag") { dragged, loc, facing, click ->
        //log.info("drag:" + dragged + ", " + loc + ", " + facing + ", " + click);

        if( !env.getVar("objectMoveAllowed", false) ) {
            // Drag all you want, nothing is going to happen
            return;
        }

        // Figure out the proper y-up facing
        def xAxis = facing.mult(new Vec3d(1, 0, 0));
        double rads = Math.atan2(-xAxis.z, xAxis.x);
        facing = new Quatd().fromAngles(0, rads, 0);

        // Figure out where the bottom center is so that
        // we can project down for ground
        def center = loc;
        def offset = env.getVar("draggedOffset", null);
        def yOffset = 1.0; 
        if( offset ) {
            yOffset = offset.y;
            def localOffset = offset.clone();
            localOffset.y = 0;            
            facing.mult(localOffset, localOffset);
            center += localOffset;
        }
        
        // If the center is already in a block then don't let the movement happen
        // or at least try a higher pick offset
        // FIXME: make these physics queries instead
        int cellType = MaskUtils.getType(world.getWorldCell(center));
        if( cellType != 0 ) {
            // Give us a little more space
            yOffset += 1;
        }

        //log.info("loc:" + loc + "  center:" + center + "  yOffset:" + yOffset);
        // Check the ground
        def ray = new Rayd(center.add(0, yOffset, 0), new Vec3d(0, -1, 0));
        def hits = activator.pick(ray, 1);
        if( hits.hasNext() ) {
            def contact = hits.next(); 
            def hit = contact.point.y;
            //log.info("  hit:" + hit + "  contact:" + contact);            
            if( loc.y < hit ) {
                center.y += hit - loc.y;
                loc.y = hit;
            }
        }

        // FIXME: make these physics queries instead
        cellType = MaskUtils.getType(world.getWorldCell(center));       
        if( cellType != 0 ) {
            return;
        }
        
        // Now that we know where we'll put it, we need to check to see
        // if it was dragged into an area that isn't allowed.
        if( !getPerms(activator, loc).canMoveObject() ) {
            echo("You do not have permission to move objects into that area.");
            return;
        }

        def objPos = dragged[SpawnPosition]?.location;
        if( objPos ) {
            double moveLimit = dragged.type.getTypeVar("moveLimit", 0.5); 
            // See if we're dragging too far
            if( loc.distance(objPos) > moveLimit ) {
                log.info("Uh oh.  Trying to move too much, from:" + objPos + " to:" + loc
                          + "Max dist:" + moveLimit + " actual:" + loc.distance(objPos));
                return; 
            }
        }
        
        // Base object has a move... so everything should have a move
        // that we will be trying to drag.
        dragged.run(env, "move", loc, facing);
        //dragged << new SpawnPosition(loc, facing);
    }

    addAction("dragEnd") { dragged ->
        log.info("dragEnd:" + dragged);

        // just clear the working env vars
        env.setVar("draggedObject", null);
        env.setVar("draggedOffset", null);
        env.setVar("objectMoveAllowed", null);
    }

    addAction("contextAction") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;

        log.info("contextAction:" + clicked + ", " + pos + " name:" + clicked?.name);

        // Here we want to let the player run some actions on the object

        // Assemble the action list
        def options = [];
        def clickedContext = clicked.context;
        log.info("Checking actions for type:" + clicked.type);        
        for( def action : clicked.type.contextActions ) {
            if( !action.canRun(env, clickedContext) ) {
                log.info("Skipping:" + action);
                continue;
            }
            options.add(new Option(action.getName()));
        }
        session.showPrompt(clicked, PromptType.List, clicked.name);
        session.showOptions(clicked, options); 
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);
    }

    addAction("altPressBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("altPressBlock:" + loc);
    }

    addAction("altClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("altClickBlock:" + loc);
    }

    addAction("mainPressBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("mainPressBlock:" + loc);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("mainClickBlock:" + loc);
    }

    addAction("mainPressObject") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;
        log.info("mainPressObject:" + clicked + "  pos:" + pos);
    }

    addAction("mainClickObject") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;
        log.info("mainClickObject:" + clicked + "  pos:" + pos);
    }

}

// For things that can be in inventory and equipped and stuff
createType("BaseItem").with {

    // Let all equippable items default to the hand manipulator with
    // default actions.  This will automatically let the user continue
    // to perform standard manipulation actions if the object doesn't
    // define its own.
    supertypes("ObjectManipulator");

    // For now, all objects will be equippable if they are in inventory
    addAction("Equip") { ->
        // This should be in the "can run" check
        // ...but we still need to check here as canRun() is speculative and "Equip"
        // may be called directly outside of those normal checks.
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            log.error("Activator:" + activator + " tried to equip foreign object:" + object);
            return;
        }

        // Right now there is only one hand "Holder"
        activator << Holding.create(object, object.type.manipulator?:"Default")
 
        // If it has an onEquip action then run that
        if( canRun("onEquip") ) {
            run("onEquip");
        }
        
    }.onlyIf { ->
        // Conditions for equipping:
        // -item is contained by the thing running the action
        // -item is not already equipped
        
        // Do the faster O(1) check first
        def holding = activator[Holding];
        if( object.isSameEntity(holding?.target) ) {
            return false;        
        }
         
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            return false;
        }
        
        return true;
    }

    addAction("Unequip") { ->
        // This should be in the "can run" check
        if( !activator.contains(object) ) {
            // We can't equip things we aren't carrying
            log.error("Activator:" + activator + " tried to unequip foreign object:" + object);
            return;
        }

        // See if the current holder is holding us or not
        def holding = activator[Holding];
        log.info("Unequip holding:" + holding);
        if( !object.isSameEntity(holding?.target) ) {
            log.warn("Activator:" + activator + " trying to unequip item that is not equipped:" + object);
        }

        log.info("Unequpping:" + object);
        activator.remove(Holding);
        
        // If it has an onUnequip action then run that
        if( canRun("onUnequip") ) {
            run("onUnequip");
        }
    }.onlyIf { ->
        // Conditions for unequipping:
        // -item is equipped already
        // ...that's it.
        def holding = activator[Holding];
        return object.isSameEntity(holding?.target);        
    }
}

type("Default").with {
    // By default, objects will have manipulator actions but not be equippable.
    // This is mostly because we let the empty hand be its own target when
    // it is not holding things... and equipping/unequipping a hand doesn't
    // seem logical.
    supertypes("ObjectManipulator");
}

createType("BuildWand").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
              "The Build Wand is used for placing and removing blocks.",
              "",
              "The right mouse button will place the current block type",
              "and the left mouse button will remove the block under the",
              "cursor (highlighted yellow).",
              "",
              "Press 'e' to open the block type selector menu.",
              "Press 'c' to select the block under the cursor as the",
              "current type.",
              "",
              "The selected block will appear in the bottom center of the",
              "screen.  If the block supports multiple rotations then the",
              "mouse wheel will change the block rotation."
            );
    setTypeVar("handleLocation", new Vec3d(1.5, 1.5, 3));

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/wand1.blocks", standardItemScale * 0.5)); 
    setTypeVar("volume", new ObjectVolume(new Vec3d(3, 3, 13).multLocal(standardItemScale * 0.5)));

    addAction("startDigging") { Vec3i loc ->
        log.info("" + context + ".startDigging(" + loc + ")");
    }

    addAction("stopDigging") { Vec3i loc ->
        log.info("" + context + ".stopDigging(" + loc + ")");

        // Perform validation to make sure the player isn't cheating.
        // FIXME: add distance check
        // FIXME: add line of sight check

        // Are they allowed to edit here?
        if( !getPerms(activator, loc).canRemoveBlock() ) {
            echo("You do not have permission to remove blocks here.");
            return;
        }
        world.setWorldCell(loc.toVec3d(), 0);

        // 2022-11-09 - Puting this note here but it applies to the
        // whole digger tool.
        //
        // I originally thought to limit the client's ability to specify
        // specific blocks/types and resolve all of that on the server.
        // That way there would be no possibility for "cheating" or hacked
        // clients poking data all over the world as they see fit.
        // But, there is the possibility that they've turned a little before
        // their click is processed... and since that information comes in as UDP
        // there is a good chance we then point in the wrong place here.
        // This is especially apparent when the client starts predicting their
        // own block edits to avoid the round-trip delay.  It happens quite often
        // that the client thinks it placed a block in one place but the server
        // thinks another.
        //
        // So we fall back.  Now the client will tell us where it clicked
        // and what type was selected.  We have to do extra work to see if
        // they are allowed to do that or are cheating.
        //
        // Also, thinking about this today, maybe we want to allow certain
        // users to cheat by adding some perms... we can do that now.
        //
        //def hits = activator.pick();
        //
        //if( hits.hasNext() ) {
        //    def intersect = hits.next();
        //    if( !getPerms(activator, intersect.getBlock()).canRemoveBlock() ) {
        //        echo("You do not have permission to remove blocks here.");
        //        return;
        //    }
        //    world.setWorldCell(intersect.getBlock().toVec3d(), 0);
        //}
    }

    addAction("startFilling") { Vec3i loc, Integer selected ->
        log.info("" + context + ".startFilling()");
    }

    addAction("stopFilling") { Vec3i loc, Integer selected ->
        log.info("" + context + ".stopFilling()");

        // Perform validation to make sure the player isn't cheating.
        // FIXME: add distance check
        // FIXME: add line of sight check

        // Are they allowed to edit here?
        if( !getPerms(activator, loc).canAddBlock() ) {
            echo("You do not have permission to place blocks here.");
            return;
        }
        log.info("claim:" + getClaim(activator, loc));
        world.setWorldCell(loc.toVec3d(), selected);

        //def selected = env.getVar("selectedType", 1);
        //def hits = activator.pick();
        //if( hits.hasNext() ) {
        //    def intersect = hits.next();
        //    def pos = intersect.getBlock().toVec3d();
        //    pos.addLocal(intersect.getNormal());
        //    if( !getPerms(activator, pos).canAddBlock() ) {
        //        echo("You do not have permission to place blocks here.");
        //        return;
        //    }
        //    log.info("claim:" + getClaim(activator, pos));
        //    world.setWorldCell(pos, selected);
        //}
    }

    addAction("selectType") { Integer index ->
        log.info("" + context + ".selectType(" + index + ")");

        env.setVar("selectedType", index);
        context << ObjectFocus.blockType(index);
    }

    addAction("rotate") { Integer index ->
        log.info("" + context + ".rotate(" + index + ")");
        // Handled by the client right now
    }
}

createType("ObjectTool").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "The Object Tool is used for placing and updating objects.",
               "",
               "Clicking the right mouse button in an open location will",
               "place the current object type centered over that location.",
               "",
               "The mouse wheel will switch the object type.  The current",
               "object type is displayed in the bottom middle of the screen.",
               "",
               "Clicking the right mouse button on an existing object will",
               "open its action menu.",
               "",
               "Press the right mouse button down on an object and drag to",
               "move it.",
               "While dragging an object, the mouse wheel will rotate the",
               "object.",
               "",
               "New object types can be created in the blueprint editor",
               "accessible from the player menu (tab).",
             );

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/etcher.blocks", standardItemScale * 0.5)); 
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));
             

    addAction("onEquip") { ->
        log.info("Subclass equip");

        // Need to run the real equip
        run(type("BaseItem"), "Equip");

        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);

        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }

        // See if the one we have is valid
        def focus = target[ObjectFocus]?.entityId;
        if( focus && blueprints.contains(focus) ) {
            // We're done... the current focus is valid.
        } else {
            // Just use the first item
            focus = blueprints.find();
            target << ObjectFocus.entity(focus);
        }

        env.setVar("selectedBlueprint", focus);

        return true;
    }
    
    addAction("place") { blockHit ->
        Vec3d loc = blockHit.location;
        log.info("place:" + loc);

        if( !getPerms(activator, loc).canAddObject() ) {
            echo("You do not have permission to place objects here.");
            return;
        }

        def bp = env.getVar("selectedBlueprint", null);
        if( bp == null ) {
            // Not our job to set it
            return false;
        }
        log.info("" + activator + " creating object at:" + loc + " with blueprint:" + bp);

        def shape = bp[ShapeInfo];

        // Need to get the size so we can find its base's center.
        // We probably should keep this as part of the object type or even a separate
        // component.
        def cells = shape.loadCells();
        def size = cells.size;
        def offset = size * 0.5 * shape.scale;
        offset.y = 0;
        log.info("cells:" + cells + "  offset:" + offset);

        loc -= offset;

        // FIXME: make a createObject() method that encapsulates some of
        // this, calls a default object init action, etc..
        def pos = new SpawnPosition(loc, new Quatd());
        def name = bp[ObjectName];
        def placed = createEntity(
                pos, shape, name,
                new BlueprintSource(bp),
                new CreatedBy(activator.id)
            );

        return true;
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);

        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);

        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;
        if( blueprints.isEmpty() ) {
            target.remove(ObjectFocus);
            env.setVar("selectedBlueprint", null);
            return true;
        }

        // See what the current index is
        def selected = env.getVar("selectedBlueprint", null);
        int index = blueprints.findIndexOf { it == selected };
        index = (index + delta) % blueprints.size();
        if( index < 0 ) {
            index += blueprints.size();
        }
        selected = blueprints[index];
        target << ObjectFocus.entity(selected);
        env.setVar("selectedBlueprint", selected);
    }

    addAction("contextAction") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;

        log.info("contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";

        // Here we want to let the player run some actions on the object

        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        def owner = clicked[OwnedBy]?.owner;
        log.info("Player:" + activator.id + " creator:" + creator + " owner:" + owner);
        boolean isOurs = activator.isSameEntity(creator) || activator.isSameEntity(owner);
        boolean canRemove = isOurs || (creator == null && owner == null); 

        // Assemble the action list
        def options = [];
        def clickedContext = clicked.context;
        log.info("Checking actions for type:" + clicked.type);        
        for( def action : clicked.type.contextActions ) {
            if( !action.canRun(env, clickedContext) ) {
                log.info("Skipping:" + action);
                continue;
            }
            if( "Take".equals(action.getName()) ) {
                // We'll assume that if it can be 'taken' then deleting it
                // would be destructive.
                canRemove = false;
            }
            options.add(new Option(action.getName()));
        }

        // Can't copy the blueprint unless it has a shape
        boolean hasShape = clicked[ShapeInfo] != null;
        if( hasShape ) {
            // For the blueprint related actions, we are the thing upon
            // which the action is run and the object is the parameter.
            options.add(new Option(target, "copyBlueprint", clicked));
        }

        // Instead of hasShape... we should probably base this on
        // the actions that the object has.  That way we can have non-removable
        // objects that have shapes.  And then move "deleteObject" to the objects
        // that support delete... the others can echo a message that they can't
        // be removed.
        def objPos = clicked[SpawnPosition]?.location;
        if( canRemove && hasShape && getPerms(activator, objPos).canRemoveObject() ) {
            // Options to call us back as the manipulator
            options.add(new Option(target, "deleteObject", clicked));
        }

        // If it's our own object then allow us to make it solid or not
        // or to make it a physics object.
        if( isOurs && canRemove ) { //activator.isSameEntity(creator) ) {
            Mass mass = clicked[Mass];
            if( mass == null ) {
                options.add(new Option(target, "makeSolid", clicked));
                options.add(new Option(target, "makePhysical", clicked));
            } else if( mass.getMass() == 0 ) {
                options.add(new Option(target, "makeIllusion", clicked));
                options.add(new Option(target, "makePhysical", clicked));
            } else {
                options.add(new Option(target, "makeIllusion", clicked));
                options.add(new Option(target, "makeStatic", clicked));
            }
        }

        session.showPrompt(clicked, PromptType.List, name);
        session.showOptions(clicked, options);
    }

    // Targetted context operations
    //-----------------------------------------------
    addAction("copyBlueprint") { clicked ->
        log.info("copyBlueprint:" + clicked);

        def bpSource = clicked[BlueprintSource];
        def name = clicked.name;
        if( name == null ) {
            if( bpSource ) {
                name = bpSource.blueprintId.name?:"Unknown";
            } else {
                name = "Unknown";
            }
        }

        def shape = clicked[ShapeInfo];
        def cells = shape.loadCells();

        // Load the cells just so we can get a proper offset
        def offset = cells.size * 0.5;
        offset.x = 5 - offset.x;
        offset.y = 0;
        offset.z = 5 - offset.z;
        offset = offset.floor(); // back to vec3i

        log.info("name:" + name + "  shape:" + shape + "  offset:" + offset);

        if( bpSource ) {
            createEntity(
                        new BlueprintInfo(activator.id, name + " Copy", offset),
                        ObjectName.create(name),
                        shape,
                        // We'll make this player the creator but also
                        // set the blueprint source so we can still track the
                        // original author.  This player is likely to make changes
                        // the blueprint and we won't change the creator when that
                        // happens... so we need to change it now.
                        new CreatedBy(activator.id),
                        bpSource
                        );
        } else {
            // Assume we have to create the blueprint manually
            createBlueprintFromCells(activator.id, name + " Copy", name, cells, false, shape.scale);
        }
    }

    addAction("deleteObject") { clicked ->
        log.info("deleteObject:" + clicked);

        def objPos = clicked[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canRemoveObject() ) {
            echo("You do not have permission to remove objects here.");
            return;
        }
        // We should probably check to see if we can in case we hacked
        // in around the context menu
        removeEntity(clicked);
    }

    addAction("makeSolid") { clicked ->
        clicked << new Mass(0);
    }

    addAction("makePhysical") { clicked ->
        // Calculate the mass of the object

        // Set the mass
        clicked << new Mass(50);
    }

    addAction("makeIllusion") { clicked ->
        clicked.remove(Mass);
    }

    addAction("makeStatic") { clicked ->
        clicked << new Mass(0);
    }

}

createType("Backpack").with {
    // Right now, just extend base object since we cannot pickup/put down
    // a backpack yet and therefore there is also no equip/unequip... but
    // we might want to look at it or get its name.
    supertypes("BaseObject");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/backpack2.blocks", standardItemScale)); 
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    // Backpack requires some custom fixup
    def backpackScale = getTypeVar("shape").scale; 
    def size = getTypeVar("shape").loadCells().size.toVec3d();
    size.x = 8;
    size.z = 5;
    size *= backpackScale;
    
    def innerSize = new Vec3d(7, 9, 3.5);
    innerSize *= backpackScale;
    
    def offset = size.subtract(innerSize).mult(0.5);
    offset.y = backpackScale; // hard coded to 1 block thickness for the base
    
    setTypeVar("volume", new ObjectVolume(size));
    setTypeVar("containerVolume", new ContainerVolume(innerSize, offset));

    addAction("drop") { dropped, xSlot, ySlot ->
        log.info("drop:" + dropped + " at:" + xSlot + ", " + ySlot);

        // Find out the root of this container object
        def root = object[ContainedIn].root;
        dropped << new ContainedIn(root, target, xSlot, ySlot);
    }

    addAction("itemActions") { clicked ->
        log.info("itemActions:" + clicked);

        // Confirm that it really is in this container just in case
        if( !object.isSameEntity(clicked[ContainedIn].container) ) {
            log.warn("Running actions from container:" + target + " on object not in the container:" + clicked);
            return false;
        }

        // Assemble the action list
        def options = [];
        def clickedContext = clicked.context;
        for( def action : clicked.type.contextActions ) {
            if( !action.canRun(env, clickedContext) ) {
                log.info("Skipping:" + action);
                continue;
            }
            options.add(new Option(action.getName()));
        }
        session.showPrompt(clicked, PromptType.List, clicked.name);
        session.showOptions(clicked, options);

        return true;
    }
}

import com.simsilica.mworld.tile.Resolution;
import com.simsilica.mworld.tile.tree.*;

findTree = { Vec3i block ->
    def tileId = TileId.fromWorld(block);
    def treeTile = world.getTrees(tileId, null);
    def tileLoc = block - tileId.getWorld(null);
    //treeTile.getTrees(block).each {
    //}
    return treeTile.getTrees(block).find { tree -> tree.x == tileLoc.x && tree.z == tileLoc.z }
}

createType("Axe").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Swing");

    setTypeVar("description",
        "This was axe was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(6, 0, 5));

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/axe1.blocks", standardItemScale)); 
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    addAction("mainPressBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("start chopping:" + block);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("chop:" + loc);

        int cell = world.getWorldCell(block as Vec3d);
        int type = MaskUtils.getType(cell);
        int sides = MaskUtils.getSideMask(cell);

        def genType = CellGenType.getType(cell);

        echo("Chop:" + genType);

        def found = findTree(block);
        echo("Tree:" + found);

        // The below would be nearly 100% accurate but would require
        // straddling columns to get the full tree and requires shuffling
        // lots of additional leaf data around just to pull out a single
        // tree.  Instead, we will let TreeType guess if block type is
        // part of it or not.
        //def colId = ColumnId.fromWorld(block);
        //def colData = new ColumnData(colId, GameConstants.MAX_BUILD_HEIGHT / 32);
        //
        //def tileId = TileId.fromWorld(block);
        //def treeTile = world.getTrees(tileId, null);
        //
        //def localTrees = treeTile.getTrees(colId);
        //if( localTrees.isEmpty() ) {
        //    return false;
        //}
        //
        //Random rand = new Random(columnId.getId()); // consistent random per column
        //def cells = new TileColumnCells(colData);
        //for( Tree tree : localTrees ) {
        //    //log.info("  insert:" + tree + " into:" + col.getColumnId());
        //    if( tree == found ) {
        //        // Clear everything so far
        //        colData.leafs.each { leaf ->
        //            leaf.rawCells.clear();
        //        }
        //    }
        //    tree.type.insertTree(tree, cells, rand);
        //    if( tree == found ) {
        //        break;
        //    }
        //}

        // Guess which blocks are part of this tree
        def tileId = TileId.fromWorld(block);
        def tileLoc = block - tileId.getWorld(null);

        int size = 1 + found.radius * 2;
        int xBase = found.x - found.radius;
        int yBase = found.y;
        int zBase = found.z - found.radius;

        CellArray cells = new CellArray(size, found.height, size);

        Vec3d base = new Vec3d(xBase, yBase, zBase);
        //log.info("base:" + base);
        base = base + tileId.getWorld(null);
        //log.info("world base:"+ base);

        for( int y = 0; y < found.height; y++ ) {
            for( int z = 0; z < size; z++ ) {
                def sb = new StringBuilder();
                for( int x = 0; x < size; x++ ) {
                    cell = world.getWorldCell(base.add(x, y, z));
                    genType = CellGenType.getType(cell);
                    if( genType == CellGenType.Flora ) {
                        type = MaskUtils.getType(cell);
                        if( found.type.isBlock(found, x - found.radius, y, z - found.radius, type) ) {
                            //sb.append("[1]");
                            cells.setCell(x, y, z, type);
                        } else {
                            //sb.append("[0]");
                        }
                    } else {
                        //sb.append("[0]");
                    }
                }
                //log.info("[" + y + "][" + z + "] = " + sb);
            }
        }

        // Make sure the cells have accurate masks
        MaskUtils.calculateSideMasks(cells);

        //createBlueprintFromCells(activator.id, "Tree:" + block.x + "x" + block.z, "Tree", cells, false, 0.25);
        createBlueprintFromCells(activator.id, "Tree:" + block.x + "x" + block.z, "Tree", cells, false, 1);
    }


    addAction("altClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        echo("Loc:" + loc + " block:" + block);

        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);

        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;

        def tree = blueprints.find { it.name.startsWith("Tree") };

        echo("Tree blueprint:" + tree);

        def bp = tree;

        def shape = bp[ShapeInfo];

        // Need to get the size so we can find its base's center.
        // We probably should keep this as part of the object type or even a separate
        // component.
        def cells = shape.loadCells();

        def worldEdits = new WorldEdits();

        for( int x = 0; x < cells.sizeX; x++ ) {
            for( int z = 0; z < cells.sizeZ; z++ ) {
                for( int y = 0; y < cells.sizeY; y++ ) {
                    int val = cells.getCell(x, y, z);
                    world.setWorldCell(loc.add(x, y, z), val, worldEdits);
                }
            }
        }

        world.deliverEvents(worldEdits);

        echo("World implemention:" + world.getClass() + "  thread:" + Thread.currentThread());

        return true;

    }

    addAction("startSwing") { facing ->
        log.info("startSwing:" + facing);
    }

    addAction("swing") { facing ->
        log.info("swing:" + facing);
    }

    addAction("stopSwing") { facing ->
        log.info("stopSwing:" + facing);
    }

}

createType("Sword").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Swing");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/sword1.blocks", standardItemScale)); 
    // The sword requires some custom sizing to make sense in inventory              
    def size = getTypeVar("shape").loadCells().size.toVec3d();
    size.x = 3;
    size *= standardItemScale;                
    setTypeVar("volume", new ObjectVolume(size));

    setTypeVar("description",
        "This %name was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(3, 1, 4));
    setTypeVar("handleRotation", new Quatd().fromAngles(0, 0, -Math.PI * 0.5));

    addAction("startSwing") { facing ->
        log.info("startSwing:" + facing);
    }

    addAction("swing") { facing ->
        log.info("swing:" + facing);
    }

    addAction("stopSwing") { facing ->
        log.info("stopSwing:" + facing);
    }
}

createType("Book").with {
    supertypes("BaseItem");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/book.blocks", standardItemScale)); 
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    setTypeVar("description",
        "This was book was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(0, 4, 1));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, Math.PI, 0));
}


createType("TestObject").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
        "This was book was created by %creator with ID %id.");

    addAction("Test") { ->
        //log.info("" + this + "." + getName() + "(" + Arrays.asList(args) + ")");
        log.info("" + this + "." + "Test" + "(" + ")");
        log.info("owner:" + owner + " delegate:" + delegate);
        log.info("Activator:" + activator);
        log.info("Activator:" + getActivator());
        //log.info("variables:" + getVariables()); // from 'delegate'
        log.info("activator.vars:" + activator.vars); // from 'delegate'
        log.info("target:" + getTarget()); // from 'this'

        // This goes to a global binding
        testProperty = 123;

        //log.info("variables:" + getVariables()); // from 'delegate'

        //activator.target = "Test Target";
        //log.info("activator.vars:" + activator.vars); // from 'delegate'
        //log.info("target:" + getTarget()); // from 'this'?  Yep.

        //log.info("vars:" + activator.vars);
        //log.info("test:" + activator.vars.get("selectedType"));
        //log.info("test1:" + activator.getProperty("selectedType"));
        //log.info("test2:" + activator.selectedType);
        //activator.foo = 123;
        //log.info("test3:" + activator.foo);
        return true;
    }

    addAction("Test2") { String name ->
        log.info("Test2:" + name);
        return true;
    }

    addAction("Test3") { ->
        log.info("Calling delegate version of run()");
        // This one calls it directly on ActionContext which is our 'this'
        object.run(env, "Test2", "This is a test")
        log.info("Calling non-delegate version of run()");
        run("Test2", "This is a different test")
        //run("BaseTest")

        def manipulator = object.getType().getTypeVar("manipulator", "Default");

        log.info("Manipulator:" + manipulator);
        log.info("test:" + (object.manipulator?:"Default"));

        log.info("holding1:" + Holding.create(object.getTarget(), manipulator, entityData));
        log.info("holding2:" + Holding.create(object.getTarget(), manipulator));
        log.info("holding3:" + Holding.create(object, manipulator));
        log.info("holding4:" + Holding.create(activator, manipulator));

        log.info("contains1:" + activator.contains(object));

        // What about running an action from a different type or one of
        // our super types?
        log.info("Running it the hard way--------------------");
        type("BaseObject").findAction("Look").run(env, context);

        log.info("Running it the easy way--------------------");
        run(type("BaseObject"), "Look");

        return true;
    }

    addGetter("test") { -> //target ->
        log.info("getter-test");
        log.info("  getter-ed:" + system(EntityData));
        log.info("  getter-target:" + target);
        log.info("  getter-activator:" + activator);
        log.info("  owner:" + owner);
        log.info("  delegate:" + delegate);
        log.info("  this:" + this);
        return "foo-123"
    }

    //addGetter("name") { ->
    //    log.info("target:" + target);
    //    log.info("delegate:" + delegate);
    //    def on = delegate[ObjectName];
    //    log.info("name test:" + on);
    //    log.info("delegate test:" + delegate[ObjectName]);
    //    if( on != null ) {
    //        return on.getName(entityData);
    //    }
    //    def n = delegate[Name];
    //    if( n != null ) {
    //        return n.getName();
    //    }
    //    return "Entity@" + target.getId();
    //}

}

createType("TestWand").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "A Test Wand."
             );

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/metal-wand.blocks", standardItemScale * 0.5)); 
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        echo("Clicked:" + block)

        //double temperature = worldManager.worldFractal.getTemperature(block.x, block.y, block.z);
        //double weather = worldManager.worldFractal.getWeather(block.x, block.y, block.z);
        //double ecology = worldManager.worldFractal.getEcology(block.x, block.y, block.z);
        //
        //// Adjust temperature for elevation
        ////int seaLevel = 128;
        ////int treeLine = seaLevel + 490;
        ////double height = Math.abs(block.y - seaLevel) / (double)treeLine;
        ////echo("height scale:" + height);
        ////double baselineHeight = 0.2;
        ////double effectiveTemperature = temperature;
        ////if( height > baselineHeight ) {
        ////    double heightEffect = (height - baselineHeight) / (1.0 - baselineHeight);
        ////    double temperatureAdjust = heightEffect * 0.4;
        ////    effectiveTemperature = temperature - temperatureAdjust;
        ////}
        //double effectiveTemperature = worldManager.worldFractal.getAdjustedTemperature(block.x, block.y, block.z);
        //
        //// Adjust precipiation by temperature if we are in particularly
        //// hot climates.  Anything above 40C is desert and will significantly
        //// affect whether precipitation accumulates.
        //// 40C in the range of -20 to 50 should be 0.857
        ////double effectiveWeather = weather;
        ////if( effectiveTemperature > 0.857 ) {
        ////    double weatherAdjust = (effectiveTemperature - 0.857) / (1.0 - 0.857);
        ////    weatherAdjust = 1 - weatherAdjust;
        ////    // Weather adjust should be in 0..1 where 0 is no precip. and
        ////    // 1 is max precip
        ////    effectiveWeather *= weatherAdjust;
        ////}
        //double effectiveWeather = worldManager.worldFractal.getAdjustedWeather(block.x, block.y, block.z, effectiveTemperature);
        //
        ////// Temper ecology by how wet things are on average
        ////double effectiveEcology = ecology;
        ////if( effectiveWeather < 0.25 ) {
        ////    effectiveEcology *= (effectiveWeather / 0.25);
        ////} else if( effectiveWeather > 0.75 ) {
        ////    // Increase ecology, up to double
        ////    effectiveEcology *= 1 + ((effectiveWeather - 0.75)/0.25);
        ////    if( effectiveEcology > 1 ) {
        ////        effectiveEcology = 1;
        ////    }
        ////}
        //double effectiveEcology = worldManager.worldFractal.getAdjustedEcology(block.x, block.y, block.z, effectiveWeather);

        def info = worldManager.worldFractal.getBioInfo(block.x, block.y, block.z, null);

        echo("Temperature:" + info.baseTemperature + "  effective:" + info.temperature);
        echo("Weather:" + info.basePrecipitation + "  effective:" + info.precipitation);
        echo("Climate:" + info.climate + "  Soil Quality:" + info.soilQuality);
        echo("Vegetation:" + info.baseVegetationLevel + "  effective:" + info.vegetationLevel);

        int foliage = (int)Math.round(info.vegetationLevel * 3) & 0x3;
        int wetness = (int)Math.round(info.precipitation * 3) & 0x3;
        echo("foliage level:" + foliage + "/3   wetness level:" + wetness + "/3");

        def tileId = TileId.fromWorld(block.x, block.y, block.z);
        def terrain = world.getTerrainImage(tileId, TerrainImageType.Terrain, null);
        def origin = tileId.getWorld(null);        
        int type = terrain.getType((int)(block.x - origin.x), (int)(block.z - origin.z));
        byte light = terrain.getLight((int)(block.x - origin.x), (int)(block.z - origin.z));
        int baseType = TerrainTypes.getBaseType(type); 
        echo("Encoded type:" + baseType + " foliage level:" + TerrainTypes.getFoliageLevel(type) + "/3 wetness level:" + TerrainTypes.getWetnessLevel(type) + "/3 frozen:" + TerrainTypes.isFrozen(type));

        double c = (info.temperature * 70) - 20;
        double f = c * 1.8 + 32;
        double p = info.precipitation * 100;
        echo(String.format("%.01f C (%.01f F)  Avg. rainfall: %.02f cm (%.02f inches)", c, f, p*2.5, p));
        
        // Show what the tree generator would use for type selection
        int TEMPERATURE_ZONES = 7;
        double VEG1 = 0.2;
        double VEG2 = 0.25;
        double VEG3 = 0.4;
        double VEG4 = 0.5;
        double VEG5 = 0.75;
        double VEG6 = 1;

        double PROB1 = 0.001;
        double PROB2 = 0.003;
        double PROB3 = 0.01;
        double PROB4 = 0.1;
        double PROB5 = 0.75;
        double PROB6 = 1.0;
        int treeLine = 490 + GameConstants.SEA_LEVEL;
        def density = "LOW";
        double chance = 0;
        double veg = 0;
        
        if( baseType == TerrainTypes.TYPE_SAND ) {
            chance = info.vegetationLevel * 0.001;
        } else {
            veg = info.vegetationLevel;
 
            if( veg < VEG1 ) {
                chance = (veg/VEG1);
                chance = chance * PROB1;
                density = "LOW";
            } else if( veg < VEG2 ) { 
                chance = (veg - VEG1)/(VEG2 - VEG1);
                chance = PROB1 + (chance * (PROB2 - PROB1)); 
                density = "LOW";
            } else if( veg < VEG3 ) { 
                chance = (veg - VEG2)/(VEG3 - VEG2);
                chance = PROB2 + (chance * (PROB3 - PROB2)); 
                density = "MEDIUM";
            } else if( veg < VEG4 ) { 
                chance = (veg - VEG3)/(VEG4 - VEG3);
                chance = PROB3 + (chance * (PROB4 - PROB3)); 
                density = "MEDIUM";
            } else if( veg < VEG5 ) { 
                chance = (veg - VEG4)/(VEG5 - VEG4);
                chance = PROB4 + (chance * (PROB5 - PROB4)); 
                density = "HIGH";
            } else if( veg < VEG6 ) { 
                chance = (veg - VEG5)/(VEG6 - VEG5);
                chance = PROB5 + (chance * (PROB6 - PROB5)); 
                density = "HIGH";
            } else {
                chance = 1;
                density = "HIGH";
            }
        } 
        if( block.y > treeLine ) {
            // only the very very smallest chance
            t = 0.0001 * info.soilQuality; 
        }
        int zone = (int)(TEMPERATURE_ZONES * info.temperature);

        echo(String.format("Density: %s  Zone: %d  Chance: %.03f  Splat: %d", density, zone, chance, light));
    }

    addAction("place") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("place:" + loc);

        if( !getPerms(activator, loc).canAddObject() ) {
            echo("You do not have permission to place objects here.");
            return;
        }

        //echo("Placing speakers.");

        boolean useSpeakers = false;
        def shape;
        if( useSpeakers ) {
            shape = ShapeInfo.create("/Models/objects/speakers.blocks", 0.25, entityData);

            // Need to get the size so we can find its base's center.
            // We probably should keep this as part of the object type or even a separate
            // component.
            def cells = shape.loadCells();
            def size = cells.size;
            def offset = size * 0.5 * shape.scale;
            offset.y = 0;
            log.info("cells:" + cells + "  offset:" + offset);

            loc -= offset;

        } else {
            shape = ShapeInfo.create("animals/gaefen.rig", 1, entityData);
            //shape = ShapeInfo.create("human/female/human-female.rig", 1, entityData);
        }


        // FIXME: make a createObject() method that encapsulates some of
        // this, calls a default object init action, etc..
        def pos = new SpawnPosition(loc, new Quatd());

        if( useSpeakers ) {
            def startTime = gameSystems.stepTime.time;
            //def endTime = gameSystems.stepTime.getFutureTime(30); // 30 seconds
            def endTime = gameSystems.stepTime.getFutureTime(32); // 32 seconds

            def placed = createEntity(
                    pos, shape,
                    ObjectName.create("Speakers", entityData),
                    ObjectTypeInfo.create("Speakers", entityData),
                    //SoundInfo.create("/Sounds/Effects/machine-spin-mono.ogg", startTime, 1, true, entityData),
                    SoundInfo.create("/Sounds/Effects/TestSounds.ogg", startTime, 1, true, entityData),
                    //SoundInfo.create("/Sounds/Effects/machine-spin-mono.wav", 0, 1, true, entityData),
                    new CreatedBy(activator.id),
                    new Decay(startTime, endTime)
                );
        } else {
            def startTime = gameSystems.stepTime.time;
            def endTime = gameSystems.stepTime.getFutureTime(120); // 2 minutes

            def placed = createEntity(
                    pos, shape,
                    ObjectName.create("Gaefen", entityData),
                    new Mass(10),
                    new CreatedBy(activator.id),
                    AgentType.create("Gaefin", AgentType.LEVEL_HIGH, entityData),
                    new Decay(startTime, endTime)
                );

            //def anim = createEntity(
            //        new AnimationConfig(placed, new LayerConfig(null, "Idle", "Walk", "Run"))
            //    );
        }

        return true;
    }

    addAction("contextAction") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;

        log.info("contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";

        // Here we want to let the player run some actions on the object

        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        log.info("Player:" + activator.id + " creator:" + creator);

        def options = [];
        options.add(new Option("Look"));

        // Instead of hasShape... we should probably base this on
        // the actions that the object has.  That way we can have non-removable
        // objects that have shapes.  And then move "deleteObject" to the objects
        // that support delete... the others can echo a message that they can't
        // be removed.
        // FIXME: use the runnable action list when we have it... ie: not all
        // actions, just the ones that canRun().
        def objPos = clicked[SpawnPosition]?.location;
        if( getPerms(activator, objPos).canRemoveObject() ) {
            // Options to call us back as the manipulator
            options.add(new Option(target, "deleteObject", clicked));
        }
        session.showPrompt(clicked, PromptType.List, name);
        session.showOptions(clicked, options);
    }

    addAction("deleteObject") { clicked ->
        log.info("deleteObject:" + clicked);

        def objPos = clicked[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canRemoveObject() ) {
            echo("You do not have permission to remove objects here.");
            return;
        }
        // We should probably check to see if we can in case we hacked
        // in around the context menu
        removeEntity(clicked);
    }
}

createType("Speakers").with {
    supertypes("BaseObject");

    addAction("Look") { ->
        def lines = [];

        lines += "Examine: " + object.name;

        lines += "Sound:" + object[SoundInfo]

        session.showPrompt(target, PromptType.Dialog, lines.join("\n"));
        session.showOptions(target, []);

        return true;
    }

}

