/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

// Get the encounter system
encounterSystem = system(EncounterSystem.class);

// Keep track of the registered encounters so that we
// can remove them again
registeredEncounters = [];

DefaultEncounterGenerator.metaClass {

    probability() { Closure func ->
        delegate.setProbability( { func() as Double });
    }

    nextTriggerTime() { Closure func ->
        delegate.setNextTriggerTime(func);
    }

    onTrigger() { Closure func ->
        delegate.setOnTrigger(func);
    }

    isValid() { Closure func ->
        delegate.setValidCheck(func);
    }
}


addEncounterType = { String encounterId, Closure config ->

    log.info("addEncounterType(" + encounterId + ")");

    def init = encounterSystem.addZoneInitializer { Zone zone ->
            def gen = new DefaultEncounterGenerator(zone, encounterId);
            gen.with config;
            if( gen.isValid() ) {   
                zone.addGenerator(gen);
            }
        };
    registeredEncounters.add(init);        
}

class SoundSettings {
    double gain;
    double probability;
    RandomLocation locations;
    double minInterval;
    double maxInterval;
    double duration;
    java.util.function.Supplier<Double> volume;
}

addSoundEncounter = { String encounterId, Closure config -> 
    def settings = new SoundSettings();
    settings.with config;
    
    addEncounterType(encounterId) {
        isValid { zone ->
            return settings.locations.hasLocations(zone);
        }
        
        probability {
            // Short circuit if the volume is currently 0
            double v = settings.volume.get();
            if( v <= 0 ) {
                return 0.0;
            } 
            return settings.probability; 
        };

        nextTriggerTime { time ->
            long result = time.getFutureTime(settings.minInterval + Math.random() * (settings.maxInterval - settings.minInterval)); 
            //log.info("nextTriggerTime(" + time + "):" + result); 
            return result; 
        }
 
        onTrigger { zone, time ->
            if( log.isTraceEnabled() ) {
                log.trace("onTrigger(" + zone + ", " + time + ")");                
                log.trace("encounter ID:" + encounterId);
            }
 
            double v = settings.volume.get() * settings.gain;
            if( v <= 0 ) {
                return null;
            }
        
            // Create the sound encounter and return it
            def loc = settings.locations.findLocation(zone);
            if( loc == null ) {
                log.warn("Random location not found for:" + zone + "  encounterId:" + encounterId);
                return null;
            } 

            // Positioned in the center of the block           
            SpawnPosition pos = new SpawnPosition(loc.toVec3d().add(0.5, 0.5, 0.5), new Quatd());
        
            // A test shape for debugging
            //ShapeInfo shape = ShapeInfo.create("/Models/objects/speakers.blocks", 1, entityData);
 
            // The sound is 32 seconds long
            long soundEnd = time.getFutureTime(settings.duration);
            def info = SoundInfo.create(encounterId, SoundInfo.TYPE_AMBIENT, 
                                        time.getTime(), v, false, entityData);
        
            def sound = createEntity(
                            info, pos, new Decay(time.getTime(), soundEnd),
                            // For debugging:
                            //shape,
                            //ObjectName.create("Speakers", entityData),
                            //ObjectTypeInfo.create("Speakers", entityData),
                            );
                            
            log.info("Created sound:" + sound + " id:" + encounterId + "  at:" + pos.location + "  info:" + info);
        
            return new SoundEncounter(entityData, sound, soundEnd);
        }        
    }
}


cleanupEncounterTypes = {
    registeredEncounters.each { encounterType ->
        encounterSystem.removeZoneInitializer(encounterType);
    }
    registeredEncounters.clear();
}

// Cleanup any of the registered settings
mod.onTerminate {
    cleanupEncounterTypes();
}

