/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

import org.slf4j.*;
//import org.apache.log4j.Category;
//import org.apache.log4j.NDC;
// I don't know why those fail.
import com.simsilica.action.*;
import com.simsilica.mod.*;
import com.simsilica.mworld.World;
import com.simsilica.ext.mblock.*;
import mythruna.GameConstants;
import mythruna.assembly.db.SubassemblyId;
import mythruna.assembly.io.SubassemblyJson;
import mythruna.sim.Activator;
import mythruna.world.WorldManager;
import mythruna.net.server.GameSessionHostedService.ActivatorImpl;

// Define access to the services
system = { Class type ->
    return gameSystems.get(type, true);
}

optionalSystem = { Class type ->
    return gameSystems.get(type);
}

entityData = system(EntityData);
world = system(World);
stepTime = gameSystems.stepTime;

// For the moment, server side only
worldStats = optionalSystem(WorldStats);

worldTime = system(WorldTime);

now = { ->
    return stepTime.time;
}

future = { double seconds ->
    return stepTime.getFutureTime(seconds);
}

loadText = { id ->
    return worldManager.getTextDb().getText(id, null);
}

gameActionSystem = optionalSystem(GameActionSystem);

// Add something to the once-a-frame spooler.  Spooled closures
// are run and discarded but only one per frame on a first-come basis.
spool = { closure ->
    def savedEnv = ActionEnvironment.getCurrentEnvironment();

    system(GameActionSystem).spool {
        ActionEnvironment.pushCurrentEnvironment(savedEnv);
        try {
            closure.call();
        } finally {
            ActionEnvironment.popCurrentEnvironment(savedEnv);
        }
    }
}

spoolAsWorld = { closure ->
    def savedEnv = gameActionSystem.worldEnvironment;

    system(GameActionSystem).spool {
        ActionEnvironment.pushCurrentEnvironment(savedEnv);
        try {
            closure.call();
        } finally {
            ActionEnvironment.popCurrentEnvironment(savedEnv);
        }
    }
}

// Add something to be run at the start of the next game loop frame
nextLoop = { closure ->
    def savedEnv = ActionEnvironment.getCurrentEnvironment();

    gameSystems.enqueue {
        ActionEnvironment.pushCurrentEnvironment(savedEnv);
        try {
            closure.call();
        } catch( Exception e ) {
            log.error("Error running:" + closure, e);
        } finally {
            ActionEnvironment.popCurrentEnvironment(savedEnv);
        }
    }
}

onBackground = { closure ->
    def savedEnv = ActionEnvironment.getCurrentEnvironment(false);
    if( savedEnv == null ) {
        savedEnv = gameActionSystem.worldEnvironment;
    }
    system(ThreadPoolSystem).execute {
        ActionEnvironment.pushCurrentEnvironment(savedEnv);
        try {
            closure.call();
        } catch( Exception e ) {
            log.error("Error running:" + closure, e);
        } finally {
            ActionEnvironment.popCurrentEnvironment(savedEnv);
        }
    }
}

// Physics only exists on the server and these APIs are both
// client and server
physics = gameSystems.get(PhysicsSpace, false);
if( physics ) {
    // Then we should have mphys also
    shapeFactory = gameSystems.get(MPhysSystem.class).bodyFactory.shapeFactory;
}

findEntities = { ComponentFilter filter, Class... types ->
    if( filter != null && (types == null || types.length == 0) ) {
        types = [filter.getComponentType()];
    }
    if( filter != null && !types.contains(filter.getComponentType()) ) {
        throw new IllegalArgumentException("Filtered type:" + filter.getComponentType() + " is not in components list:" + types);
    }
    return entityData.findEntities(filter, types);
}

findEntity = { ComponentFilter filter, Class... types ->
    if( types == null || types.length == 0 ) {
        types = [filter.getComponentType()];
    }
    if( filter != null && !types.contains(filter.getComponentType()) ) {
        throw new IllegalArgumentException("Filtered type:" + filter.getComponentType() + " is not in components list:" + types);
    }
    return entityData.findEntity(filter, types);
}

findEntity2 = { EntityCriteria criteria ->
    return entityData.findEntity(criteria);
}

createEntity = { EntityComponent... init ->
    def result = entityData.createEntity();
    entityData.setComponents(result, init);
    return result;
}

removeEntity = { entity ->
    long start = System.nanoTime();
    def entityId = resolveEntityId(entity);
    entityId.runIfExists("onRemove");
    long run = System.nanoTime();
    entityData.removeEntity(entityId);
    long end = System.nanoTime();
    long delta = end - start;
    if( delta > 16000000L && "GameLoopThread".equals(Thread.currentThread().name)) {
        log.warn(String.format("removeEntity(%s) took: %.03f ms, run: %.03f, remove: %.03f",
                                entity,
                                (end - start)/1000000.0,
                                (run - start)/1000000.0,
                                (end - run)/1000000.0
                              ));
    }
}

worldManager = optionalSystem(WorldManager);
if( worldManager ) {
    stringToCellArray = CellArrayFunctions.chain(
                            CellArrayFunctions.cellArrayStorage(worldManager.getCellArrayStorage()),
                            CellArrayFunctions.blocksResource(),
                            );
}

loadShape = { String shapeName, double scale, def mass ->
    if( !(mass instanceof Mass) ) {
        mass = new Mass(mass);
    }
    return shapeFactory.createShape(shapeName, scale, mass);
}

loadCellsForName = { String shapeName ->
    return stringToCellArray.apply(shapeName);
}

loadSubassembly = { def name ->
    if( !(name instanceof ShapeName) ) {
        name = ShapeName.parse(name);
    }
    if( "asm" == name.type ) {
        def id = SubassemblyId.fromString(name.getName());
        return worldManager?.subassemblyStorage?.apply(id);
    }
    if( "assembly" == name.type ) {
        return SubassemblyJson.loadResource(name.getFullName());
    }
    throw new IllegalArgumentException("Unsupported subassembly type:" + name.getType());
}


// Convenience method for resolving an object to its EntityId wherever
// that is needed.
resolveEntityId = { Object o ->
    if( o == null ) {
        return null;
    } else if( o instanceof EntityId ) {
        return (EntityId)o;
    } else if( o instanceof Activator ) {
        return ((Activator)o).id;
    } else if( o instanceof ActionContext ) {
        return ((ActionContext)o).target;
    } else if( o instanceof ActionEnvironment ) {
        return ((ActionEnvironment)o).activator.id;
    }
    throw new IllegalArgumentException("Object cannot be resolved to EntityId:" + o);
}

findBlueprint = { EntityId parent, String bpName ->
    def filter1 = Filters.fieldEquals(BlueprintInfo.class, "parent", parent);
    def filter2 = Filters.fieldEquals(BlueprintInfo.class, "name", bpName);
    def filter = Filters.and(BlueprintInfo.class, filter1, filter2);
    return entityData.findEntity(filter, BlueprintInfo.class);
}

findDesign = { EntityId parent, String bpName ->
    def filter1 = Filters.fieldEquals(AssemblyBlueprintInfo.class, "parent", parent);
    def filter2 = Filters.fieldEquals(AssemblyBlueprintInfo.class, "name", bpName);
    def filter = Filters.and(AssemblyBlueprintInfo.class, filter1, filter2);
    return entityData.findEntity(filter, AssemblyBlueprintInfo.class);
}

createBlueprintFromCells = { EntityId parent, String bpName, String objectName, CellArray cells, boolean carved, double scale ->

    // See if there is already an entity for the bpName
    EntityId entity = findBlueprint(parent, bpName);
    if( entity == null ) {
        entity = entityData.createEntity();
        log.info("Creating blueprint for:" + bpName);
    } else {
        BlueprintInfo bpInfo = entityData.getComponent(entity, BlueprintInfo.class);
        log.info("Found existing blueprint for:" + bpName + " : " + bpInfo);
    }

    CellArrayId cellId = worldManager.getCellArrayStorage().store(cells);

    Vec3i offset = new Vec3i();
    offset.x = 5 - (cells.getSizeX()/2);
    offset.z = 5 - (cells.getSizeZ()/2);

    String shapeName = cellId.toFileName();
    if( carved ) {
        shapeName = "c_" + shapeName;
    }

    log.info("createing shapeId:" + shapeName);
    entityData.setComponents(entity,
            new BlueprintInfo(parent, bpName, offset),
            ObjectName.create(objectName, entityData),
            ShapeInfo.create(shapeName, scale, entityData),
            new CreatedBy(parent)
            );

    return entity;
}

getBody = { Object o ->
    if( physics == null ) {
        return null;
    }
    def id = resolveEntityId(o);
    def result = physics.binIndex.getRigidBody(id);
    if( result != null ) {
        return result;
    }
    result = physics.binIndex.getStaticBody(id);
    return result;
}

// Optional only because this API file runs on both client and server and there
// are no agents on the client.
agents = optionalSystem(AgentSystem);
getBrain = { Object o ->
    if( agents == null ) {
        return null;
    }
    return agents.getAgentDriver(resolveEntityId(o))?.brain;
}

