/*
 * Copyright (c) 2023, Simsilica, LLC
 * All rights reserved.
 */

class TransactionalWorld {
    World world;
    WorldEdits worldEdits;

    public TransactionalWorld( World world ) {
        this.world = world;
        this.worldEdits = new WorldEdits();
    }

    public int setWorldCell( Vec3d worldLoc, int type ) {
        return world.setWorldCell(worldLoc, type, worldEdits);
    }

    public int getWorldCell( Vec3d worldLoc ) {
        // We should check the worldEdits to see if we've already
        // changed it.
        return world.getWorldCell(worldLoc);
    }

    public void close() {
        world.deliverEvents(worldEdits);
    }
}

createType("SmallChestLid").with {
    supertypes("BaseObject");
    setTypeVar("description", "The lid is closed tight.");
}

createType("WoodChest").with {
    supertypes("BaseAssembly");

    setTypeVar("description", "A strong wooden chest.");
    setTypeVar("shape",
               //ShapeInfo.create("/Models/objects/sm-chest1-bottom.blocks", 0.25, entityData));
               ShapeInfo.create("/Models/objects/wood-chest1.assembly", 1, entityData));
    //setTypeVar("shapeBaseCenter", new Vec3d(0.375, 0, 0.375));
    setTypeVar("shapeBaseCenter", new Vec3d());

    addAction("initialize") { -> // no args
        //self << ObjectName.create("Small Chest", entityData);
        self << ObjectName.create("Wood Chest", entityData);
        self << self.type.shape;
        self << new Mass(0);
    }

    addAction("place") { Vec3d loc, Quatd rotation ->
        //loc.x = Math.floor(loc.x) + 0.5;
        //loc.y = Math.floor(loc.y);
        //loc.z = Math.floor(loc.z) + 0.5;
        loc = loc - self.type.shapeBaseCenter;

        def pos = new SpawnPosition(loc, rotation);
        def offset = rotation.mult(new Vec3d(0.125, 0.5, 0.125));

        self << pos;

        //// We can't build assemblies yet so let's just manually place some things
        //def top = createEntity(
        //        ObjectTypeInfo.create("SmallChestLid", entityData),
        //        new SpawnPosition(loc.add(offset), rotation),
        //        ShapeInfo.create("/Models/objects/sm-chest1-top.blocks", 0.25, entityData)
        //    );
        // Except we can build assemblies now!!!
    }
}

createType("Sconce").with {
    supertypes("BaseObject");

    setTypeVar("description", "A metal bracket with burning fuel.");
    setTypeVar("shape",
               ShapeInfo.create("/Models/objects/sconce-iron.blocks", 0.25, entityData));
    setTypeVar("shapeBaseCenter", new Vec3d(0.5, -0.25, 0.125));

    addAction("initialize") { -> // no args
        self << ObjectName.create("Sconce", entityData);
        self << self.type.shape;
    }

    addAction("place") { Vec3d loc, Quatd rotation ->
        //loc.x = Math.floor(loc.x) + 0.5;
        //loc.y = Math.floor(loc.y);
        //loc.z = Math.floor(loc.z) + 0.5;
        loc = loc - rotation.mult(self.type.shapeBaseCenter);

        def pos = new SpawnPosition(loc, rotation);

        self << pos;
    }
}

createSconce = { env, loc, norm ->
    double rads;
    if( norm.x != 0 || norm.z != 0 ) {
        rads = Math.atan2(-norm.z, norm.x);
    } else {
        rads = 0;
    }

    def v = loc.clone();
    v.x = Math.floor(v.x) + 0.5;
    //v.y = Math.floor(v.y);
    v.z = Math.floor(v.z) + 0.5;

    def placed = createEntity(ObjectTypeInfo.create("Sconce", entityData));
    placed.run("initialize");
    placed.run("place", v, new Quatd().fromAngles(0, rads, 0));
}

createType("Grate").with {
    supertypes("BaseObject");

    setTypeVar("description", "A metal grate fixed into the floor.");
    setTypeVar("shape",
               ShapeInfo.create("/Models/objects/grate-iron.blocks", 0.125, entityData));
    setTypeVar("shapeBaseCenter", new Vec3d(0.5, 0, 0.5));

    addAction("initialize") { -> // no args
        self << ObjectName.create("Grate", entityData);
        self << self.type.shape;
    }

    addAction("place") { Vec3d loc, Quatd rotation ->
        loc = loc - rotation.mult(self.type.shapeBaseCenter);
        def pos = new SpawnPosition(loc, rotation);
        self << pos;
    }
}

createGrate = { env, loc, norm ->
    double rads;
    if( norm.x != 0 || norm.z != 0 ) {
        rads = Math.atan2(-norm.z, norm.x);
    } else {
        rads = 0;
    }

    def v = loc.clone();
    v.x = Math.floor(v.x) + 0.5;
    def yOffset = v.y;
    v.y = Math.floor(v.y);
    v.z = Math.floor(v.z) + 0.5;
    yOffset = (int)((yOffset - v.y) / 0.25);
    v.y += (yOffset * 0.25) + 0.125;

    def placed = createEntity(ObjectTypeInfo.create("Grate", entityData));
    placed.run("initialize");
    placed.run("place", v, new Quatd().fromAngles(0, rads, 0));
    placed << new Mass(0);
}


createType("Campfire").with {
    supertypes("BaseItem");

    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "A campfire."
             );
    setTypeVar("shape",
               ShapeInfo.create("/Models/objects/campfire-lit.blocks", 0.25, entityData));
    setTypeVar("shapeBaseCenter", new Vec3d(0.375, 0, 0.375));

    addAction("initialize") { -> // no args
        self << ObjectName.create(self.type.name, entityData);
        self << self.type.shape;
    }

    addAction("place") { Vec3d loc ->
        //loc.x = Math.floor(loc.x) + 0.5;
        //loc.y = Math.floor(loc.y);
        //loc.z = Math.floor(loc.z) + 0.5;
        loc = loc - self.type.shapeBaseCenter;

        def pos = new SpawnPosition(loc, new Quatd());

        self << pos;

        def startTime = gameSystems.stepTime.time;

        // Just to keep a whole mess of them from being directly in sync
        // if started at the same time.
        startTime -= gameSystems.stepTime.toSimTime(Math.random() * 10);

        // For now, volume also controls max distance and 1 = 48m.  0.5 is probably
        // still too high for that.
        self << SoundInfo.create("/Sounds/Effects/campfire1-mono.ogg", startTime, 0.5, true, entityData);
    }

    addAction("move") { loc, facing ->
        // Can't drag the campfires because it will make their light blocks weird
    }

    addAction("placeAtCell") { def loc ->
        if( loc instanceof Vec3i ) {
            loc = new Vec3d(loc);
        }
        loc.x += 0.5;
        loc.z += 0.5;
        run("place", loc);
    }

    addAction("upgrade") { -> // no args
        def startTime = gameSystems.stepTime.time; // nanoseconds, remember
        startTime -= gameSystems.stepTime.toSimTime(Math.random() * 10);
        self << SoundInfo.create("/Sounds/Effects/campfire1-mono.ogg", startTime, 0.5, true, entityData);
        //self << SoundInfo.create("/Sounds/Effects/TestSounds.ogg", startTime, 0.5, true, entityData);
    }

    addAction("activate") { ->
        def loc = self.location;
        if( loc != null ) {
            int existing = MaskUtils.getType(world.getWorldCell(self.location));
            int flame = BlockTypeIndex.findType(new BlockName("fire", "flame-lg"));
            world.setWorldCell(self.location, flame);
        }
        def startTime = gameSystems.stepTime.time;

        // For now, volume also controls max distance and 1 = 48m.  0.5 is probably
        // still too high for that.
        self << SoundInfo.create("/Sounds/Effects/campfire1-mono.ogg", startTime, 0.5, true);
    }

    addAction("deactivate") { ->
        def loc = self.location;
        if( loc != null ) {
            int existing = MaskUtils.getType(world.getWorldCell(self.location));
            int flame = BlockTypeIndex.findType(new BlockName("fire", "flame-lg"));
            if( existing == flame ) {
                world.setWorldCell(self.location, 0);
            }
        }
        self.remove(SoundInfo.class);
    }

    addAction("Light") { ->
        run("activate");
        self << getTypeVar("shape", null);
    }.onlyIf { ->
        return self[ShapeInfo] != getTypeVar("shape", null);
    }

    addAction("Douse") { ->
        run("deactivate");
        self << ShapeInfo.create("/Models/objects/campfire-unlit.blocks", 0.25);
    }.onlyIf { ->
        return self[ShapeInfo] == getTypeVar("shape", null);
    }

    addAction("onRemove") { ->
        run("deactivate");
    }
}

// Platform for returning to the start of a dungeon
createType("ExitPlatform").with {
    supertypes("BaseObject");

    setTypeVar("description", "A glowing contraption.");
    setTypeVar("shape",
               ShapeInfo.create("/Models/objects/exit-pad.carved.blocks", 0.25, entityData));
    setTypeVar("shapeBaseCenter", new Vec3d(5, 0, 5) * 0.25);

    addAction("initialize") { -> // no args
        self << ObjectName.create("Exit Pad", entityData);
        self << self.type.shape;
    }

    addAction("place") { Vec3d loc, Quatd rotation ->
        def blockLoc = loc.floor().toVec3d();
        loc = loc - rotation.mult(self.type.shapeBaseCenter);
        def pos = new SpawnPosition(loc, rotation);
        self << pos;

        int dimBlue = BlockTypeIndex.findType(new BlockName("magic-blue", "mini1"));
        world.setWorldCell(blockLoc, dimBlue);
    }

    addAction("Take") { ->
    }.onlyIf { ->
        // We want things looking for "Take" to find it but not to actually
        // allow being taken.
        return false;
    }

    addAction("Return to Entrance") { ->
        def entrance = self[EntityLink]?.target;
        //echo("This would be cool. target:" + entrance);
        def target = entrance[SpawnPosition];
        echo("Warping to:" + target?.location);

        //activator << entrance[SpawnPosition]
        def phys = system(PhysicsSpace)
        phys.teleport(activator.id, target.location.add(0.5, 0, 0.5), target.orientation);
    }
}

// A standard object type that can be placed on the ground and later retrieved
createType("Placeable").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Creator");

    // Subtypes can define:
    // placedShape - defaults to current shape or the type's default "shape"
    // shapeBaseCenter - prescaled offset to the center of the object's base
    //                   defaults to nothing.

    addAction("place") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = loc.floor();
        log.info("place:" + self + " at:" + loc);

        // See if this object is even still held by the player.
        // They may have double-clicked or something.
        if( activator[Holding]?.target != self ) {
            // Since these objects can only be dropped by targeted placing
            // then checking for 'held' should be good enough
            log.info("Skipping double click for:" + self);
            return;
        }

        if( !getPerms(activator, loc).canAddObject() ) {
            echo("You do not have permission to add an object here.");
            return;
        }

        if( activator[Holding]?.target == self ) {
            log.info("Unholding:" + self);
            activator.remove(Holding);
        }

        def shape = self.type.getTypeVar("placedShape", self[ShapeInfo]);
        if( shape == null ) {
            shape = self.type.shape;
        }
        def scale = shape.scale;

        def v = loc;
        if( self.type.shapeBaseCenter ) {
            v -= self.type.shapeBaseCenter;
        }
        def pos = new SpawnPosition(v, new Quatd());

        self << new ShapeInfo(shape.shapeId, scale);
        self << new Mass(0);
        self << pos;
        self << new OwnedBy(activator.id);

        log.info("Placed object at:" + pos.location);
        self.remove(ContainedIn.class);

        runIfExists("onPlaced", pos, blockHit);
    }

    addAction("Take") { ->
        log.info("Activator:" + activator + " picking up object " + self.name);

        //self <<
        // FIXME: hook up activator.id a little more consistently
        def backpack = findItem(activator.id, "Backpack");

        def shape = self[ShapeInfo];

        // Need to find an empty spot or not allow it to be picked up
        self << new ContainedIn(activator.id, backpack, 18, 0);
        self << self.type.getTypeVar("shape", null)
        self << new Mass(0.1);
        self.remove(SpawnPosition);

        // If the player isn't holding anything right now then equip it
        if( activator[Holding] == null ) {
            log.info("Not holding anything...");
            if( canRun("Equip") ) {
                log.info("Trying to equip:" + self.name);
                run("Equip");
            }
        }

        runIfExists("onTaken", backpack);
    }.onlyIf { ->
        // Can only pick up the object if it originally belonged to
        // the player and is not currently contained
        if( self[ContainedIn] ) {
            return false;
        }
        if( !activator.isSameEntity(self[OwnedBy]?.owner) ) {
            return false;
        }
        return true;
    }

    addAction("onTaken") { container ->
        if( self.isSameEntity(activator[Holding]?.target) ) {
            echo("Equipped " + self.name + ".")
        } else {
            echo("Retrieved " + self.name + ".")
        }
    }

}

createType("DropTest").with {
    supertypes("Placeable");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(1, 3, 1));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "Testing item drop and retrieval."
             );

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/dungeon-generator.blocks", standardItemScale * 0.5));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    // Placeable setup.  Note: bigger than natural item scale in this case
    setTypeVar("placedShape",
               ShapeInfo.create("/Models/items/dungeon-generator.blocks", 0.25));
    setTypeVar("shapeBaseCenter", new Vec3d(1.5 * 0.25, 0, 1.5 * 0.25));

    addAction("onPlaced") { pos, blockHit ->
        echo("onPlaced:" + pos);

        def placed = createEntity(
            ObjectTypeInfo.create("ExitPlatform"),
            new Mass(0)
            );
        def v = pos.location.floor().toVec3d();
        v.addLocal(0.5, 0, 0.5);

        placed.run("initialize");
        placed.run("place", v, pos.orientation);
    }
}

createType("DungeonGenerator").with {
    //supertypes("BaseItem");
    supertypes("Placeable");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(1, 3, 1));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "Generates test dungeons."
             );

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/dungeon-generator.blocks", standardItemScale * 0.5));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    // Placeable setup.  Note: bigger than natural item scale in this case
    setTypeVar("placedShape",
               ShapeInfo.create("/Models/items/dungeon-generator.blocks", 0.25));
    setTypeVar("shapeBaseCenter", new Vec3d(1.5 * 0.25, 0, 1.5 * 0.25));

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        echo("Clicked:" + block)

        if( !getPerms(activator, loc).canRemoveBlock() ) {
            echo("You do not have permission to change blocks here.");
            return;
        }

        //int testType = BlockTypeIndex.findType(new BlockName("mud-stone", "cube"));
        //
        //def xWorld = new TransactionalWorld(world);
        //
        //for( int x = 0; x < 10; x++ ) {
        //    for( int z = 0; z < 10; z++ ) {
        //        for( int y = 0; y < 10; y++ ) {
        //            xWorld.setWorldCell(loc.add(x, y, z), testType);
        //        }
        //    }
        //}
        //
        //xWorld.close();

        //def placed = createEntity(ObjectTypeInfo.create("Campfire", entityData));
        //placed.run(env, "initialize");
        //placed.run(env, "place", loc);

        if( false ) {
            // Find all of the existing campfires and delete them
            def type = ObjectTypeInfo.create("Campfire", entityData)
            def filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            def found = findEntities(filter, ObjectTypeInfo);
            int count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " campfire objects");
        }

        if( false ) {
            // Testing object placement
            def placed = createEntity(ObjectTypeInfo.create("WoodChest", entityData));
            placed.run("initialize");
            placed.run("place", loc);
        }

        if( true ) {
            // Find all of the existing test objects and delete them
            def type = ObjectTypeInfo.create("Sconce", entityData)
            def filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            def found = findEntities(filter, ObjectTypeInfo);
            int count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " sconce objects");

            type = ObjectTypeInfo.create("Grate", entityData)
            filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            found = findEntities(filter, ObjectTypeInfo);
            count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " grate objects");

            type = ObjectTypeInfo.create("MapBoard", entityData)
            filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            found = findEntities(filter, ObjectTypeInfo);
            count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " MapBoard objects");

            type = ObjectTypeInfo.create("Campfire", entityData)
            filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            found = findEntities(filter, ObjectTypeInfo);
            count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " Campfire objects");

            type = ObjectTypeInfo.create("WoodChest", entityData)
            filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            found = findEntities(filter, ObjectTypeInfo);
            count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " WoodChest objects");

            type = ObjectTypeInfo.create("SmallChestLid", entityData)
            filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            found = findEntities(filter, ObjectTypeInfo);
            count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            if( count ) {
                echo("Removed " + count + " SmallChestLid objects");
            }

            type = ObjectTypeInfo.create("ExitPlatform", entityData)
            filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            found = findEntities(filter, ObjectTypeInfo);
            count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " ExitPlatform objects");

            type = ObjectTypeInfo.create("DungeonMarker", entityData)
            filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            found = findEntities(filter, ObjectTypeInfo);
            count = 0;
            found.each {
                log.info("Deleting:" + it);
                entityData.removeEntity(it);
                count++;
            }
            echo("Removed " + count + " DungeonMarker objects");
        }

        if( false ) {
            def type = ObjectTypeInfo.create("Campfire", entityData)
            def filter = fieldEquals(ObjectTypeInfo, "typeId", type.typeId);
            def found = findEntities(filter, ObjectTypeInfo);
            def count = 0;
            found.each {
                log.info("Upgrading:" + it);
                it.run("upgrade");
                count++;
            }
            echo("Upgraded " + count + " Campfire objects");
        }

        if( false ) {
            // By default the sconse faces down positive x... so
            // if we want to rotate it to match normal we need to find
            // the right angle from that.
            double rads;
            def norm = blockHit.normal;
            if( norm.x != 0 || norm.z != 0 ) {
                rads = Math.atan2(-norm.z, norm.x);
            } else {
                rads = 0;
            }

            def v = loc.add(norm * 0.9);
            v.x = Math.floor(v.x) + 0.5;
            v.y = Math.floor(v.y);
            v.z = Math.floor(v.z) + 0.5;

            def placed = createEntity(ObjectTypeInfo.create("Sconce", entityData));
            placed.run("initialize");
            placed.run("place", v, new Quatd().fromAngles(0, rads, 0));

            echo("loc:" + loc + "  v:" + v);

            //placed = createEntity(ObjectTypeInfo.create("Sconce", entityData));
            //placed.run(env, "initialize");
            //placed.run(env, "place", loc, new Quatd().fromAngles(0, Math.PI * 0.5, 0));
        }

        if( false ) {
            double rads;
            def norm = blockHit.normal;
            if( norm.x != 0 || norm.z != 0 ) {
                rads = Math.atan2(-norm.z, norm.x);
            } else {
                rads = 0;
            }

            def v = loc.add(norm * 0.9);
            v.x = Math.floor(v.x) + 0.5;
            def yOffset = v.y;
            v.y = Math.floor(v.y);
            v.z = Math.floor(v.z) + 0.5;
            yOffset = (int)((yOffset - v.y) / 0.25);
            v.y += (yOffset * 0.25) + 0.125;

            def placed = createEntity(ObjectTypeInfo.create("Grate", entityData));
            placed.run("initialize");
            placed.run("place", v, new Quatd().fromAngles(0, rads, 0));
            placed << new Mass(0);

            echo("loc:" + loc + "  v:" + v);

            //placed = createEntity(ObjectTypeInfo.create("Sconce", entityData));
            //placed.run(env, "initialize");
            //placed.run(env, "place", loc, new Quatd().fromAngles(0, Math.PI * 0.5, 0));
        }

    }

    addAction("onPlaced") { objectPos, blockHit ->
        Vec3d objectLoc = objectPos.location;
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("place:" + loc);

        // These checks need to be bigger for 'real life'
        // FIXME: make the perms check based on the whole dungeon area and not just the click
        //def perms = getPerms(activator, loc);
        //if( !perms.canRemoveBlock() ) {
        //    echo("You do not have permission to change blocks here.");
        //    return;
        //}
        //if( !perms.canAddObject() ) {
        //    echo("You do not have permission to add an object here.");
        //    return;
        //}
        // FIXME: hook in more advanced perm checks from the base type


        //for( int i = 0; i < 10; i++ ) {
        //    world.setWorldCell(loc.subtract(0, i, 0), 0);
        //}
        //
        //int flame = BlockTypeIndex.findType(new BlockName("fire", "flame-lg"));
        //
        //for( int x = -40; x < 40; x++ ) {
        //    for( int z = -40; z < 40; z++ ) {
        //        for( int y = 0; y < 3; y++ ) {
        //            world.setWorldCell(loc.add(x, -10 + y, z), 0);
        //        }
        //        if( (x % 10) == 0 && (z % 10) == 0 ) {
        //            world.setWorldCell(loc.add(x, -10, z), flame);
        //        }
        //    }
        //}
        //
        //   ************
        //   ************
        //   ************
        //   ************
        //   []**********
        //   ************
        //   ************
        //   ************
        //   ************
        //
        //

        def style = createBasicStyle(BlockTypeIndex.findType(new BlockName("mud-stone", "cube")),
                                     BlockTypeIndex.findType(new BlockName("stone", "cube")),
                                     BlockTypeIndex.findType(new BlockName("cobble", "cube"))
                                     );
        def style2 = createBasicStyle(BlockTypeIndex.findType(new BlockName("dirt", "cube")),
                                     BlockTypeIndex.findType(new BlockName("stone", "cube")),
                                     BlockTypeIndex.findType(new BlockName("cobble", "cube"))
                                     );

        int smallFlame = BlockTypeIndex.findType(new BlockName("fire", "flame-sm"));
        int flame = BlockTypeIndex.findType(new BlockName("fire", "flame-lg"));

        def drawFlameSconces = { w, pos, info ->
                if( info.wallLength <= 6 ) {
                    if( info.wallIndex == (int)(info.wallLength / 2) ) {
                        int sconce = Math.min(info.wallHeight - 1, 2);

                        // See if there is a doorway here
                        boolean hasDoor = w.getWorldCell(pos.subtract(info.normal).toVec3d()) == 0;

                        //world.setWorldCell(pos.toVec3d().add(0, info.wallHeight - 1, 0), smallFlame);
                        if( !hasDoor && sconce >= 2 ) {
                            //echo("flame at:" + pos.toVec3d().add(0, sconce, 0));
                            w.setWorldCell(pos.toVec3d().add(0, sconce, 0), smallFlame);
                            createSconce(env, pos.toVec3d().add(0, sconce-1, 0), info.normal);
                        } else {
                            //echo("flame at:" + pos.toVec3d().add(0, -1, 0));
                            w.setWorldCell(pos.toVec3d().add(0, -1, 0), smallFlame);
                            createGrate(env, pos.toVec3d().add(0, -1 + 0.75, 0), info.normal);
                        }
                    }
                    return;
                }
                if( info.wallIndex == (int)(info.wallLength / 4) ) {
                    int sconce = Math.min(info.wallHeight - 1, 2);
                    // See if there is a doorway here
                    boolean hasDoor = w.getWorldCell(pos.subtract(info.normal).toVec3d()) == 0;
                    if( !hasDoor && sconce >= 2 ) {
                        w.setWorldCell(pos.toVec3d().add(0, sconce, 0), smallFlame);
                        createSconce(env, pos.toVec3d().add(0, sconce-1, 0), info.normal);
                    } else {
                        w.setWorldCell(pos.toVec3d().add(0, -1, 0), smallFlame);
                        createGrate(env, pos.toVec3d().add(0, -1 + 0.75, 0), info.normal);
                    }
                }
                if( info.wallIndex == (int)(info.wallLength / 4) * 3 ) {
                    // See if there is a doorway here
                    boolean hasDoor = w.getWorldCell(pos.subtract(info.normal).toVec3d()) == 0;
                    int sconce = Math.min(info.wallHeight - 1, 2);
                    if( !hasDoor && sconce >= 2 ) {
                        w.setWorldCell(pos.toVec3d().add(0, sconce, 0), smallFlame);
                        createSconce(env, pos.toVec3d().add(0, sconce-1, 0), info.normal);
                    } else {
                        w.setWorldCell(pos.toVec3d().add(0, -1, 0), smallFlame);
                        createGrate(env, pos.toVec3d().add(0, -1 + 0.75, 0), info.normal);
                    }
                }
            }
        def drawMagicSconces = { w, pos, info, magicLight ->
                if( info.wallLength <= 6 ) {
                    if( info.wallIndex == (int)(info.wallLength / 2) ) {
                        int sconce = Math.min(info.wallHeight - 1, 2);
                        // See if there is a doorway here
                        boolean hasDoor = w.getWorldCell(pos.subtract(info.normal).toVec3d()) == 0;
                        if( !hasDoor && sconce >= 1 ) {
                            w.setWorldCell(pos.toVec3d().add(0, sconce, 0), magicLight);
                            createSconce(env, pos.toVec3d().add(0, sconce-0.6, 0), info.normal);
                        } else {
                            w.setWorldCell(pos.toVec3d().add(0, -1, 0), magicLight);
                            createGrate(env, pos.toVec3d().add(0, -1 + 0.75, 0), info.normal);
                        }
                    }
                    return;
                }
                if( info.wallIndex == (int)(info.wallLength / 4) ) {
                    int sconce = Math.min(info.wallHeight - 1, 2);
                    // See if there is a doorway here
                    boolean hasDoor = w.getWorldCell(pos.subtract(info.normal).toVec3d()) == 0;
                    if( !hasDoor && sconce >= 1 ) {
                        w.setWorldCell(pos.toVec3d().add(0, sconce, 0), magicLight);
                        createSconce(env, pos.toVec3d().add(0, sconce-0.6, 0), info.normal);
                    } else {
                        w.setWorldCell(pos.toVec3d().add(0, -1, 0), magicLight);
                        createGrate(env, pos.toVec3d().add(0, -1 + 0.75, 0), info.normal);
                    }
                }
                if( info.wallIndex == (int)(info.wallLength / 4) * 3 ) {
                    // See if there is a doorway here
                    boolean hasDoor = w.getWorldCell(pos.subtract(info.normal).toVec3d()) == 0;
                    int sconce = Math.min(info.wallHeight - 1, 2);
                    if( !hasDoor && sconce >= 1 ) {
                        w.setWorldCell(pos.toVec3d().add(0, sconce, 0), magicLight);
                        createSconce(env, pos.toVec3d().add(0, sconce-0.6, 0), info.normal);
                    } else {
                        w.setWorldCell(pos.toVec3d().add(0, -1, 0), magicLight);
                        createGrate(env, pos.toVec3d().add(0, -1 + 0.75, 0), info.normal);
                    }
                }
            }

        int dimRed = BlockTypeIndex.findType(new BlockName("magic-red", "mini1"));
        int dimGreen = BlockTypeIndex.findType(new BlockName("magic-green", "mini1"));
        int dimBlue = BlockTypeIndex.findType(new BlockName("magic-blue", "mini1"));
        def lightingTypes = [
                drawFlameSconces,
                drawFlameSconces,
                drawFlameSconces,
                drawFlameSconces,
                { w, pos, info ->
                    drawMagicSconces(w, pos, info, dimRed);
                },
                { w, pos, info ->
                    drawMagicSconces(w, pos, info, dimGreen);
                },
                { w, pos, info ->
                    drawMagicSconces(w, pos, info, dimBlue);
                }
            ];
        //lightingTypes.add(drawFlameSconces);
        //lightingTypes.add(drawMagicSconces.clone());
        //lightingTypes.add(drawMagicSconces.clone());
        //lightingTypes.add(drawMagicSconces.clone());
        //lightingTypes[1].magicLight = BlockTypeIndex.findType(new BlockName("magic-red", "mini1"));
        //lightingTypes[2].magicLight = BlockTypeIndex.findType(new BlockName("magic-green", "mini1"));
        //lightingTypes[3].magicLight = BlockTypeIndex.findType(new BlockName("magic-blue", "mini1"));

        if( false ) {
            // Draw a test room
            echo("Drawing test room");
            def xWorld = new TransactionalWorld(world);

            // Clear an area
            for( int x = 0; x < 20; x++ ) {
                for( int z = 0; z < 20; z++ ) {
                    for( int y = 0; y < 10; y++ ) {
                        xWorld.setWorldCell(loc.add(x, y, z), 0);
                    }
                }
            }

            def testRoom = createRoom(style);
            testRoom.loc = new Vec3i(0, 0, 0);
            testRoom.size = new Vec3i(3, 2, 4);

            def roomLoc = loc.add(1, 0, 1).floor();
            testRoom.insert(roomLoc, xWorld, false);

            xWorld.setWorldCell(loc.add(3, 0, 0), 0);
            xWorld.setWorldCell(loc.add(3, 1, 0), 0);
            xWorld.setWorldCell(loc.add(4, 0, 0), 0);
            xWorld.setWorldCell(loc.add(4, 1, 0), 0);

            //int flame = BlockTypeIndex.findType(new BlockName("fire", "flame-lg"));
            int greenLight = BlockTypeIndex.findType(new BlockName("magic-green", "mini1"));
            int redLight = BlockTypeIndex.findType(new BlockName("magic-red", "mini1"));

            //int magicLight = greenLight;

            //drawMagicSconces.magicLight = redLight;

            //decorateRoomPerimeter(xWorld, roomLoc, testRoom, drawMagicSconces);
            //decorateRoomPerimeter(xWorld, roomLoc, testRoom, drawFlameSconces);
            decorateRoomPerimeter(xWorld, roomLoc, testRoom, lightingTypes[2]);

            xWorld.close();
            return;
        }


        boolean decorateRooms = true;
        boolean clearOldDungeon = false;

        long worldSeed = worldManager.getInfo().getSeed();
        long leafSeed = LeafId.fromWorld(loc).getId();
        long seed = worldSeed + leafSeed; //1;

        int levelCount = 5;
        def starterRoom = createRoom(style);
        starterRoom.type = 3; // entrance room
        starterRoom.loc = new Vec3i(0, 0, 20);
        starterRoom.size = new Vec3i(3, 12, 3);

        def offset = new Vec3i(-starterRoom.loc.x * 2, -starterRoom.size.y + 3, -starterRoom.loc.z * 2);
        offset.x++;
        offset.z -= 3;

        int minFeatureSize = 2;
        int maxFeatureSize = 8;
        int minInfillSize = 2;
        int maxInfillSize = 4;

        def generatorTemplate = createDungeonGenerator(seed);
        generatorTemplate.featureStyle = style;
        generatorTemplate.infillStyle = style2;
        generatorTemplate.tunnelStyle = style;
        generatorTemplate.featureSize = createFeatureSizeFunction(minFeatureSize, maxFeatureSize, 2, 4);
        generatorTemplate.infillSize = createSizeFunction(minInfillSize, maxInfillSize, 2, 3);

        if( clearOldDungeon ) {
            int dirt = BlockTypeIndex.findType(new BlockName("dirt", "cube"));
            int yMax = 8 + levelCount * 7;
            Vec3d base = loc.add(offset).floor().toVec3d();
            base.y -= levelCount * 7;
            for( int y = 0; y < yMax; y++ ) {
                def xWorld = new TransactionalWorld(world);
                for( int x = 0; x < generatorTemplate.xMapSize; x++ ) {
                    for( int z = 0; z < generatorTemplate.zMapSize; z++ ) {
                        xWorld.setWorldCell(base.add(x, y, z), dirt);
                    }
                }
                xWorld.close();
            }

            echo("Wiped out any previous structures at this location");

//            return;
        }

        def levels = [];

        def entrances = [];
        entrances.add(starterRoom);

        def stairs = null;

        def lastGenerator = null;
        for( int level = 0; level < levelCount; level++ ) {
            //def generator = createDungeonGenerator(seed);
            def generator = generatorTemplate.copySetup(seed); //createDungeonGenerator(seed);
            levels.add(generator);

            if( level == 0 ) {
                generator.addEntrance(starterRoom);
            } else {
                // Base it on the last level
                //def stairs = findStairs(lastGenerator, false);
                def exitRooms = stairs*.room;
                for( def exit : exitRooms ) {
                    exit.type = 2; // exit
                    def room = createRoom(style);
                    room.loc = exit.loc.clone();
                    room.size = new Vec3i(3, 6, 3);
                    room.size.x = Math.min(3, exit.size.x);
                    room.size.z = Math.min(3, exit.size.z);
                    room.type = 3; // entrance
                    generator.addEntrance(room);
                    //maxRooms--;
                }
            }

            generator.generateLevel();

            // Now that we've generated the level, do some analysis
            // Finding the stairs will force the path index to be created.
            stairs = findStairs(generator, false);

            // Mark any of the exit rooms as type 2
            for( def path : stairs ) {
                //path.room.type = 2;
                // We need to wait until the next level is generated else
                // we will think we have stairs on the bottom level.

                // We could also mark the adjacent rooms as "near stairs"
                // mmm... or the tunnels.
            }

            // Find the longest infill dead ends and mark
            // them as treasure rooms and their nearest feature
            // room as a boss room.
            // Actually, we'll keep track of all of the dead ends and
            // then decide what to do with them based on room type.
            def deadEnds = [];
            for( def path : generator.pathIndex.values() ) {
                // Only the tips have a depth set
                //if( path.room.type == 1 && path.deadEndDepth > 0 ) {
                if( path.deadEndDepth > 0 ) {
                    deadEnds.add(path);
                }
            }
            // Sort them by length
            deadEnds.sort { p1, p2 -> -(p1.length <=> p2.length) }

            double longest = 0;
            for( def path : deadEnds ) {
                if( longest == 0 ) {
                    longest = path.length;
                }
                // For now, any feature room on a dead-end path is
                // a boss and any infill room is a treasure room.
                for( def step = path; step != null && step.deadEnd; step = step.parent ) {
                    double chance = 1.0; //step.length / longest;
                    double roll = generator.rand.nextDouble();
                    if( roll < chance ) {
                        if( step.room.type == 0 ) {
                            // Boss room
                            step.room.type = 10;
                        } else if( step.room.type == 1 ) {
                            // Treasure room
                            step.room.type = 20;
                        }
                    }
                }
            }

            seed++;
            lastGenerator = generator;
        }

        def dungeonMarker = null;

        for( int level = 0; level < levelCount; level++ ) {
            def generator = levels[level];

            def xWorld = new TransactionalWorld(world);
            def base = loc.add(offset).floor().toVec3d();
            //generator.insert(loc.subtract(0, 10, 40).floor(), xWorld, false);
            generator.insert(base.toVec3i(), xWorld, false);

            xWorld.setWorldCell(loc, 0);
            xWorld.setWorldCell(loc.add(0, 1, 0), 0);
            xWorld.setWorldCell(loc.add(0, 0, -1), 0);
            xWorld.setWorldCell(loc.add(0, 1, -1), 0);

            // Go through each of the rooms and plot a light in their center or corners
            //int flame = BlockTypeIndex.findType(new BlockName("fire", "flame-lg"));
            //int flameSmall = BlockTypeIndex.findType(new BlockName("fire", "flame-sm"));
            int pillar = BlockTypeIndex.findType(new BlockName("bmarble", "vcyl25"));
            //def base = loc.add(offset);
            for( def room : generator.rooms ) {

                // Figure out what kind of lighting to do
                // < 3x3, no special lighting.
                // between 3 and 4, either regular campfire or sconces + small campfire.
                // > 4x4, always some kind of sconce
                // If it's a special room, always flame sconces.
                // Random chance that sconce rooms become grate rooms.
                def lighting = null;
                int centerType = flame;
                if( room.size.x < 3 && room.size.z < 3 ) {
                    // No special lighting
                } else if( room.type == 2 || room.type == 3 ) {
                    // The lighting will be provided
                } else if( room.size.x < 4 && room.size.z < 4 ) {
                    if( generator.rand.nextDouble() < 0.5 ) {
                        // Regular campfire, no exta lighting
//lighting = lightingTypes[0];
                    } else {
                        centerType = smallFlame;
                        //centerType = BlockTypeIndex.findType(new BlockName("magic-white", "mini2"));
                        int randIndex = generator.rand.nextInt(lightingTypes.size());
                        lighting = lightingTypes[randIndex];
                        //lighting = lightingTypes[1];
if( lighting == null ) {
    echo("lighting index is bad:" + randIndex);
}
                    }
                } else {
                    int randIndex = generator.rand.nextInt(lightingTypes.size());
                    lighting = lightingTypes[randIndex];
                }

                if( lighting != null ) {
                    decorateRoomPerimeter(xWorld, base, room, lighting);
                }

                def roomBase = base.add(room.loc.mult(2));
                if( room.type == 20 || room.type == 10 ) {
                    // Plop down some loot
                    int lootCount = 1 + generator.rand.nextInt((int)(room.type / 5));
                    int xRange = (int)Math.max(0, room.size.x * 2 - 4)
                    int zRange = (int)Math.max(0, room.size.z * 2 - 4)
                    if( xRange == 0 && zRange == 0 ) {
                        lootCount = 0;
                    }
                    echo("Creating " + lootCount + " chests");
                    for( int i = 0; i < lootCount; i++ ) {
                        int x = xRange == 0 ? 0 : generator.rand.nextInt(xRange);
                        int z = zRange == 0 ? 0 : generator.rand.nextInt(zRange);
                        double angle = generator.rand.nextDouble() * Math.PI * 2;

                        def placed = createEntity(ObjectTypeInfo.create("WoodChest", entityData));
                        def c = roomBase.add(2 + x, 0, 2 + z);
                        placed.run("initialize");
                        placed.run("place", c, new Quatd().fromAngles(0, angle, 0));
                    }
                }

                def v = roomBase;
                v = v.add(room.size.x, 0, room.size.z);

                // Type 2 rooms don't have a floor
                if( room.type != 2 ) {
                    int lightType = centerType;
                    boolean campfire = true;
                    if( room.type == 10 ) {
                        lightType = BlockTypeIndex.findType(new BlockName("magic-red", "mini2"));
                        v.y--;
                        createGrate(env, v.add(0, 0.75, 0), new Vec3i(1, 0, 0));
                        campfire = false;
                    } else if( room.type == 20 ) {
                        lightType = BlockTypeIndex.findType(new BlockName("magic-green", "mini2"));
                        v.y--;
                        createGrate(env, v.add(0, 0.75, 0), new Vec3i(1, 0, 0));
                        campfire = false;
                    }
                    xWorld.setWorldCell(v, lightType);
                    v = v.add(0.5, 0, 0.5);

                    if( campfire && decorateRooms ) {
                        def placed = createEntity(ObjectTypeInfo.create("Campfire", entityData));
                        placed.run("initialize");
                        placed.run("place", v);
                    }
                }
            }

            if( true ) {
                int light = BlockTypeIndex.findType(new BlockName("magic-white", "mini1"));
                generator.entrances.each { room ->
                    def v = room.loc.mult(2);
                    v = base.add(v.add(0, room.size.y, 0));
                    int bars;
                    if( level == 0 ) {
                        bars = BlockTypeIndex.findType(new BlockName("steel", "hcyl25", 0));
                    } else {
                        bars = 0;
                    }
                    int xSize = Math.min(room.size.x * 2 - 2, 4);
                    int zSize = Math.min(room.size.z * 2 - 2, 4);
                    for( int x = 0; x < xSize; x++ ) {
                        for( int z = 0; z < zSize; z++ ) {
                            xWorld.setWorldCell(v.add(x + 1, 0, z + 1), bars);
                        }
                    }

                    if( level != 0 ) {
                        // Place some lights around the ring
                        xWorld.setWorldCell(v.add(0, 1, 0), pillar);
                        xWorld.setWorldCell(v.add(xSize + 1, 1, zSize + 1), pillar);
                        xWorld.setWorldCell(v.add(0, 1, zSize + 1), pillar);
                        xWorld.setWorldCell(v.add(xSize + 1, 1, 0), pillar);
                        xWorld.setWorldCell(v.add(0, 2, 0), flame);
                        xWorld.setWorldCell(v.add(xSize + 1, 2, zSize + 1), flame);
                        xWorld.setWorldCell(v.add(0, 2, zSize + 1), flame);
                        xWorld.setWorldCell(v.add(xSize + 1, 2, 0), flame);
                    }

                    // Place some lights in the corners in the floor
                    v = base.add(room.loc.mult(2));
                    v.y--;

                    xWorld.setWorldCell(v, light);
                    xWorld.setWorldCell(v.add(room.size.x * 2 - 1, 0, room.size.z * 2 - 1), light);
                    xWorld.setWorldCell(v.add(0, 0, room.size.z * 2 - 1), light);
                    xWorld.setWorldCell(v.add(room.size.x * 2 - 1, 0, 0), light);

                    createGrate(env, v.add(0, 0.75, 0), new Vec3i(1, 0, 0));
                    createGrate(env, v.add(room.size.x * 2 - 1, 0.75, room.size.z * 2 - 1), new Vec3i(1, 0, 0));
                    createGrate(env, v.add(0, 0.75, room.size.z * 2 - 1), new Vec3i(1, 0, 0));
                    createGrate(env, v.add(room.size.x * 2 - 1, 0.75, 0), new Vec3i(1, 0, 0));
                }
            }


            xWorld.close();

            if( true ) {
                // Create a minimap graphic
                int roomType = BlockTypeIndex.findType(new BlockName("W&D", "cube"));
                int tunnelType = BlockTypeIndex.findType(new BlockName("stone", "cube"));
                int woodType = BlockTypeIndex.findType(new BlockName("wood", "cube"));
                int halfStone = BlockTypeIndex.findType(new BlockName("stone", "slab-dn"));
                int stairsType = BlockTypeIndex.findType(new BlockName("wmarble", "cube"));
                int entranceType = BlockTypeIndex.findType(new BlockName("bmarble", "cube"));
                world.setWorldCell(loc.add(-1, 0, 1), halfStone);
                world.setWorldCell(loc.add(-1, 0, 2), halfStone);

                int[] mapTypes = [roomType, tunnelType, stairsType, entranceType];
                def mapCells = generator.createBlockMap(woodType, mapTypes);
                echo("Map size:" + mapCells.getSize());

                def cellId = worldManager.getCellArrayStorage().store(mapCells);
                def mapShape = ShapeInfo.create(cellId.toFileName(), 0.025, entityData);
                def mapLoc = loc.floor().toVec3d().add(-1.5, 0.5 + (levelCount - level) * 0.125, 1);
                echo("Creating " + cellId.toFileName() + " at:" + mapLoc);
                def mapObject = createEntity(
                            ObjectName.create("Map:" + seed, entityData),
                            ObjectTypeInfo.create("MapBoard", entityData),
                            mapShape,
                            new SpawnPosition(mapLoc, new Quatd()),
                            new OwnedBy(activator.id)
                            );

                //createBlueprintFromCells(activator.id, "Map:" + seed, "MapBoard", mapCells, false, 0.05);

                if( level == 0 ) {
                    // Once we know the actual size of the first level
                    // we can create the dungeon marker at the entrance that the player
                    // clicked.
                    dungeonMarker = createEntity(
                            MapMarker.create("dungeon-entrance", (int)(mapCells.getSize().x/2), block.toVec3d()),
                            ObjectTypeInfo.create("DungeonMarker", entityData),
                            new Name("Dungeon:" + mapObject.getId()),
                            new SpawnPosition(block.toVec3d(), new Quatd()),
                            new OwnedBy(activator.id)
                        );
                }


                // Create a map for each level in its entrance rooms... especially
                // helpful for deeper in the dungeon
                generator.entrances.each { room ->
                    def v = room.loc.mult(2).toVec3d();
                    v = base.add(v.add(room.size.x + 1, 0.5, room.size.z));

                    mapObject = createEntity(
                            ObjectName.create("Map:" + seed, entityData),
                            ObjectTypeInfo.create("MapBoard", entityData),
                            mapShape,
                            new SpawnPosition(v, new Quatd()),
                            new OwnedBy(activator.id)
                            );

                }
            }

            offset.y -= 7;

            //def oldStarterRoom = starterRoom;
            //
            //starterRoom = createRoom(style);
            //starterRoom.loc = new Vec3i(0, 0, 20);
            //starterRoom.size = new Vec3i(3, 6, 3);

        }

        if( true ) {
            def bottom = levels[-1];
            int max = 0;
            def maxRoom = null;
            def maxPath = null;

            bottom.pathIndex.entrySet().each { entry ->
                if( entry.value.length > max ) {
                    max = entry.value.length;
                    maxRoom = entry.key;
                    maxPath = entry.value;
                }
            }
            try {
                def v = null;
                if( maxRoom != null ) {
                    def base = loc.add(offset).floor().toVec3d();
                    // Counteract the -=7 we do at the end of each loop above
                    base.y += 7;
                    v = maxRoom.loc.mult(2).toVec3d();
                    v = base.add(v.add(maxRoom.size.x, 0, maxRoom.size.z));
                }
                run("placeEnd", dungeonMarker, v, maxRoom, maxPath);
            } catch( Exception e ) {
                log.error("Failed to place end", e);
            }
        }


        return true;
    }

    addAction("placeEnd") { entranceMarker, roomLoc, maxRoom, maxPath ->
        if( maxRoom == null ) {
            echo("Failed to find ending room");
        } else {
            echo("Last room:" + roomLoc + "  distance:" + maxPath.length);

            // For now we will manually implement a 'drop' behavior
            if( activator[Holding]?.target == self ) {
                echo("Unholding:" + self);
                activator.remove(Holding);
            }

            // We only have to move the object to the end
            def v = roomLoc.add(0.5, 0, 0.5);
            if( self.type.shapeBaseCenter ) {
                v -= self.type.shapeBaseCenter;
            }
            self << new SpawnPosition(v, new Quatd());

            // Add an exit platform to go with it
            def placed = createEntity(
                ObjectTypeInfo.create("ExitPlatform"),
                new Mass(0),
                EntityLink.create(self, entranceMarker, "warp")
                );
            def v2 = roomLoc.add(0.5, 0, 0.5);
            v2.addLocal(1, 0, 1);
            placed.run("initialize");
            placed.run("place", v2, new Quatd());
        }
    }

    //addAction("Take") { ->
    //    echo("Picking up object " + object.name);
    //
    //    echo("Activator:" + activator);
    //
    //    //object <<
    //    // FIXME: hook up activator.id a little more consistently
    //    def backpack = findItem(activator.id, "Backpack");
    //
    //    def shape = object[ShapeInfo];
    //
    //    object << new ContainedIn(activator.id, backpack, 18, 8);
    //    object << object.type.getTypeVar("shape", null)
    //    object << new Mass(0.1);
    //    object.remove(SpawnPosition);
    //}.onlyIf { ->
    //    // Can only pick up the object if it originally belonged to
    //    // the player and is not currently contained
    //    if( object[ContainedIn] ) {
    //        return false;
    //    }
    //    if( !activator.isSameEntity(object[OwnedBy]?.owner) ) {
    //        return false;
    //    }
    //    return true;
    //}

    //addAction("contextAction") { objectHit ->
    //    def clicked = objectHit.entityId;
    //    def pos = objectHit.location;
    //
    //    log.info("contextAction:" + clicked + ", " + pos);
    //    def name = clicked.name?:"Unknown";
    //
    //    // Here we want to let the player run some actions on the object
    //
    //    // Is it our object?
    //    def creator = clicked[CreatedBy]?.creatorId;
    //    log.info("Player:" + activator.id + " creator:" + creator);
    //
    //    def options = [];
    //    options.add(new Option("Look"));
    //
    //    session.showPrompt(clicked, PromptType.List, name);
    //    session.showOptions(clicked, options);
    //}
}

createType("DungeonWand").with {
    supertypes("DungeonGenerator");
}

