/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

// Some essentially temporary test objects for testing inventory, item
// holding, item manipulation, and other test things.


import com.simsilica.mworld.tile.Resolution;
import com.simsilica.mworld.tile.tree.*;


findTree = { Vec3i block ->
    def tileId = TileId.fromWorld(block);
    def treeTile = world.getTrees(tileId, null);
    def tileLoc = block - tileId.getWorld(null);
    //treeTile.getTrees(block).each {
    //}
    return treeTile.getTrees(block).find { tree -> tree.x == tileLoc.x && tree.z == tileLoc.z }
}

createType("SwingableItem").with {
    supertypes("BaseItem");

    setTypeVar("activityTypes", ["Swing Left", "Swing Down"]);

    // Note: onEquip is not called when we first load the game and the tool is already
    // equipped.  So we really need some other action that can be invoked in both cases.
    addAction("onEquip") { ->
        log.info("Subclass equip");

        // Need to run any super-class onEquip... but our current version
        // of this will fail if the action is not defined
        //run(type("BaseItem"), "onEquip");

        // Set our focus
        run("rotate", 0);
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);

        def activityTypes = getTypeVar("activityTypes", null);
        int index = env.getVar("activityIndex", 0);
        index = (index + delta) % activityTypes.size();
        if( index < 0 ) {
            index += activityTypes.size();
        }
        def focus = activityTypes[index];
        self << ObjectFocus.stringValue(focus, entityData);
        env.setVar("activityIndex", index);
        env.setVar("currentActivity", focus);
    }

    addAction("mainPress") { ->
        log.info("start chopping:nothing");
        def activity = env.getVar("currentActivity", "Swing Left");
        activator.body.driver.startActivity(activity);
    }

    addAction("mainClick") { ->
        log.info("stop chopping:nothing");
        def activity = env.getVar("currentActivity", "Swing Left");
        activator.body.driver.endActivity(activity);
    }

    addAction("mainPressObject") { objectHit ->
        superRun(objectHit);
        log.info("start chopping:" + objectHit);
        def activity = env.getVar("currentActivity", "Swing Left");
        activator.body.driver.startActivity(activity);
    }

    addAction("mainClickObject") { objectHit ->
        superRun(objectHit);
        log.info("stop chopping:" + objectHit);
        def activity = env.getVar("currentActivity", "Swing Left");
        activator.body.driver.endActivity(activity);
    }

    addAction("mainPressBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("start chopping:" + block);
        def activity = env.getVar("currentActivity", "Swing Left");
        activator.body.driver.startActivity(activity);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("chop:" + loc);

        def activity = env.getVar("currentActivity", "Swing Left");
        activator.body.driver.endActivity(activity);
    }
}


createType("Axe").with {
    supertypes("SwingableItem");

    setTypeVar("manipulator", "Hand");

    setTypeVar("description",
        "This was axe was created by %creator with ID %id.");
    // X is along the axe blade, y is how far away from the palm it is, z is the length of the handle
    setTypeVar("handleLocation", new Vec3d(6, 1, 5));
    setTypeVar("handleRotation", new Quatd().fromAngles(0, 0, -Math.PI * 0.5));

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/axe1.blocks", standardItemScale));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    //addAction("mainPress") { ->
    //    log.info("start chopping:nothing");
    //    activator.body.driver.startActivity("leftSwing");
    //}
    //
    //addAction("mainClick") { ->
    //    log.info("stop chopping:nothing");
    //    activator.body.driver.endActivity("leftSwing");
    //}
    //
    //addAction("mainPressObject") { objectHit ->
    //    superRun(objectHit);
    //    log.info("start chopping:" + objectHit);
    //    activator.body.driver.startActivity("leftSwing");
    //}
    //
    //addAction("mainClickObject") { objectHit ->
    //    superRun(objectHit);
    //    log.info("stop chopping:" + objectHit);
    //    activator.body.driver.endActivity("leftSwing");
    //}
    //
    //addAction("mainPressBlock") { blockHit ->
    //    Vec3d loc = blockHit.location;
    //    Vec3i block = blockHit.block;
    //    log.info("start chopping:" + block);
    //    activator.body.driver.startActivity("leftSwing");
    //}

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("chop:" + loc);

        superRun(blockHit);
        //activator.body.driver.endActivity("leftSwing");

        int cell = world.getWorldCell(block as Vec3d);
        int type = MaskUtils.getType(cell);
        int sides = MaskUtils.getSideMask(cell);

        def genType = CellGenType.getType(cell);

        echo("Chop:" + genType);

        def found = findTree(block);
        echo("Tree:" + found);
        if( found == null ) {
            return;
        }

        // The below would be nearly 100% accurate but would require
        // straddling columns to get the full tree and requires shuffling
        // lots of additional leaf data around just to pull out a single
        // tree.  Instead, we will let TreeType guess if block type is
        // part of it or not.
        //def colId = ColumnId.fromWorld(block);
        //def colData = new ColumnData(colId, GameConstants.MAX_BUILD_HEIGHT / 32);
        //
        //def tileId = TileId.fromWorld(block);
        //def treeTile = world.getTrees(tileId, null);
        //
        //def localTrees = treeTile.getTrees(colId);
        //if( localTrees.isEmpty() ) {
        //    return false;
        //}
        //
        //Random rand = new Random(columnId.getId()); // consistent random per column
        //def cells = new TileColumnCells(colData);
        //for( Tree tree : localTrees ) {
        //    //log.info("  insert:" + tree + " into:" + col.getColumnId());
        //    if( tree == found ) {
        //        // Clear everything so far
        //        colData.leafs.each { leaf ->
        //            leaf.rawCells.clear();
        //        }
        //    }
        //    tree.type.insertTree(tree, cells, rand);
        //    if( tree == found ) {
        //        break;
        //    }
        //}

        // Guess which blocks are part of this tree
        def tileId = TileId.fromWorld(block);
        def tileLoc = block - tileId.getWorld(null);

        int size = 1 + found.radius * 2;
        int xBase = found.x - found.radius;
        int yBase = found.y;
        int zBase = found.z - found.radius;

        CellArray cells = new CellArray(size, found.height, size);

        Vec3d base = new Vec3d(xBase, yBase, zBase);
        //log.info("base:" + base);
        base = base + tileId.getWorld(null);
        //log.info("world base:"+ base);

        for( int y = 0; y < found.height; y++ ) {
            for( int z = 0; z < size; z++ ) {
                def sb = new StringBuilder();
                for( int x = 0; x < size; x++ ) {
                    cell = world.getWorldCell(base.add(x, y, z));
                    genType = CellGenType.getType(cell);
                    if( genType == CellGenType.Flora ) {
                        type = MaskUtils.getType(cell);
                        if( found.type.isBlock(found, x - found.radius, y, z - found.radius, type) ) {
                            //sb.append("[1]");
                            cells.setCell(x, y, z, type);
                        } else {
                            //sb.append("[0]");
                        }
                    } else {
                        //sb.append("[0]");
                    }
                }
                //log.info("[" + y + "][" + z + "] = " + sb);
            }
        }

        // Make sure the cells have accurate masks
        MaskUtils.calculateSideMasks(cells);

        //createBlueprintFromCells(activator.id, "Tree:" + block.x + "x" + block.z, "Tree", cells, false, 0.25);
        createBlueprintFromCells(activator.id, "Tree:" + block.x + "x" + block.z, "Tree", cells, false, 1);
    }


    addAction("altClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        echo("Loc:" + loc + " block:" + block);

        def filter = fieldEquals(BlueprintInfo, "parent", activator.id);

        // Get the blueprints as a tree set just so they are ordered by
        // ID... which is roughly by creation time.
        def blueprints = findEntities(filter, BlueprintInfo) as TreeSet;

        def tree = blueprints.find { it.name.startsWith("Tree") };

        echo("Tree blueprint:" + tree);

        def bp = tree;

        def shape = bp[ShapeInfo];

        // Need to get the size so we can find its base's center.
        // We probably should keep this as part of the object type or even a separate
        // component.
        def cells = shape.loadCells();

        def worldEdits = new WorldEdits();

        for( int x = 0; x < cells.sizeX; x++ ) {
            for( int z = 0; z < cells.sizeZ; z++ ) {
                for( int y = 0; y < cells.sizeY; y++ ) {
                    int val = cells.getCell(x, y, z);
                    world.setWorldCell(loc.add(x, y, z), val, worldEdits);
                }
            }
        }

        world.deliverEvents(worldEdits);

        echo("World implemention:" + world.getClass() + "  thread:" + Thread.currentThread());

        return true;

    }

    //addAction("startSwing") { facing ->
    //    log.info("startSwing:" + facing);
    //}
    //
    //addAction("swing") { facing ->
    //    log.info("swing:" + facing);
    //}
    //
    //addAction("stopSwing") { facing ->
    //    log.info("stopSwing:" + facing);
    //}

    addAction("onRemove") { ->
        log.info("onRemove()");

        def on = self[ObjectName];
        log.info("removing:" + on + " " + on.name + " self:" + self.name);
        if( "Test Axe" == on.name ) {
            log.info("calling findEntities()");
            int count = findEntities(Filters.fieldEquals(ObjectName.class, "nameId", on.nameId)).size();
            log.info("test axe count:" + (count - 1));
            echo("" + (count - 1) + " test axes remaining.");
        }

        log.info("calling superRun()");
        superRun();
        log.info("done calling superRun()");
    }
}

createType("Sword").with {
    supertypes("SwingableItem");

    setTypeVar("manipulator", "Hand");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/sword1.blocks", standardItemScale));
    // The sword requires some custom sizing to make sense in inventory
    def size = getTypeVar("shape").loadCells().size.toVec3d();
    size.x = 3;
    size *= standardItemScale;
    setTypeVar("volume", new ObjectVolume(size));

    setTypeVar("description",
        "This %name was created by %creator with ID %id.");
    // X is along the hilt, y is how far away from the palm it is, z is the length of the blade
    setTypeVar("handleLocation", new Vec3d(3, 1, 3.5));
    setTypeVar("handleRotation", new Quatd().fromAngles(0, 0, -Math.PI * 0.5));

    //addAction("startSwing") { facing ->
    //    log.info("startSwing:" + facing);
    //}
    //
    //addAction("swing") { facing ->
    //    log.info("swing:" + facing);
    //}
    //
    //addAction("stopSwing") { facing ->
    //    log.info("stopSwing:" + facing);
    //}
}

createType("PickAxe").with {
    supertypes("SwingableItem");

    setTypeVar("manipulator", "Hand");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/pick-axe.carved.blocks", standardItemScale));
    // The sword requires some custom sizing to make sense in inventory
    def size = getTypeVar("shape").loadCells().size.toVec3d();
    //size.x = 3;
    size *= standardItemScale;
    setTypeVar("volume", new ObjectVolume(size));

    setTypeVar("description",
        "This %name was created by %creator with ID %id.");
    //setTypeVar("handleLocation", new Vec3d(4, 3, 1));
    // Z is distance from palm, y is along the handle
    setTypeVar("handleLocation", new Vec3d(4, 2, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, -Math.PI * 0.5));
}


createType("Book").with {
    supertypes("BaseItem");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/book.blocks", standardItemScale));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    setTypeVar("description",
        "This was book was created by %creator with ID %id.");
    setTypeVar("handleLocation", new Vec3d(0, 4, 1));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, Math.PI, Math.PI * 0.5));
}

createType("GourdGun").with {
    supertypes("BaseItem");

    //setTypeVar("shape",
    //           ShapeInfo.create("/Models/items/gourd-gun.assembly", standardItemScale));
    // FIXME: fix player inventory to be able to show assemblies
    setTypeVar("shape",
               ShapeInfo.create("/Models/items/black-steel-gun-receiver1.carved.blocks", standardItemScale));
    // Cannot easily get the volume for assemblies so we will
    // hard code it for the moment.  FIXME: implement assembly volume detection.
    //setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));
    setTypeVar("volume",
        new ObjectVolume(new Vec3d(10 * standardItemScale, 3 * standardItemScale, 1 * standardItemScale)));

    setTypeVar("description",
        "This gun magically shoots small gourds.");
    setTypeVar("handleLocation", new Vec3d(1, 1, 1));
    setTypeVar("handleRotation", new Quatd().fromAngles(0, Math.PI * 0.5, Math.PI * 0.65));

    addAction("mainClickBlock") { blockHit ->
        log.info("-------------------------GourdGun.mainClickBlock(" + blockHit + ")");
        gourdGunTestClick.call(session);
    }

    addAction("mainClickObject") { objectHit ->
        log.info("-------------------------GourdGun.mainClickObject(" + blockHit + ")");
        gourdGunTestClick.call(session);
    }

    addAction("mainClick") { ->
        log.info("-------------------------GourdGun.mainClick()");
        gourdGunTestClick.call(session);
    }

}


createType("TestObject").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Digger");
    setTypeVar("description",
        "This was object was created by %creator with ID %id.");

    addAction("Test") { ->
        //log.info("" + this + "." + getName() + "(" + Arrays.asList(args) + ")");
        log.info("" + this + "." + "Test" + "(" + ")");
        log.info("owner:" + owner + " delegate:" + delegate);
        log.info("Activator:" + activator);
        log.info("Activator:" + getActivator());
        //log.info("variables:" + getVariables()); // from 'delegate'
        log.info("activator.vars:" + activator.vars); // from 'delegate'
        log.info("self:" + getSelf()); // from 'this'

        // This goes to a global binding
        testProperty = 123;

        //log.info("variables:" + getVariables()); // from 'delegate'

        //activator.target = "Test Target";
        //log.info("activator.vars:" + activator.vars); // from 'delegate'
        //log.info("target:" + getTarget()); // from 'this'?  Yep.

        //log.info("vars:" + activator.vars);
        //log.info("test:" + activator.vars.get("selectedType"));
        //log.info("test1:" + activator.getProperty("selectedType"));
        //log.info("test2:" + activator.selectedType);
        //activator.foo = 123;
        //log.info("test3:" + activator.foo);
        return true;
    }

    addAction("Test2") { String name ->
        log.info("Test2:" + name);
        return true;
    }

    addAction("Test3") { ->
        log.info("Calling delegate version of run()");
        // This one calls it directly on ActionContext which is our 'this'
        //object.run(env, "Test2", "This is a test")
        self.run("Test2", "This is a test")
        log.info("Calling non-delegate version of run()");
        run("Test2", "This is a different test")
        //run("BaseTest")

        def manipulator = self.getType().getTypeVar("manipulator", "Default");

        log.info("Manipulator:" + manipulator);
        log.info("test:" + (self.manipulator?:"Default"));

        log.info("holding1:" + Holding.create(self.getTarget(), manipulator, entityData));
        log.info("holding2:" + Holding.create(self.getTarget(), manipulator));
        log.info("holding3:" + Holding.create(self, manipulator));
        log.info("holding4:" + Holding.create(activator, manipulator));

        log.info("contains1:" + activator.contains(self));

        // What about running an action from a different type or one of
        // our super types?
        log.info("Running it the hard way--------------------");
        type("BaseObject").findAction("Look").run(env, context);

        log.info("Running it the easy way--------------------");
        run(type("BaseObject"), "Look");

        return true;
    }

    addGetter("test") { -> //target ->
        log.info("getter-test");
        log.info("  getter-ed:" + system(EntityData));
        log.info("  getter-self:" + self);
        log.info("  getter-activator:" + activator);
        log.info("  owner:" + owner);
        log.info("  delegate:" + delegate);
        log.info("  this:" + this);
        return "foo-123"
    }

    //addGetter("name") { ->
    //    log.info("target:" + target);
    //    log.info("delegate:" + delegate);
    //    def on = delegate[ObjectName];
    //    log.info("name test:" + on);
    //    log.info("delegate test:" + delegate[ObjectName]);
    //    if( on != null ) {
    //        return on.getName(entityData);
    //    }
    //    def n = delegate[Name];
    //    if( n != null ) {
    //        return n.getName();
    //    }
    //    return "Entity@" + target.getId();
    //}

}

import mythruna.es.debug.*;
import com.simsilica.mblock.db.*;

placeableStructures = [:] as TreeMap;
def extractsDir = new File("extracts");
if( extractsDir.exists() ) {
    extractsDir.eachFile { f ->
        def blocks = com.simsilica.mblock.io.BlocksFileFormat.readBlocks(f, true);

        Vec3i[] range = CellUtils.calculateSize(blocks.cells);

        int xs = range[1].x - range[0].x + 1;
        int ys = range[1].y - range[0].y + 1;
        int zs = range[1].z - range[0].z + 1;

        CellArray cells = new CellArray(xs, ys, zs);

        CellUtils.copyData(blocks.cells, range[0].x, range[0].y, range[0].z,
                           cells, 0, 0, 0,
                           xs, ys, zs);
        CellArrayId cellId = worldManager.getCellArrayStorage().store(cells);

        log.info("stored extract:" + f + " into cellID:" + cellId);

        placeableStructures.put(f.name, cellId);
    }
}

createType("Extractor").with {
    supertypes("ObjectTool");

    setTypeVar("manipulator", "Creator");
    setTypeVar("description",
              "The extractor wand."
            );
    setTypeVar("handleLocation", new Vec3d(1.5, 1.5, 3));

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/wand3.blocks", standardItemScale * 0.5));
    setTypeVar("volume", new ObjectVolume(new Vec3d(3, 3, 13).multLocal(standardItemScale * 0.5)));

    addAction("showDefaultBlockAction")  { blockHit ->

        def cellGenType = CellGenType.getType(blockHit.value);
        if( cellGenType == CellGenType.None ) {
            session.showPrompt(null, PromptType.Flyover, "Extract");
        } else {
            session.showPrompt(null, PromptType.Flyover, "Reset Column");
        }
    }

    // Note: onEquip is not called when we first load the game and the tool is already
    // equipped.  So we really need some other action that can be invoked in both cases.
    addAction("onEquip") { ->
        // Set our focus
        run("rotate", 0);
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);

        def modes = getTypeVar("extractorModes", null);
        if( modes == null ) {
            modes = new ArrayList<>(placeableStructures.keySet());
            setTypeVar("extractorModes", modes);
        }
        if( modes.isEmpty() ) {
            self << ObjectFocus.stringValue("None", entityData);
            env.setVar("extractorFocus", -1);
            env.setVar("extractorName", null);
            return;
        }

        int index = env.getVar("extractorFocus", 0);
        index = (index + delta) % modes.size();
        if( index < 0 ) {
            index += modes.size();
        }

        def focus = modes[index];
        self << ObjectFocus.stringValue(focus, entityData);
        env.setVar("extractorFocus", index);
        env.setVar("extractorName", focus);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;

        def cellGenType = CellGenType.getType(blockHit.value);
        if( cellGenType == CellGenType.None ) {
            echo("Extracting starting at:" + block)
            extractorTestClick.call(session, blockHit);
        } else {
            def colId = ColumnId.fromWorld(loc);
            echo("Resetting world at:" + colId);
            worldManager.regenerateColumn(colId);
        }
    }

    addAction("place") { blockHit ->

        def name = env.getVar("extractorName", null);
        if( name == null ) {
            echo("No placeable structures loaded.");
            return;
        }

        Vec3i block = blockHit.block;
        def cellId = placeableStructures[name];

        echo("Place:" + name + " at:" + block);

        def entity = type("PlacedStructure").newInstance(
                    ShapeInfo.create(cellId.toFileName(), 1),
                    ObjectName.create("Placed Structure", entityData),
                    new Mass(0),
                    new CreatedBy(activator.id)
                );
        entity.run("placeInWorld", block.toVec3d(), new Quatd());


        //Vec3d loc = block.add(0, 5, 0).toVec3d();
        //
        //def entity = type("Airship").newInstance(
        //        ObjectName.create("Small Airship", entityData),
        //        new Mass(0),  // static for now
        //        new CreatedBy(activator.id),
        //        new LargeObject(),
        //
        //        // We'll force a grid cell since right now the object
        //        // is static.
        //        new LargeGridCell(GameConstants.LARGE_OBJECT_GRID.worldToId(loc))
        //    );
        //entity.run("placeInWorld", loc, new Quatd());
    }

    addAction("snap") { clicked ->

        // Normalize the spawn position to the closest block and
        // 90 degree rotation
        Vec3d pos = clicked.location;
        pos = pos.round().toVec3d();

        Vec3d dir = clicked.orientation.mult(Vec3d.UNIT_Z);
        if( Math.abs(dir.x) > Math.abs(dir.z) ) {
            dir.x = Math.signum(dir.x);
            dir.z = 0;
        } else {
            dir.x = 0;
            dir.z = Math.signum(dir.z);
        }
        double rads = Math.atan2(dir.x, dir.z);

        clicked << new SpawnPosition(pos, new Quatd().fromAngles(0, rads, 0));
    }
}

createType("PlacedStructure").with {
    supertypes("BaseObject")

    setTypeVar("description",
              "A temporary placed structure."
            );

    addAction("Build") { ->

        def name = self[ShapeInfo].shapeName;
        echo("name:" + name);
        def cellId = CellArrayId.fromFileName(name);
        echo("cellId:" + cellId);
        def cells = worldManager.getCellArrayStorage().get(cellId);
        def originalSize = cells.size;
        echo("cells:" + cells);

        // Find some normalized direction to rotate the cells
        def orientation = self.orientation;
        Vec3d zDir = orientation.mult(Vec3d.UNIT_Z);
        if( Math.abs(zDir.x) > Math.abs(zDir.z) ) {
            zDir.x = Math.signum(zDir.x);
            zDir.z = 0;
        } else {
            zDir.x = 0;
            zDir.z = Math.signum(zDir.z);
        }
        double rads = Math.atan2(zDir.x, zDir.z);
        echo("rotation:" + Math.toDegrees(rads));
        log.info("rotation:" + Math.toDegrees(rads));

        echo("size before:" + cells.size);
        log.info("size before:" + cells.size);

        int rotate = (int)(Math.toDegrees(rads)/90);
        echo("rotate:" + rotate);
        log.info("rotate:" + rotate);
        cells = CellUtils.rotate(cells, -rotate);

        echo("size after:" + cells.size);
        log.info("size after:" + cells.size);

        if( true ) {
            int xs = cells.sizeX;
            int ys = cells.sizeY;
            int zs = cells.sizeZ;

            def loc = self.location.floor().toVec3d();

            // After we rotate then the target location will be different, too.
            // Whatever the minimum of the rotated box would be.
            Vec3d xCorner = orientation.mult(Vec3d.UNIT_X.mult(originalSize.x));
            Vec3d zCorner = orientation.mult(Vec3d.UNIT_Z.mult(originalSize.z));
            echo("xCorner:" + xCorner + "  zCorner:" + zCorner);
            log.info("xCorner:" + xCorner + "  zCorner:" + zCorner);

            def target = loc.add(xCorner).add(zCorner);
            target = target.minLocal(loc);

            echo("target:" + target + "  original:" + loc);
            log.info("target:" + target + "  original:" + loc);

            /*for( int y = 0; y < ys; y++ ) {
                world.setWorldCell(loc.add(0, y, 0), 1);
                world.setWorldCell(loc.add(xCorner).add(0, y, 0), 3);
                world.setWorldCell(loc.add(zCorner).add(0, y, 0), 4);
                world.setWorldCell(loc.add(xCorner).add(zCorner).add(0, y, 0), 5);
            }*/

            for( int x = 0; x < xs; x++ ) {
                for( int z = 0; z < zs; z++ ) {
                    for( int y = 0; y < ys; y++ ) {
                        int cell = cells.getCell(x, y, z);
                        world.setWorldCell(target.add(x, y, z), cell);
                    }
                }
            }
        }

    }.onlyIf { ->
        // Only if it's the structure we placed
        return activator.id == self[CreatedBy]?.creatorId;
    }
}

createType("Airship").with {
    supertypes("BaseObject")

    setTypeVar("shape",
               ShapeInfo.create("/Models/objects/airship-small1.assembly", 1.0));

    setTypeVar("description",
              "An airship!"
            );

}


createType("TestWand").with {
    supertypes("BaseItem");

    setTypeVar("manipulator", "Creator");
    setTypeVar("handleLocation", new Vec3d(0, 3, 0));
    setTypeVar("handleRotation", new Quatd().fromAngles(Math.PI * 0.5, 0, 0));
    setTypeVar("description",
               "A Test Wand."
             );

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/metal-wand.blocks", standardItemScale * 0.5));
    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));
    setTypeVar("adminModes", ["Sound", "Sphere", "Projectile", "Campfire",
                         "Test Chest", "Test Backpack", "Test Pouch", "Test Sword", "Debug Arrow",
                         "Chip Test", "Light On", "Light Off", "Airship",
                         "Random Human", "Random Human Male", "Random Human Female",
                         "Random Elf Male", "Random Elf Female",
                         "Random Human Male Giant",
                         "Random Butterfly", "Gaefen", "Butterfly", "Dog", "Bird", "Crow"])

    setTypeVar("modes", ["Sound", "Sphere", "Projectile", "Campfire",
                         "Random Butterfly", "Gaefen", "Butterfly", "Dog", "Bird", "Crow"])

    addAction("showDefaultBlockAction")  { blockHit ->
        //log.info("base-objects: showDefaultBlockAction:" + blockHit);
        def mode = env.getVar("testWandMode", "Gaefen");
        if( "Chip Test" == mode ) {
            session.showPrompt(null, PromptType.Flyover, "Chip");
        } else {
            session.showPrompt(null, PromptType.Flyover, "Climate Info");
        }
    }

    // Note: onEquip is not called when we first load the game and the tool is already
    // equipped.  So we really need some other action that can be invoked in both cases.
    addAction("onEquip") { ->
        log.info("Subclass equip");

        // Need to run any super-class onEquip... but our current version
        // of this will fail if the action is not defined
        //run(type("BaseItem"), "onEquip");

        // Set our focus
        run("rotate", 0);
    }

    addAction("rotate") { delta ->
        log.info("rotate:" + delta);

        def modes = getTypeVar("modes", null);
        def perms = account.getProperty("permissions", List.class);
        if( perms != null && (perms?.contains("admin") || perms?.contains("singlePlayer")) ) {
            modes = getTypeVar("adminModes", null);
        }

        int index = env.getVar("testWandFocus", 0);
        index = (index + delta) % modes.size();
        if( index < 0 ) {
            index += modes.size();
        }
        def focus = modes[index];
        self << ObjectFocus.stringValue(focus, entityData);
        env.setVar("testWandFocus", index);
        env.setVar("testWandMode", focus);
    }

    addAction("mainClickObject") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;
        log.info("mainClickObject(" + clicked + ", " + pos + ")");
        def mode = env.getVar("testWandMode", "Gaefen");
        if( "Chip Test" == mode ) {
            ////system(DestructibleObjectSystem.class).chopObject(clicked, pos, null);
            //if( objectHit.entityId.hasAction("chopObject", objectHit) ) {
            objectHit.entityId.runIfExists("chopObject", objectHit);
            return;
        }
        superRun(objectHit);
    }

    addAction("mainClickBlock") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        echo("Clicked:" + block)

        def mode = env.getVar("testWandMode", "Gaefen");
        if( "Chip Test" == mode ) {

            system(DestructibleObjectSystem.class).chopBlock(loc, block, blockHit.normal, null);

            //int cell = world.getWorldCell(block as Vec3d);
            //int type = MaskUtils.getType(cell);
            //
            //def entity = createEntity(
            //    //new SpawnPosition(activator.pos, new Quatd()),
            //    new SpawnPosition(block as Vec3d, new Quatd()),
            //    ShapeInfo.create("c_" + type + ".fax", 0.25),
            //    //ShapeInfo.create("/Models/objects/speakers.blocks", 0.25),
            //    new Mass(0),
            //    new CreatedBy(activator.id),
            //    new DestructibleBlock(type, future(30)),
            //    Decay.seconds(60)
            //);
            //
            //world.setWorldCell(block as Vec3d, 0);

            return;
        }
        if( "Light On" == mode ) {
            // T001108 testing
            //def testEntity = type("TestType").newInstance();
            //echo("created:" + testEntity + "  type:" + testEntity[ObjectTypeInfo]);
            //testEntity.run("testAction");
            //echo("tested:" + testEntity + "  type:" + testEntity[ObjectTypeInfo]);

            world.debugSetWorldLight(loc.add(blockHit.normal.mult(0.01)), LightUtils.toLight(15, 15, 15, 15));
            return;
        }
        if( "Light Off" == mode ) {
            world.debugSetWorldLight(loc.add(blockHit.normal.mult(0.01)), LightUtils.toLight(0, 0, 0, 0));
            return;
        }

        //double temperature = worldManager.worldFractal.getTemperature(block.x, block.y, block.z);
        //double weather = worldManager.worldFractal.getWeather(block.x, block.y, block.z);
        //double ecology = worldManager.worldFractal.getEcology(block.x, block.y, block.z);
        //
        //// Adjust temperature for elevation
        ////int seaLevel = 128;
        ////int treeLine = seaLevel + 490;
        ////double height = Math.abs(block.y - seaLevel) / (double)treeLine;
        ////echo("height scale:" + height);
        ////double baselineHeight = 0.2;
        ////double effectiveTemperature = temperature;
        ////if( height > baselineHeight ) {
        ////    double heightEffect = (height - baselineHeight) / (1.0 - baselineHeight);
        ////    double temperatureAdjust = heightEffect * 0.4;
        ////    effectiveTemperature = temperature - temperatureAdjust;
        ////}
        //double effectiveTemperature = worldManager.worldFractal.getAdjustedTemperature(block.x, block.y, block.z);
        //
        //// Adjust precipiation by temperature if we are in particularly
        //// hot climates.  Anything above 40C is desert and will significantly
        //// affect whether precipitation accumulates.
        //// 40C in the range of -20 to 50 should be 0.857
        ////double effectiveWeather = weather;
        ////if( effectiveTemperature > 0.857 ) {
        ////    double weatherAdjust = (effectiveTemperature - 0.857) / (1.0 - 0.857);
        ////    weatherAdjust = 1 - weatherAdjust;
        ////    // Weather adjust should be in 0..1 where 0 is no precip. and
        ////    // 1 is max precip
        ////    effectiveWeather *= weatherAdjust;
        ////}
        //double effectiveWeather = worldManager.worldFractal.getAdjustedWeather(block.x, block.y, block.z, effectiveTemperature);
        //
        ////// Temper ecology by how wet things are on average
        ////double effectiveEcology = ecology;
        ////if( effectiveWeather < 0.25 ) {
        ////    effectiveEcology *= (effectiveWeather / 0.25);
        ////} else if( effectiveWeather > 0.75 ) {
        ////    // Increase ecology, up to double
        ////    effectiveEcology *= 1 + ((effectiveWeather - 0.75)/0.25);
        ////    if( effectiveEcology > 1 ) {
        ////        effectiveEcology = 1;
        ////    }
        ////}
        //double effectiveEcology = worldManager.worldFractal.getAdjustedEcology(block.x, block.y, block.z, effectiveWeather);

        // Get the block and stuff itself
        def blockType = BlockTypeIndex.get(blockHit.type);
        def lightValue = world.getWorldLight(blockHit.location);
        echo("Block:" + blockType.name + "  light:" + String.format("%x", LightUtils.getLight(lightValue)));

        def info = worldManager.worldFractal.getBioInfo(block.x, block.y, block.z, null);

        echo("Temperature:" + info.baseTemperature + "  effective:" + info.temperature);
        echo("Weather:" + info.basePrecipitation + "  effective:" + info.precipitation);
        echo("Climate:" + info.climate + "  Soil Quality:" + info.soilQuality);
        echo("Vegetation:" + info.baseVegetationLevel + "  effective:" + info.vegetationLevel);

        int foliage = (int)Math.round(info.vegetationLevel * 3) & 0x3;
        int wetness = (int)Math.round(info.precipitation * 3) & 0x3;
        echo("foliage level:" + foliage + "/3   wetness level:" + wetness + "/3");

        def tileId = TileId.fromWorld(block.x, block.y, block.z);
        def terrain = world.getTerrainImage(tileId, TerrainImageType.Terrain, null);
        def origin = tileId.getWorld(null);
        int type = terrain.getType((int)(block.x - origin.x), (int)(block.z - origin.z));
        byte light = terrain.getLight((int)(block.x - origin.x), (int)(block.z - origin.z));
        int baseType = TerrainTypes.getBaseType(type);
        echo("Encoded type:" + baseType + " foliage level:" + TerrainTypes.getFoliageLevel(type) + "/3 wetness level:" + TerrainTypes.getWetnessLevel(type) + "/3 frozen:" + TerrainTypes.isFrozen(type));

        double c = (info.temperature * 70) - 20;
        double f = c * 1.8 + 32;
        double p = info.precipitation * 100;
        echo(String.format("%.01f C (%.01f F)  Avg. rainfall: %.02f cm (%.02f inches)", c, f, p*2.5, p));

        // Show what the tree generator would use for type selection
        int TEMPERATURE_ZONES = 7;
        double VEG1 = 0.2;
        double VEG2 = 0.25;
        double VEG3 = 0.4;
        double VEG4 = 0.5;
        double VEG5 = 0.75;
        double VEG6 = 1;

        double PROB1 = 0.001;
        double PROB2 = 0.003;
        double PROB3 = 0.01;
        double PROB4 = 0.1;
        double PROB5 = 0.75;
        double PROB6 = 1.0;
        int treeLine = 490 + GameConstants.SEA_LEVEL;
        def density = "LOW";
        double chance = 0;
        double veg = 0;

        if( baseType == TerrainTypes.TYPE_SAND ) {
            chance = info.vegetationLevel * 0.001;
        } else {
            veg = info.vegetationLevel;

            if( veg < VEG1 ) {
                chance = (veg/VEG1);
                chance = chance * PROB1;
                density = "LOW";
            } else if( veg < VEG2 ) {
                chance = (veg - VEG1)/(VEG2 - VEG1);
                chance = PROB1 + (chance * (PROB2 - PROB1));
                density = "LOW";
            } else if( veg < VEG3 ) {
                chance = (veg - VEG2)/(VEG3 - VEG2);
                chance = PROB2 + (chance * (PROB3 - PROB2));
                density = "MEDIUM";
            } else if( veg < VEG4 ) {
                chance = (veg - VEG3)/(VEG4 - VEG3);
                chance = PROB3 + (chance * (PROB4 - PROB3));
                density = "MEDIUM";
            } else if( veg < VEG5 ) {
                chance = (veg - VEG4)/(VEG5 - VEG4);
                chance = PROB4 + (chance * (PROB5 - PROB4));
                density = "HIGH";
            } else if( veg < VEG6 ) {
                chance = (veg - VEG5)/(VEG6 - VEG5);
                chance = PROB5 + (chance * (PROB6 - PROB5));
                density = "HIGH";
            } else {
                chance = 1;
                density = "HIGH";
            }
        }
        if( block.y > treeLine ) {
            // only the very very smallest chance
            def t = 0.0001 * info.soilQuality;
        }
        int zone = (int)(TEMPERATURE_ZONES * info.temperature);

        echo(String.format("Density: %s  Zone: %d  Chance: %.03f  Splat: %d", density, zone, chance, light));
    }

    addAction("place") { blockHit ->
        Vec3d loc = blockHit.location;
        Vec3i block = blockHit.block;
        log.info("place:" + loc);

        if( !getPerms(activator, loc).canAddObject() ) {
            echo("You do not have permission to place objects here.");
            return false;
        }

        def mode = env.getVar("testWandMode", "Gaefen");

        def stepTime = gameSystems.stepTime;

        def mobDecay = 600;

        def components = [];
        def entity = null;
        switch( mode ) {
            case "Projectile":
                gourdGunTestClick.call(session);
                break;
            case "Sound":
                entity = type("Speakers").newInstance(
                        ObjectName.create("Speakers"),
                        SoundInfo.create("/Sounds/Effects/TestSounds.ogg", now(), 1, true),
                        new CreatedBy(activator.id),
                        Decay.seconds(32)
                    );
                entity.run("placeInWorld", loc, new Quatd());
                break;
            case "Sphere":
                entity = type("TestSphere").newInstance(
                    ShapeInfo.create("sphere", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Sphere", entityData),
                    new Mass(60),
                    new CreatedBy(activator.id),
                    Decay.seconds(120)
                );
                entity.run("placeInWorld", loc, new Quatd());
                break;
            case "Debug Arrow":
                entity = createEntity(
                    new SpawnPosition(loc, new Quatd()),
                    new DebugShape(DebugShape.TYPE_ARROW, new Vec3d(1, 1, 1), ColorRGBA.Red, 0),
                    Decay.seconds(120)
                );
                echo("Created test arrow:" + entity[DebugShape] + " at:" + loc);
                break;
            case "Campfire":
                entity = type("Campfire").newInstance();
                entity.run("placeAtCell", loc.add(blockHit.normal.mult(0.1)).floor());
                entity.run("activate");
                break;
            case "Test Chest":
                entity = type("TestChest").newInstance();
                entity.run("placeInWorld", loc, activator.orientation);
                entity.run("generateLoot");
                break;
            case "Test Backpack":
                entity = type("Backpack").newInstance(
                    ObjectName.create("Test Backpack"),
                    new CreatedBy(activator.id)
                );
                entity.run("placeInWorld", loc, activator.orientation);
                break;
            case "Test Pouch":
                entity = type("Pouch").newInstance(
                    ObjectName.create("Test Pouch"),
                    new CreatedBy(activator.id)
                );
                entity.run("placeInWorld", loc, activator.orientation);
                break;
            case "Test Sword":
                entity = type("Sword").newInstance();
                entity.run("placeInWorld", loc, activator.orientation);
                break;
            case "Airship":
                Vec3d airshipLoc = block.add(0, 5, 0).toVec3d();
                entity = type("Airship").newInstance(
                        ObjectName.create("Small Airship", entityData),
                        new Mass(0),  // static for now
                        new CreatedBy(activator.id),
                        new LargeObject(),

                        // We'll force a grid cell since right now the object
                        // is static.
                        new LargeGridCell(GameConstants.LARGE_OBJECT_GRID.worldToId(airshipLoc))
                    );
                entity.run("placeInWorld", airshipLoc, new Quatd());
                break;
            case "Gaefen":
                entity = type("Gaefen").newInstance(
                    //ShapeInfo.create("animals/gaefen.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    //ObjectName.create("Gaefen", entityData),
                    //new Mass(10),
                    new CreatedBy(activator.id),
                    //AgentType.create("Gaefen", AgentType.LEVEL_HIGH, entityData),
                    Decay.seconds(mobDecay)
                );
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());

                //entity = createEntity(
                //    ShapeInfo.create("animals/gaefen.rig", 1),
                //    new SpawnPosition(loc, new Quatd()),
                //    ObjectName.create("Gaefen", entityData),
                //    new Mass(10),
                //    new CreatedBy(activator.id),
                //    AgentType.create("Gaefen", AgentType.LEVEL_HIGH, entityData),
                //    Decay.seconds(120)
                //);
                break;
            case "Butterfly":
                entity = type("Butterfly").newInstance(
                    //ShapeInfo.create("animals/butterfly.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    //ObjectName.create("Butterfly", entityData),
                    //new Mass(0.1),
                    new CreatedBy(activator.id),
                    //AgentType.create("Butterfly", AgentType.LEVEL_HIGH, entityData),
                    Decay.seconds(mobDecay)
                );
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                break;
            case "Random Butterfly":
                entity = type("Butterfly").newInstance(
                    //ShapeInfo.create("animals/butterfly.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    //ObjectName.create("Butterfly", entityData),
                    //new Mass(0.1),
                    new CreatedBy(activator.id),
                    //AgentType.create("Butterfly", AgentType.LEVEL_HIGH, entityData),
                    Decay.seconds(mobDecay)
                );
                entity.run("randomizeAppearance", new Random());
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                break;
            case "Test Butterfly":
                entity = type("Butterfly").newInstance(
                    //ShapeInfo.create("animals/butterfly.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    //ObjectName.create("Butterfly", entityData),
                    //new Mass(0.1),
                    new CreatedBy(activator.id),
                    //AgentType.create("Butterfly", AgentType.LEVEL_HIGH, entityData),
                    Decay.seconds(mobDecay)
                );
                entity.remove(AgentType.class);
                entity.run("randomizeAppearance", new Random());
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                break;
            case "Dog":
                entity = type("Dog").newInstance(
                    //ShapeInfo.create("animals/dog/dog-lab.rig", 2),
                    new SpawnPosition(loc, new Quatd()),
                    //ObjectName.create("Dog", entityData),
                    //new Mass(10),
                    new CreatedBy(activator.id),
                    //AgentType.create("Dog", AgentType.LEVEL_HIGH, entityData),
                    Decay.seconds(mobDecay)
                );
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                break;
            case "Bird":
                entity = type("Bird").newInstance(
                    new SpawnPosition(loc, new Quatd()),
                    new CreatedBy(activator.id),
                    Decay.seconds(mobDecay)
                );
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                break;
            case "Crow":
                entity = type("Crow").newInstance(
                    new SpawnPosition(loc, new Quatd()),
                    new CreatedBy(activator.id),
                    Decay.seconds(mobDecay)
                );
                entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
                echo("Crow shape:" + entity[ShapeInfo]);
                break;
            case "Random Human":
                entity = createRandomNpc(resolveEntityId(session), loc, new Random());
                break;
            case "Random Human Male":
                entity = createEntity(
                    ShapeInfo.create("human/male/human-male.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Human Male", entityData),
                    ObjectTypeInfo.create("NPC"),
                    Race.create("human-male", entityData),
                    new Mass(60),
                    new CreatedBy(activator.id),
                    AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)//,
                    //Decay.seconds(120)
                );
                //entity.run("resetDefaultAppearance");
                entity.run("randomize", new Random());
                break;
            case "Random Human Female":
                entity = createEntity(
                    ShapeInfo.create("human/female/human-female.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Human Female", entityData),
                    ObjectTypeInfo.create("NPC"),
                    Race.create("human-female", entityData),
                    new Mass(60),
                    new CreatedBy(activator.id),
                    AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)//,
                    //Decay.seconds(120)
                );
                //entity.run("resetDefaultAppearance");
                entity.run("randomize", new Random());
                break;
            case "Random Human Male Giant":
                entity = createEntity(
                    ShapeInfo.create("human/male/human-male.rig", 1.5),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Human Male", entityData),
                    ObjectTypeInfo.create("NPC"),
                    Race.create("human-male", entityData),
                    new Mass(90),
                    new CreatedBy(activator.id),
                    AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)//,
                    //Decay.seconds(120)
                );
                //entity.run("resetDefaultAppearance");
                entity.run("randomize", new Random());

                //def tool = type("Axe").newInstance(
                //    new CreatedBy(activator.id),
                //    new ContainedIn(entity, null, 0, 0)
                //);
                //tool.run("Equip");
                break;
            case "Random Elf Male":
                entity = createEntity(
                    ShapeInfo.create("elf/male/elf-male.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Elf Male", entityData),
                    ObjectTypeInfo.create("NPC"),
                    Race.create("elf-male", entityData),
                    new Mass(60),
                    new CreatedBy(activator.id),
                    AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)//,
                    //Decay.seconds(120)
                );
                //entity.run("resetDefaultAppearance");
                entity.run("randomize", new Random());
                break;
            case "Random Elf Female":
                entity = createEntity(
                    ShapeInfo.create("elf/female/elf-female.rig", 1),
                    new SpawnPosition(loc, new Quatd()),
                    ObjectName.create("Elf Female", entityData),
                    ObjectTypeInfo.create("NPC"),
                    Race.create("elf-female", entityData),
                    new Mass(60),
                    new CreatedBy(activator.id),
                    AgentType.create("Basic NPC", AgentType.LEVEL_HIGH, entityData)//,
                    //Decay.seconds(120)
                );
                //entity.run("resetDefaultAppearance");
                entity.run("randomize", new Random());
                break;
            //case "Trophy Test":
            //    entity = type("2024 Trophy Pile").newInstance(
            //        new CreatedBy(activator.id),
            //        Decay.seconds(mobDecay)
            //    );
            //    entity.run("placeInWorld", loc.add(0, 2, 0), new Quatd());
            //    echo("test trophy pile:" + entity[ShapeInfo]);
            //    break;
            //case "Town Info":
            //    testTownInfo.call(session, self, loc);
            //    break;
            default:
                echo("Unknown mode:" + mode);
                return false;
        }
        return true;
    }

    addAction("contextAction") { objectHit ->
        def clicked = objectHit.entityId;
        def pos = objectHit.location;

        log.info("test wand: contextAction:" + clicked + ", " + pos);
        def name = clicked.name?:"Unknown";

        // Here we want to let the player run some actions on the object

        // Is it our object?
        def creator = clicked[CreatedBy]?.creatorId;
        def owner = clicked[OwnedBy]?.owner;
        log.info("Player:" + activator.id + " creator:" + creator + " owner:" + owner);
        boolean isOurs = activator.isSameEntity(creator) || activator.isSameEntity(owner);
        boolean canRemove = isOurs || (creator == null && owner == null);

        // Assemble the action list
        def options = [];
        clicked.run("listActions") { action ->
            //echo("test:" + action);
            if( "Take".equals(action.getName()) ) {
                // We'll assume that if it can be 'taken' then deleting it
                // would be destructive.
                canRemove = false;
            }
            options.add(new Option(action.getName()));
        }

        options.add(new Option(self, "showDebugInfo", clicked));

        // Instead of hasShape... we should probably base this on
        // the actions that the object has.  That way we can have non-removable
        // objects that have shapes.  And then move "deleteObject" to the objects
        // that support delete... the others can echo a message that they can't
        // be removed.
        // FIXME: use the runnable action list when we have it... ie: not all
        // actions, just the ones that canRun().
        def objPos = clicked[SpawnPosition]?.location;
        if( getPerms(activator, objPos).canRemoveObject() ) {
            // Options to call us back as the manipulator
            options.add(new Option(self, "deleteObject", clicked));
        }
        session.showPrompt(clicked, PromptType.List, name);
        session.showOptions(clicked, options);
    }

    addAction("showDebugInfo") { clicked ->
        log.info("showDebugInfo:" + clicked);
        echo "------Debug info for:" + clicked + "  type:" + clicked.type?.name + " name:" + clicked.name;
        log.info("------Debug info for:" + clicked + "  type:" + clicked.type?.name + " name:" + clicked.name);
        entityData.componentHandlers.keySet().each { componentType ->
            if( clicked[componentType] ) {
                echo "  " + clicked[componentType];
                log.info("  " + clicked[componentType]);
            }
        }
        gameSystems.getArray().each { sys ->
            if( sys instanceof DiagnosticsProducer ) {
                echo ">" + sys.getClass().getSimpleName();
                log.info(">" + sys.getClass().getSimpleName());
                def diags = sys.getDiagnostics(clicked);
                diags.each {
                    echo "  " + it
                    log.info("  " + it);
                }
            }
        }

        if( system(AgentSystem).getAgentDriver(clicked) ) {
            system(AgentSystem).getAgentDriver(clicked).debugAgent = true;
        }


    }

    addAction("deleteObject") { clicked ->
        log.info("deleteObject:" + clicked);

        def objPos = clicked[SpawnPosition]?.location;
        if( !getPerms(activator, objPos).canRemoveObject() ) {
            echo("You do not have permission to remove objects here.");
            return;
        }
        clicked.runIfExists("cleanup");

        // We should probably check to see if we can in case we hacked
        // in around the context menu
        log.info("removing:" + clicked);
        removeEntity(clicked);
    }
}

createType("TestChest").with {
    supertypes("BaseCloseable");

    setTypeVar("shape", ShapeInfo.create("/Models/objects/sm-chest1-bottom.blocks", 0.25));

    // inside is 3x2x2 high (technically 1.75 high)
    // outside is 5x4x2
    def innerSize = new Vec3d(3, 1.75, 2);
    innerSize *= 0.25;

    def size = getTypeVar("shape").loadCells().size.toVec3d();
    size *= 0.25;
    def offset = size.subtract(innerSize).mult(0.5);

    setTypeVar("volume", new ObjectVolume(size));
    setTypeVar("containerVolume", new ContainerVolume(innerSize, offset));

    addAction("generateLoot") { ->
        echo("generate loot");
    }

    addAction("onOpen") { ->
        echo("onOpen run");
        session.openContainer(self);
    }

    addAction("Open") { ->
        runIfExists("onOpen");
    }

    //addAction("Close") { ->
    //}
}

createType("TestDoor").with {
    supertypes("BaseCloseable");
}

createType("TestDesk").with {
    supertypes("BaseCloseable");
}


createType("Speakers").with {
    supertypes("BaseObject");

    setTypeVar("shape", ShapeInfo.create("/Models/objects/speakers.blocks", 0.25));

    addAction("Look") { ->
        def lines = [];

        lines += "Examine: " + self.name;

        lines += "Sound:" + self[SoundInfo]

        session.showPrompt(self, PromptType.Dialog, lines.join("\n\n"));
        session.showOptions(self, []);

        return true;
    }
}

createType("TestSphere").with {
    supertypes("BaseObject");

}

createType("Pouch").with {
    supertypes("BaseItem");

    setTypeVar("shape",
               ShapeInfo.create("/Models/items/pouch1.blocks", standardItemScale * 0.5));

    def size = getTypeVar("shape").loadCells().size.toVec3d();


    setTypeVar("volume", new ObjectVolume(getTypeVar("shape").volume));

    // Vec3d[0.0625, 0.0859375, 0.03125]
    def innerSize = new Vec3d(4, 5.5, 2);
    innerSize *= standardItemScale * 0.5;

    def offset = size.subtract(innerSize).mult(0.5);
    offset.y = standardItemScale * 0.5; // hard coded to 1 block thickness for the base

    setTypeVar("containerVolume", new ContainerVolume(innerSize, offset));

    setTypeVar("description",
        "A small leather pouch.");
}

npcRandom = new Random(0);
skinToneIndex = SkinToneIndex.create { id ->
    return worldManager.getTextDb().getText(id, null);
}
hairToneIndex = HairToneIndex.create { id ->
    return worldManager.getTextDb().getText(id, null);
}

characterNameGenerator = CharacterNameGenerator.create { id ->
    return worldManager.getTextDb().getText(id, null);
}

createType("QuestAxe").with {
    supertypes("Axe");

    addAction("Take") { ->
        log.info("Activator:" + activator + " picking up object " + self.name);

        //self <<
        // FIXME: hook up activator.id a little more consistently
        def backpack = findItem(activator.id, "Backpack");
        log.info("Putting it into backpack:" + backpack);

        def shape = self[ShapeInfo];

        // Need to find an empty spot or not allow it to be picked up
        self << new ContainedIn(activator.id, backpack, 18, 0);
        self << self.type.getTypeVar("shape", null)
        self << new Mass(2);
        self.remove(SpawnPosition);

        runIfExists("onTaken", backpack);
    }.onlyIf { ->
        // Can only pick up the object if it originally belonged to
        // the player and is not currently contained
        if( self[ContainedIn] ) {
            return false;
        }
        return true;
    }

    // For testing
    addAction("Run onTaken") { ->
        def backpack = findItem(activator.id, "Backpack");
        runIfExists("onTaken", backpack);
    }.onlyIf { ->
        return activator.isSameEntity(self[ContainedIn]?.root);
    }

    addAction("Drop") { ->

        def pos = activator.pos + activator.dir * 0.1;
        pos.y += 0.1;
        //echo("Dropping quest item at:" + pos);

        run("placeInWorld", pos, new Quatd());

        //questItemOnDropped.call(session, self);
        log.info("quest item dropped:" + self.name + " at:" + pos);
        findObjectiveQuests(self, resolveEntityId(session)).each { quest ->
            quest.run("itemDropped", self);
        }
    }.onlyIf { ->
        return activator.isSameEntity(self[ContainedIn]?.root);
    }

    addAction("onTaken") { container ->
        echo("Retrieved " + self.name + ".")
        //questItemOnTaken.call(session, self, container);
        findObjectiveQuests(self, resolveEntityId(session)).each { quest ->
            quest.run("itemTaken", self);
        }
    }
}


