/**
 *  Script to upgrade the InsertInfo to have proper entity IDs by
 *  filling in any existing entities or creating new entities.
 */

import mythruna.world.tile.*;
import mythruna.world.town.*;
import com.google.common.collect.*;
import com.simsilica.progress.*;


def findBestInsert = { struct, insertIndex ->
    def info = struct[StructureInfo];
    def trigger = struct[TriggerInfo];
    def pos = struct.location;

    //log.info("findBestInsert(" + struct + ") factory:" + info.getFactory(entityData) + " loc:" + pos);
    //log.info("  trigger:" + trigger);

    def result = null;
    insertIndex.keySet().each { insert ->
        if( pos.x == insert.location.x && pos.z == insert.location.z ) {
            //log.info("   found:" + insert);
            if( result != null ) {
                log.warn("Found more than one match, original:" + result + " new:" + insert);
            }
            result = insert;
            // A little bit of a sanity check
            //if( insert.size.x != trigger.size.x || insert.size.z != trigger.size.z ) {
            //    log.warn("Sizes differ between insert:" + insert.size + " and trigger:" + trigger.size);
            //}
            // Actually, it turns out that this can happen a lot.  The trigger is going
            // to be based on the actual structure size which could be a different size.
            // Some examples:
            // Struct: factory:fence-tall.blocks trigger size: xSize=4, ySize=3, zSize=1
            // Insert: type:isolated1,                   size:Vec3i[9, 1, 9]
            //
            // Struct: factory:roman-temple2.blocks trigger size: xSize=14, ySize=11, zSize=14
            // Insert: type:isolated3,                   size:Vec3i[9, 1, 9]
            //
            // Struct: factory:graves3.blocks trigger size: xSize=10, ySize=2, zSize=11
            // Insert: type:cliff2,                      size:Vec3i[9, 3, 9]
            //
            // Struct: factory:tall-dungeon.blocks trigger size: xSize=21, ySize=15, zSize=21
            // Insert: type:isolated3,                   size:Vec3i[9, 2, 9]
            //
            // probably this is a long-term problem... but it's a future problem.
        }
    }
    return result;
}

def uniqueStructureKey = { struct ->
    def info = struct[StructureInfo];
    def loc = struct.location;
    // For duplicate structures, everything but the tile ID is the same.
    return "" + info.parent + ":" + info.factoryId + ":" + info.blockDataId + ":" + loc;
}

/**
 *  Makes sure all InsertInfo objects have appropriate IDs by either
 *  matchting them up to their existing structure entities or generating a new entity
 */
def upgradeStructures = { ->
    def tileIndex = MultimapBuilder.hashKeys().hashSetValues().build();
    def dupeIndex = MultimapBuilder.hashKeys().arrayListValues().build();

    def toRemove = [] as Set;

    def needsUpgrade = false;

    def structureEntities = findEntities(null, StructureInfo, SpawnPosition, ObjectTypeInfo);
    structureEntities.each { struct ->

        def type = struct[ObjectTypeInfo].typeName;
        //def info = struct[StructureInfo];
        //def factory = info.getFactory(entityData);
        def tileId = TileId.fromWorld(struct.location);

        if( "Structure" == type ) {
            needsUpgrade = true;
        }

        tileIndex.put(tileId, struct);

        dupeIndex.put(uniqueStructureKey(struct), struct);

        //echo("" + struct + ":" + type.name + " " + factory + " " + tileId);
    }
    if( !needsUpgrade ) {
        log.info("World structure data already up to date.");
        return;
    }

    // Cleanup duplicates
    dupeIndex.keySet().each { key ->
        def dupes = dupeIndex.get(key);
        if( dupes.size() == 1 ) {
            return;
        }
        log.info("dupes:" + dupes);
        dupes.each { dupe ->
            def nativeTile = TileId.fromWorld(dupe.location);
            def tileId = dupe[StructureInfo].tileId;
            if( tileId != nativeTile ) {
                log.info("  need to remove:" + dupe);
                toRemove.add(dupe);
                tileIndex.values().remove(dupe);
            }
        }
    }

    log.info("Found " + tileIndex.size() +  " structures.  Duplicates:" + toRemove.size());
    def toSave = [] as Set;
    def insertIndex = [:];

    def progress = ProgressTrackers.openTracker("Upgrading feature data.");
    progress.setMax(tileIndex.keySet().size());
    try {
        // Every structure entity should have an insert but not every insert
        // should have a structure.  So it's better to look backwards even if it might
        // ultimately be a bit slower to snipe into the inserts every time.
        // We may also waste time looking up inserts that already have IDs... but
        // then we can sanity check that, too.  Perhaps correct it?
        tileIndex.keySet().each { tileId ->
            def structs = tileIndex.get(tileId);
            def sed = sedectileManager.getSedectile(tileId.getSedectileId());
            def featureIds = sed.getFeatureIds(tileId, PointOfInterest.class);

            featureIds.each { featureId ->
                def poi = sedectileManager.getFeatures().getFeature(featureId, PointOfInterest.class);
                poi.children.each { child ->
                    insertIndex.put(child, poi);
                }
            }

            structs.each { struct ->
                def bestInsert = findBestInsert(struct, insertIndex);
                log.info("struct:" + struct + " insert:" + bestInsert);
                if( bestInsert.id == null ) {
                    log.info("Need to update:" + bestInsert);
                    bestInsert.id = struct.id;
                    toSave.add(insertIndex.get(bestInsert));
                }
            }

            progress.increment();
        }
    } finally {
        progress.close(true);
    }

    def allPois = [] as Set;
    def root = new File(worldManager.info.directory, "tile.db");
    root.eachFileRecurse(groovy.io.FileType.FILES) {
        if( it.name.endsWith(".PointOfInterest") ) {
            //echo(" " + it);
            def idString = it.name.take(it.name.lastIndexOf(".PointOfInterest"));
            def id = Long.parseLong(idString, 16);
            def poi = sedectileManager.getFeatures().getFeature(new FeatureId(PointOfInterest, id), PointOfInterest);
            allPois.add(poi);
        }
    }

    progress = ProgressTrackers.openTracker("Adding missing entities.");
    progress.setMax(allPois.size());
    int newEntities = 0;
    try {
        allPois.each { poi ->
            poi.children.each { insert ->
                if( insert.id == null ) {
                    insert.id = entityData.createEntity().id;
                    //log.info("Created:" + insert.id + " for:" + insert);
                    toSave.add(poi);
                    newEntities++;
                }
            }
            progress.increment();
        }
    } finally {
        progress.close(true);
    }
    log.info("Created " + newEntities + " new entities");

    log.info("Need to save " + toSave.size() + " features.");
    progress = ProgressTrackers.openTracker("Saving feature data.");
    progress.setMax(toSave.size());
    try {
        toSave.each { feature ->
            //log.info("updating:" + feature);
            sedectileManager.getFeatures().updateFeature(feature);
            progress.increment();
        }
    } finally {
        progress.close(true);
    }

    progress = ProgressTrackers.openTracker("Upgrading structure data.");
    progress.setMax(structureEntities.size());
    try {
        structureEntities.each { struct ->
            if( toRemove.contains(struct) ) {
                log.info("removing:" + struct);
                entityData.removeEntity(struct);
            } else {
                String typeName = struct[ObjectTypeInfo].typeName;
                String factory = struct[StructureInfo].getFactory(entityData);
                //log.info("factory:" + factory + " typeName:" +typeName);
                if( "Structure".equals(typeName) ) {
                    def newType = "DefaultStructure";
                    if( "spirit-henge.blocks" == factory || "spirit-henge-desert.blocks" == factory ) {
                        newType = "SpiritTemple";
                    }
                    log.info("Upgrading " + struct + " type to:" + newType);
                    struct << ObjectTypeInfo.create(newType);
                }
            }
            progress.increment();
        }
    } finally {
        progress.close(true);
    }
}

def migrateOwnedByToChildOf = {
    // Migrate the town PoiLimits
    findEntities(ObjectTypeInfo.typeFilter("PoiLimits"), ObjectTypeInfo, OwnedBy).each {
        def owner = it[OwnedBy];
        log.info("Migrating 'PoiLimits':" + it + " " + owner);
        it << new ChildOf(owner.owner);
        it.remove(OwnedBy);
    }

    // Migrate any portals and moon gates that used the older style.
    // (pretty much only affects my local worlds since no release would have had this.)
    findEntities(ObjectTypeInfo.typeFilter("Portal"), ObjectTypeInfo, OwnedBy).each {
        def owner = it[OwnedBy];
        log.info("Migrating 'Portal':" + it + " " + owner);
        it << new ChildOf(owner.owner);
        it.remove(OwnedBy);
    }
    findEntities(ObjectTypeInfo.typeFilter("MoonGate"), ObjectTypeInfo, OwnedBy).each {
        def owner = it[OwnedBy];
        log.info("Migrating 'MoonGate':" + it + " " + owner);
        it << new ChildOf(owner.owner);
        it.remove(OwnedBy);
    }

    log.info("Remaining ownedBy:");
    findEntities(null, ObjectTypeInfo, OwnedBy).each {
        log.info("" + it + " ownedBy:" + it[OwnedBy]?.owner + " type:" + it.type?.name);
    }
}

/**
 *  For all existing StructureInfo entities, we want to find their
 *  parent POI and connect a ChildOf component to them.
 */
def connectInsertsToPois = {
    def tileIndex = MultimapBuilder.hashKeys().hashSetValues().build();

    def structureEntities = findEntities(null, StructureInfo, SpawnPosition, ObjectTypeInfo);
    structureEntities.each { struct ->
        def tileId = TileId.fromWorld(struct.location);
        tileIndex.put(tileId, struct);
    }

    def progress = ProgressTrackers.openTracker("Upgrading structure data.");
    progress.setMax(tileIndex.keySet().size());
    try {
        tileIndex.keySet().each { tileId ->
            def structs = tileIndex.get(tileId);
            def sed = sedectileManager.getSedectile(tileId.getSedectileId());
            def featureIds = sed.getFeatureIds(tileId, PointOfInterest.class);

            featureIds.each { featureId ->
                def poi = sedectileManager.getFeatures().getFeature(featureId, PointOfInterest.class);
                def parentEntity = new EntityId(featureId.id);
                poi.children.each { child ->

                    // If this child's entity is in the existing structure entities
                    // then write back the ChildOf component
                    def childId = new EntityId(child.id);
                    if( structs.contains(childId) ) {
                        // For sanity, see if it already has one
                        def existing = childId[ChildOf];
                        if( existing ) {
                            log.info("" + childId + " already mapped to a parent:" + existing.parent + " poi parent:" + parentEntity);
                        } else {
                            log.info("Upgrading:" + childId + " to point to parent:" + parentEntity);
                            childId << new ChildOf(parentEntity);
                        }
                    }
                }
            }
            progress.increment();
        }
    } finally {
        progress.close(true);
    }

}

def initializeSpiritTemples = {
    findEntities(ObjectTypeInfo.typeFilter("SpiritTemple"), ObjectTypeInfo).each {
        log.info("*****************************************************");
        log.info("*****************************************************");
        log.info("*****************************************************");
        log.info("Kilroy: running initialize on:" + it);
        system(GameActionSystem).runAction(it, "initialize");
    }
}

// What is the existing structure version?
def structVersionKey = "structure.data.version";
def structVersion = worldManager.worldEntity.vars.loadInt(structVersionKey, -1);
def latestVersion = 4;

log.info("Structure data version:" + structVersion);
if( structVersion < 1 ) {
    log.info("Need to upgrade structure data.");
    upgradeStructures();
}

if( structVersion < 2 ) {
    log.info("Need to migrate OwnedBy to ChildOf components");
    migrateOwnedByToChildOf();
}
if( structVersion < 3 ) {
    log.info("Need to add missing ChildOf components");
    connectInsertsToPois();
}

// Can't reliably call entity actions until the systems are all initialized
on(com.simsilica.sim.SimEvent.simInitialized) { event ->
    // Do not run if this is on the client-side... the event bus is global
    // so this event block will be called by the mythruna.client.ClientGameSystemManager
    // in addition to the server (if we are running a single player game).  In multiplayer,
    // I suspect this code will just crash for lack of a world manager.
    if( event.manager.getClass().name == "mythruna.client.ClientGameSystemManager" ) {
        return;
    }
    log.info("Kilroy:" + structVersion);
    if( structVersion < 4 ) {
        initializeSpiritTemples();
    }
    worldManager.worldEntity.vars.storeInt(structVersionKey, latestVersion);
}



