

#import "Common/ShaderLib/MultiSample.glsllib"
#import "MatDefs/hsv.glsllib"


uniform COLORTEXTURE m_Texture;
uniform DEPTHTEXTURE m_DepthTexture;
uniform sampler2D m_Noise;
uniform sampler2D m_Background;
uniform sampler2D m_Sketch;
uniform sampler2D m_SketchLight;

uniform float m_WaterColorStrength;
uniform float m_SaturationThreshold;
uniform float m_SaturationMargin;
uniform float m_BrightnessThreshold;
uniform float m_BrightnessCutoff;
uniform float m_BrightnessMargin;

uniform float m_LineStrength;
uniform float m_SketchRandomness;
uniform float m_ColorEdgeStrength;
uniform float m_DepthEdgeStrength;
uniform float m_NearEdgeFactor;
uniform float m_NearEdgeExposure;
uniform float m_MidEdgeFactor;
uniform float m_MidEdgeExposure;
uniform float m_DepthFadeExponent;
uniform float m_DepthFadeFactor;

uniform float m_ShadingStrength;
uniform float m_CrossHatchThreshold;
uniform float m_ShadingContrast;

uniform float g_Time;
uniform vec2 g_FrustumNearFar; // = vec2(0.1, 9000.0);
uniform vec2 g_Resolution;

//#define COLOR
#define SHADING
#define LINES
#define COLOR_MIX
//#define COLOR_MIX2
//#define TEST_COMBINED_EDGE

//#define OLDEDGE

varying vec2 texCoord;




float depthToZ( in float depth ) {
    //
    // z_buffer_value = a + b / z;
    //
    // Where:
    //  a = zFar / ( zFar - zNear )
    //  b = zFar * zNear / ( zNear - zFar )
    //  z = distance from the eye to the object
    //
    // Which means:
    // zb - a = b / z;
    // z * (zb - a) = b
    // z = b / (zb - a)
    //
    float a = g_FrustumNearFar.y / (g_FrustumNearFar.y - g_FrustumNearFar.x);
    float b = g_FrustumNearFar.y * g_FrustumNearFar.x / (g_FrustumNearFar.x - g_FrustumNearFar.y);
    return b / (depth - a);
}

float extractZ( in DEPTHTEXTURE texture, in vec2 uv ) {
    float zb = getDepth( texture, uv ).r;
    return depthToZ(zb);
}

float edgeLevel( in COLORTEXTURE texture, in float scale, in vec2 uv ) {
    float x = uv.x;
    float y = uv.y;
    float xScale = scale/g_Resolution.x; //1600.0;
    float yScale = scale/g_Resolution.y; //900.0;

    // Sobel filter for edge detection
    // ...should investigate Frei-Chen filter
    vec3 c00 = rgb2hsv(getColor(texture, vec2(x - 1.0 * xScale, y - 1.0 * yScale)).rgb);
    vec3 c10 = rgb2hsv(getColor(texture, vec2(x, y - 1.0 * yScale)).rgb);
    vec3 c20 = rgb2hsv(getColor(texture, vec2(x + 1.0 * xScale, y - 1.0 * yScale)).rgb);
    vec3 c01 = rgb2hsv(getColor(texture, vec2(x - 1.0 * xScale, y)).rgb);
    vec3 c11 = rgb2hsv(getColor(texture, vec2(x, y)).rgb);
    vec3 c21 = rgb2hsv(getColor(texture, vec2(x + 1.0 * xScale, y)).rgb);
    vec3 c02 = rgb2hsv(getColor(texture, vec2(x - 1.0 * xScale, y + 1.0 * yScale)).rgb);
    vec3 c12 = rgb2hsv(getColor(texture, vec2(x, y + 1.0 * yScale)).rgb);
    vec3 c22 = rgb2hsv(getColor(texture, vec2(x + 1.0 * xScale, y + 1.0 * yScale)).rgb);


    //     -1 -2 -1        1 0 -1
    // H =  0  0  0    V = 2 0 -2
    //      1  2  1        1 0 -1
    vec3 horz = c02 + 2.0 * c12 + c22 - c00 - 2.0 * c10 - c20;
    vec3 vert = c00 + 2.0 * c01 + c02 - c20 - 2.0 * c21 - c22;
    //return sqrt( (horz.rgb * horz.rgb) + (vert.rgb * vert.rgb) );
    return sqrt( (horz.z * horz.z) + (vert.z * vert.z) );
}

float distanceToRange( float distance, float max ) {
    return min(1.0, distance / max);
}

float depthEdgeLevel( in DEPTHTEXTURE texture, in float scale, in vec2 uv, float farPlane ) {
    float x = uv.x;
    float y = uv.y;
    float xScale = scale/1600.0;
    float yScale = scale/900.0;

    // Sobel filter for edge detection
    // ...should investigate Frei-Chen filter
    float c00 = distanceToRange(extractZ(texture, vec2(x - 1.0 * xScale, y - 1.0 * yScale)), farPlane);
    float c10 = distanceToRange(extractZ(texture, vec2(x, y - 1.0 * yScale)), farPlane);
    float c20 = distanceToRange(extractZ(texture, vec2(x + 1.0 * xScale, y - 1.0 * yScale)), farPlane);
    float c01 = distanceToRange(extractZ(texture, vec2(x - 1.0 * xScale, y)), farPlane);
    float c11 = distanceToRange(extractZ(texture, vec2(x, y)), farPlane);
    float c21 = distanceToRange(extractZ(texture, vec2(x + 1.0 * xScale, y)), farPlane);
    float c02 = distanceToRange(extractZ(texture, vec2(x - 1.0 * xScale, y + 1.0 * yScale)), farPlane);
    float c12 = distanceToRange(extractZ(texture, vec2(x, y + 1.0 * yScale)), farPlane);
    float c22 = distanceToRange(extractZ(texture, vec2(x + 1.0 * xScale, y + 1.0 * yScale)), farPlane);


    //     -1 -2 -1        1 0 -1
    // H =  0  0  0    V = 2 0 -2
    //      1  2  1        1 0 -1
    float horz = c02 + 2.0 * c12 + c22 - c00 - 2.0 * c10 - c20;
    float vert = c00 + 2.0 * c01 + c02 - c20 - 2.0 * c21 - c22;
    return sqrt( (horz * horz) + (vert * vert) );
}

float combinedEdgeLevel( in vec2 uv, float depthMix ) {
    // Detect edge using depth
    //float farPlane = g_FrustumNearFar.y;
    //float farPlane = 32.0;
    //float farPlane = 48.0;
    float farPlane = 64.0;
    float depthEdgeNear = depthEdgeLevel(m_DepthTexture, 1.0, uv, farPlane);
    depthEdgeNear = smoothstep(0.0, 1.0 - m_NearEdgeExposure, depthEdgeNear) * m_NearEdgeFactor;

    farPlane = 256.0;
    float depthEdgeMid = depthEdgeLevel(m_DepthTexture, 1.0, uv, farPlane);
    depthEdgeMid = smoothstep(0.0, 1.0 - m_MidEdgeExposure, depthEdgeMid) * 0.25 * m_MidEdgeFactor;

    float depthEdgeFar = depthEdgeLevel(m_DepthTexture, 1.0, uv, g_FrustumNearFar.y);
    float depthEdge = min(1.0, depthEdgeFar + depthEdgeNear + depthEdgeMid);

    // Detect edges using brightness
    float colorEdge = edgeLevel(m_Texture, 1.0, uv);
    colorEdge = smoothstep(0.1, 1.0, colorEdge);

    // Make sure that even the color-based edges have some of the depth
    // edge mixed in.
    //colorEdge = min(1.0, colorEdge + depthEdge * 16.0);
    colorEdge = min(1.0, colorEdge + (depthEdgeNear + depthEdgeMid) * m_DepthEdgeStrength);

    float totalDepth = min(1.0, depthMix * m_DepthEdgeStrength + (1.0 - m_ColorEdgeStrength));
    float result = mix(colorEdge * m_ColorEdgeStrength, depthEdge, totalDepth);

    // Always add back a little of the depth edge
    //result = depthEdgeNear + depthEdgeFar;
    //result = depthEdgeMid;
    //result = depthEdge;
    //result = depthEdgeFar;
    //result = colorEdge;
    //result = min(1.0, result + depthEdge * 0.9);
    //result = min(1.0, depthEdge * 8.0);
    //result = 1.0 - depthEdge;
    //result = result * result * result * result;
    //result = 1.0 - result;

    float depthFade = 1.0 - (depthMix * m_DepthFadeFactor * m_DepthEdgeStrength);
    //depthFade = depthFade * depthFade * depthFade * depthFade;
    depthFade = pow(depthFade, m_DepthFadeExponent);
    //depthFade = smoothstep(0.0, 0.5, depthFade);
    depthFade = max(0.3, depthFade);

    return result * depthFade;
}

vec4 multiFractal3( vec2 coord, float scale, float growth ) {
    vec4 f1 = (texture2D(m_Noise, coord * scale) - 0.5) * 2.0;
    vec4 f2 = ((texture2D(m_Noise, coord * scale * growth) - 0.5) * 2.0).gbra;
    vec4 f3 = ((texture2D(m_Noise, coord * scale * growth * growth) - 0.5) * 2.0).brga;

    return f1 + f2 * 0.5 + f3 * 0.25;
}


void main() {

    // Get the framebuffer stuff
    vec4 screenColor = getColor(m_Texture, texCoord);
    vec3 screenHsv = rgb2hsv(screenColor.rgb);
    float z = extractZ(m_DepthTexture, texCoord);
    float depthScale = z / g_FrustumNearFar.y;

#ifdef TEST_COMBINED_EDGE
    // Detect edge using depth
    //float depthEdge = depthEdgeLevel(m_DepthTexture, 1.0, texCoord, g_FrustumNearFar.y);

    // Detect edges using brightness
    //float debugEdge = edgeLevel(m_Texture, 1.0, texCoord);

    //float combinedEdge = mix(debugEdge, depthEdge, depthScale);
    float combinedEdge = combinedEdgeLevel(texCoord, depthScale);

    //gl_FragColor.rgb = vec3(depthEdge, debugEdge, 0.0);
    //gl_FragColor.rgb = vec3(combinedEdge);
    //gl_FragColor.rgb = mix(screenColor.rgb, vec3(0.0), combinedEdge);
    //gl_FragColor.rgb = mix(vec3(1.0), vec3(0.0), depthEdge);
    //gl_FragColor.rgb = mix(vec3(1.0), vec3(0.0), debugEdge);
    gl_FragColor.rgb = mix(vec3(1.0), vec3(0.0), combinedEdge);

#else

    vec4 background = texture2D(m_Background, texCoord * vec2(4.0, 2.0));

    //float time = 1.0;
    float time = g_Time;

    // Sample slightly offset portions of the screen for edge detection
    //vec2 uvOffset = texture2D(m_Noise, 0.5 * texCoord + vec2(time * 0.1, time * 0.2)).rg;
    vec2 uvOffset = texture2D(m_Noise, 1.0 * texCoord + vec2(time * 0.1, time * 0.2)).rg;
    uvOffset *= 2.0;
    uvOffset -= vec2(1.0);

    // This is the "lowest resolution" and not really the "roughest" in the sense
    // of chaos.
    //float roughScale1 = 0.01; // original value.
    // I feel like the original value varies too much from the shading layer.
    //float roughScale1 = 0.002;
    // The issue with this one is that large values will make the 'main line' of the
    // drawing grossly offset from the shading in some places.  This would be find if
    // the shading was based on the same fractal offsets.
    float roughScale1 = 0.0025 * m_SketchRandomness;
    vec2 uv = texCoord + uvOffset * vec2(roughScale1, -roughScale1);
    #ifdef OLDEDGE
        float roughest = edgeLevel(m_Texture, 1.0, uv);
    #else
        float roughest = combinedEdgeLevel(uv, depthScale);
    #endif

    // This one will be the rough sketch that the artist did before doing the clearer
    // lines above.
    uvOffset = texture2D(m_Noise, 10.0 * texCoord + vec2(time * 0.5, time * 0.7)).rg;
    uvOffset *= 2.0;
    uvOffset -= vec2(1.0);
    //float roughScale2 = 0.0025;
    float roughScale2 = 0.005 * m_SketchRandomness;
    uv = uv + uvOffset * vec2(-roughScale2, roughScale2);
    #ifdef OLDEDGE
        float rough = edgeLevel(m_Texture, 1.0, uv);
    #else
        float rough = combinedEdgeLevel(uv, depthScale);
    #endif

    // And we'll grab a third rough sketch layer to have more blending
    // flexibility.
    uvOffset = texture2D(m_Noise, 5.0 * texCoord + vec2(time * 0.3, time * 0.9)).gb;
    uvOffset *= 2.0;
    uvOffset -= vec2(1.0);
    float roughScale3 = 0.002 * m_SketchRandomness;
    //float roughScale3 = 0.005;
    uv = texCoord + uvOffset * vec2(roughScale1, -roughScale1);
    uv = uv + uvOffset * roughScale3;
    #ifdef OLDEDGE
        float rough3 = edgeLevel(m_Texture, 1.0, uv);
    #else
        float rough3 = combinedEdgeLevel(uv, depthScale);
    #endif

    //uvOffset = texture2D(m_Noise, 5.0 * texCoord + vec2(time * 0.7, time * 0.5)).rg;
    //uvOffset *= 2.0;
    //uvOffset -= vec2(1.0);
    //uv = uv + uvOffset * vec2(0.005, 0.005);
    //float rough2 = edgeLevel(m_Texture, 1.0, uv);

    //gl_FragColor.rgb = vec3(roughest, rough, debugEdge);
    //gl_FragColor.a = 1.0;

    // Magic to combine the roughnesss into a common set of drawn lines
    //float level = roughest;// * step(0.25, roughest) * 0.75;
    float level = smoothstep(0.25, 0.5, roughest);
    level += smoothstep(0.0, 0.8, rough) * 0.5;
    level += smoothstep(0.1, 0.3, rough3) * 0.5;
    //level += rough;// * step(0.1, rough) * 0.5;
    //level += rough * step(0.1, rough) * 0.5;
    //level += rough * 0.25;
    //level += rough3 * 0.5;
    //level = step(0.1, level);
    level *= step(0.1, level);
    level = clamp(level, 0.0, 1.0);

    // Sample a layered fractal at different offsets
    // TODO: multifractal?
    float fractal = 1.0;
    float offset = 1.0;
    fractal = texture2D(m_Noise, texCoord * 0.1).r;
    offset = texture2D(m_Noise, texCoord).r + time * 2.0;
    fractal *= texture2D(m_Noise, texCoord + vec2(offset * 0.01)).r;
    offset = texture2D(m_Noise, texCoord * 0.5).r + time * 4.0;
    fractal += texture2D(m_Noise, texCoord + vec2(offset * 0.1)).r;
    fractal = clamp(fractal, 0.0, 1.0);
    fractal = clamp(fractal * 0.2 + 0.8, 0.8, 1.0);

    // Let's calculate a draw area... ie: the edge of the screen should be
    // like the edge of the paper and not get any drawing at all... but we
    // also don't want it to be too uniform.
    // Calculate distance from the edge
    float xRawEdge = 1.0 - abs(texCoord.x - 0.5) * 2.0;
    float yRawEdge = 1.0 - abs(texCoord.y - 0.5) * 2.0;

    // And we only really want the curves pretty close to the edge so
    // we'll limit the effect.
    float xSoftEdge = min(1.0, xRawEdge * 2.0);
    float ySoftEdge = min(1.0, yRawEdge * 2.0);

    // Make it more pillowy
    xSoftEdge = smoothstep(0.0, 1.0, xSoftEdge);
    ySoftEdge = smoothstep(0.0, 1.0, ySoftEdge);

    // Figure out a "hard stop" limiting edge very close to the edge of the page.
    float xHardEdge = min(1.0, xRawEdge * 5.0);
    float yHardEdge = min(1.0, yRawEdge * 5.0);
    xHardEdge = smoothstep(0.0, 0.2, xHardEdge);
    yHardEdge = smoothstep(0.0, 0.3, yHardEdge);

    float drawArea = ySoftEdge * xSoftEdge;

    float hardEdge = xHardEdge * yHardEdge;

    // Add in some fractal noise to vary the edge a bit
    //float areaFractal1 = (texture2D(m_Noise, texCoord * 0.25).r - 0.5) * 2.0;
    //float areaFractal2 = (texture2D(m_Noise, texCoord * 3.0).r - 0.5) * 2.0;
    //float areaFractal = areaFractal1 + areaFractal2 * 0.5;
    ////areaFractal = clamp((areaFractal - 0.5) * 2.0, 0.0, 1.0);
    //drawArea = areaFractal;

    vec4 areaFractal = multiFractal3(texCoord - vec2(time, time), 0.5, 4.0);
    //gl_FragColor.rgb = vec3(areaFractal.r);
    drawArea += areaFractal.r * 0.125;

    drawArea *= hardEdge;

    gl_FragColor.rgb = vec3(drawArea);
    //gl_FragColor.rgb = vec3(hardEdge);

    #ifdef OLDWAY
        // Calculate a fade for sketch lines based on the distance from the center
        // of the screen.
        float centerDistance = length((vec2(0.5) - texCoord) * vec2(0.75, 0.75));
        float radial = clamp(centerDistance, 0.0, 0.5);
        radial = (0.5 - radial) * 2.0;

        //float centerDistance = length((vec2(0.5) - texCoord) * 2.0);
        //// Smaller multipliers make the drawn area bigger, ie: at 1.0
        //// the edges of the screen (and especially the corners) will have no
        //// value at all.
        //float radial = centerDistance * 0.85; //clamp(centerDistance, 0.0, 1.0);
        //radial = radial * radial * radial * radial;
        ////float radial = centerDistance; //clamp(centerDistance, 0.0, 0.5);
        ////radial = (0.5 - radial) * 2.0;
        ////radial = step(0.9, radial);
        ////gl_FragColor.rgb = vec3(1.0 - radial);

        // Similar calc for fading the shading
        float yEdge = (abs(0.5 - texCoord.y)) * 2.0;
        float xEdge = (abs(0.5 - texCoord.x)) * 2.0;
        float edge = (yEdge * yEdge * yEdge * yEdge + xEdge * xEdge * xEdge * xEdge) * 0.5;
        edge = 1.0 - clamp(edge, 0.0, 1.0);

        //edge = edge * edge;

        //edge = step(0.9, edge);
        //gl_FragColor.rgb = vec3(edge);
        //gl_FragColor.rgb = background.rgb;
    #endif

    #ifdef SHADING
        // Use the brightness level of the color to sample the shading
        // textures.
        vec4 sketchOffset1 = (texture2D(m_Noise, texCoord * vec2(2.0, 1.0)) - vec4(0.5)) * 2.0;
        vec4 sketchOffset2 = (texture2D(m_Noise, texCoord * vec2(5.0, 2.5)) - vec4(0.5)) * 2.0;
        float localStrength1 = mix(sketchOffset1.r, sketchOffset1.g, texCoord.x);
        float localStrength2 = mix(sketchOffset2.r, sketchOffset2.g, texCoord.x);
        float localStrength = localStrength1 * 0.75 + localStrength2 * 0.25;

        // Now bring it into a nice mix range for mixing an offset sketch with the
        // original sketch
        localStrength = (localStrength * 0.5) + 0.5;
        localStrength = smoothstep(0.25, 0.75, localStrength);

        //vec2 sketchCoord = texCoord * vec2(8.0, 4.0) + (sketchOffset.xy * 1.00);
        // The problem with adding fractal offsets is that the crosshatching no
        // longer looks natural.  I think we need to instead vary the 'darkness' of
        // it.
        vec2 sketchCoord = texCoord * vec2(8.0, 4.0);
        vec4 sketch = texture2D(m_Sketch, sketchCoord);
        vec4 sketchLight1 = texture2D(m_SketchLight, sketchCoord);
        vec4 sketchLight2 = texture2D(m_SketchLight, sketchCoord * vec2(-1.0, -1.0) + vec2(0.1, -0.25));
        //vec4 sketchLight = mix(sketchLight1, sketchLight2, step(0.5, texCoord.y));
        vec4 sketchLight = mix(sketchLight1, sketchLight2, localStrength);
        //vec4 sketchLight = sketchLight1;
        float colorLevel = rgb2hsv(screenColor.rgb).z;

        colorLevel = pow(colorLevel * 100.0, m_ShadingContrast) / pow(100.0, m_ShadingContrast);

        //colorLevel = 0.0;
        //float colorLevel = max(screenColor.r, max(screenColor.g, screenColor.b));

        // Trying a version that just has background or dark
        float darkness = (1.0 - colorLevel * fractal) * m_ShadingStrength;

        // Find the very darkest areas of the color image
        float color = (sketch.a * sketch.a * darkness) * step(m_CrossHatchThreshold, darkness) * 0.75;
        color += (sketchLight.a * sketchLight.a * darkness) * step(m_CrossHatchThreshold, darkness) * 0.25;
        color += (sketchLight.a * sketchLight.a * darkness) * step(darkness, m_CrossHatchThreshold);

        // Interestingly, if we don't clamp the color then we 'over bright' the paper
        // background... but without that we also lose some definition at the horizon.
        color = clamp(color, -0.1, 1.0);
        //color = clamp(color, 0.0, 1.0);

        // Lower the max value to move the shaded area closer to the edge
        // and with a shorter transition.
        float shadingMix = smoothstep(0.0, 0.25, drawArea);

        #ifdef COLOR_MIX
            float colorMix = m_WaterColorStrength;
            colorMix *= smoothstep(m_SaturationThreshold - m_SaturationMargin, m_SaturationThreshold, screenHsv.y);
            colorMix *= min(m_BrightnessCutoff, smoothstep(m_BrightnessThreshold - m_BrightnessMargin, m_BrightnessThreshold, screenHsv.z));

            //gl_FragColor.rgb = screenColor.rgb * colorMix;
            background.rgb = mix(background.rgb, screenColor.rgb, colorMix * shadingMix);
        #endif

        // Set the base shading
        //gl_FragColor.rgb = mix(background.rgb, vec3(0.0), color * edge);
        //gl_FragColor.rgb = mix(background.rgb, vec3(0.0), color * radial);
        gl_FragColor.rgb = mix(background.rgb, vec3(0.0), color * shadingMix);
        //gl_FragColor.rgb = mix(background.rgb, vec3(0.0), color);
        //gl_FragColor.rgb = mix(vec3(1.0), vec3(0.0), min(1.0, color));

        //gl_FragColor.rgb = mix(vec3(1.0), vec3(0.0), sketchLight.a * 0.9 + localStrength * 0.1);
        //gl_FragColor.rgb = mix(vec3(1.0), vec3(0.0), sketchLight.a);
        //gl_FragColor.rgb = mix(vec3(1.0), vec3(0.0), sketch.a);
        //gl_FragColor.rgb = vec3(localStrength); //mix(vec3(1.0), vec3(0.0), localStrength);
    #else
        gl_FragColor.rgb = background.rgb;
    #endif

    #ifdef LINES
        // Figure out rough sketch lines
        vec4 drawn = texture2D(m_Sketch, texCoord * vec2(10.0, 5.0));
        //float drawAmount = fractal * level * radial; // * drawn.a * drawn.a;
        float drawAmount = fractal * level; // * drawn.a * drawn.a;
        //float drawAmount = fractal * level * edge; // * drawn.a * drawn.a;
        drawAmount *= step(0.1, drawAmount);
        drawAmount = min(0.9, drawAmount);
        //drawAmount *= edge;

        float lineMix = smoothstep(0.0, 0.05, drawArea);
        lineMix = smoothstep(0.0, 0.2, drawArea);

        gl_FragColor.rgb = mix(gl_FragColor.rgb, vec3(1.0 - (drawn.a * drawn.a * drawn.a)), drawAmount * lineMix * m_LineStrength);
    #endif
#endif

    #ifdef COLOR_MIX2
        float colorMix = m_WaterColorStrength;
        colorMix *= smoothstep(m_SaturationThreshold - m_SaturationMargin, m_SaturationThreshold, screenHsv.y);
        colorMix *= smoothstep(m_BrightnessThreshold - m_BrightnessMargin, m_BrightnessThreshold, screenHsv.z);

        //gl_FragColor.rgb = screenColor.rgb * colorMix;
        gl_FragColor.rgb = mix(gl_FragColor.rgb, screenColor.rgb, colorMix);
    #endif

    #ifdef COLOR
        gl_FragColor.rgb = screenColor.rgb;
    #endif

    gl_FragColor.a = 1.0;
}
