/*
 * Copyright (c) 2023, Simsilica, LLC
 * All rights reserved.
 */

import mythruna.fabric.*;
import mythruna.fabric.io.*;
import mythruna.shape.*;

class ClothingLayerSorter implements Comparator {

    public int compare( def item1, def item2 ) {
        if( item1 == item2 ) {
            return 0;
        }
        int layer1 = item1[WornBy]?.layer?:-1
        int layer2 = item2[WornBy]?.layer?:-1
        int result = Integer.compare(layer1, layer2);
        if( result != 0 ) {
            return result;
        }
        return Long.compare(item1.getId(), item2.getId());
    }
}
clothingLayerSorter = new ClothingLayerSorter();


getWornItems = { target ->
    return findEntities(Filters.fieldEquals(WornBy, "target", resolveEntityId(target)));
}

recalculateClothing = { activator ->

    // Make sure the activator actually has a shape
    def playerShape = activator[ShapeInfo];
    def shapeName = ShapeName.parse(playerShape?.shapeName);
    if( shapeName == null ) {
        log.info("player has no shape:" + activator);
        return;
    }

    def wearing = new ArrayList(getWornItems(activator));
    log.info("Need to rebuild clothing for:" + activator + " with:");
    wearing.each {
        log.info("    " + it + " -> " + it[WornBy]);
    }

    // Sort by layer
    Collections.sort(wearing, clothingLayerSorter);

    // Accumulate
    def accumulator = new DefaultClothingAccumulator(32, 32, 32);
    log.info("Sorted:");
    wearing.each {
        def worn = it[WornBy];
        if( worn.layer == -1 ) {
            // Skip WornBy marked for removal
            return;
        }
        log.info("    " + it + " -> " + worn);
        def shape = ShapeName.parse(it[ShapeInfo]?.shapeName);
        log.info("  shape:" + shape);
        def cellArrayId = CellArrayId.fromString(shape.getName());
        log.info("  id:" + cellArrayId);
        def cells = worldManager.getCellArrayStorage().apply(cellArrayId);
        log.info("  cells:" + cells);
        accumulator.addCells(cells);
    }

    // Store and get the ID
    CellArrayId clothingId = worldManager.getCellArrayStorage().store(accumulator.getCells());

    log.info("clothingId:" + clothingId);

    // Set the new shape info on the player
    shapeName.getAddOns().clear();
    shapeName.getAddOns().add(new ShapeName("fab", clothingId.toIdString()));
    log.info("combined shape:" + shapeName);
    def newShape = playerShape.changeShapeName(shapeName.toCompositeString());
    log.info("newShape:" + newShape.toString(entityData));

    activator << newShape;

}


def clothingFormat = new ClothingFileFormat(FabricTypeIndex.getTypes(), SwatchShapeIndex.getShapes());

loadClothingResource = { String resource ->
    if( resource.indexOf("/") < 0 ) {
        resource = "/Models/clothes/" + resource;
    }
    return clothingFormat.readCells(resource);
}

importClothingResource = { String resource ->
    def cells = clothingFormat.readCells(resource);
    CellArrayId id = worldManager.getCellArrayStorage().store(cells);
    return id;
}


createType("ClothingDesign").with {
    supertypes("BaseObject");

    addAction("Wear") { ->
        log.info("wear:" + self.name);

        // See if the activator is already wearing this entity
        if( activator.isSameEntity(self[WornBy]?.target) ) {
            log.info("Already wearing");
        }
        //
        //log.info("Already wearing:" + wearing);
        //def found = wearing.find { it[WornBy].target == activator.id };
        //log.info("found:" + found);
        //if( found ) {
        //    return;
        //}

        def wearing = getWornItems(activator);
        self << new WornBy(activator.id, wearing.size());

        //recalculateClothing(activator);
    }

    addAction("Remove") { ->
        log.info("remove:" + self.name);

        // Make sure it's worn by this particular user
        def worn = self[WornBy];
        def removedLayer = worn.layer;
        if( !activator.isSameEntity(worn.target) ) {
            log.info("Not really wearing it.");
            return;
        }

        // Then remove it
        //self.remove(WornBy);
        // Set it's layer to a bad layer instead of just deleting it...
        // otherwise the appearance system will never know about the change.
        // As long as we set the layer bad once we could later remove the
        // component.
        // By setting the layer to -1 any change listeners will see the change
        // and know that target needs to recalculate clothing.  They already
        // have the info they need so we can then delete it immediately.
        log.info("Changing layer to -1");
        self << worn.changeLayer(-1);
        log.info("Removing WornBy");
        self.remove(WornBy);

        // Catch the case where maybe we crashed sometime between marking a layer
        // and removing it
        if( removedLayer >= 0 ) {
            // Need to move other worn items to fill the gap
            getWornItems(activator).each { clothing ->
                def wb = clothing[WornBy];
                if( wb.layer >= removedLayer ) {
                    clothing << wb.changeLayer(wb.layer - 1);
                }
            }
        }

        //recalculateClothing(activator);
    }
}


