/*
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

findAllItems = { root ->
    log.info("findAllItems(" + root + ")");
    def filter = Filters.fieldEquals(ContainedIn.class, "root", root);
    return findEntities(filter, ContainedIn.class);
}

findItem = { root, name ->
    log.info("findItem(" + root + ", " + name + ")");
    //def filter = Filters.fieldEquals(ContainedIn.class, "root", root);
    //for( EntityId item : findEntities(filter, ContainedIn.class) ) {
    for( EntityId item : findAllItems(root) ) {
        if( log.isTraceEnabled() ) {
            log.trace("checking:" + item);
        }
        // Note: by basing this on the name getter we make this extra fragile
        // in the face of the getter implementation... when we already know that
        // items are only created with ObjectName.  I suspect this is why some of
        // my player inventories have double items: one run with a bad 'name' getter
        // messed them up.  So I'm changing it.
        //String n = item.name;
        String n = item[ObjectName]?.name;
        if( log.isTraceEnabled() ) {
            log.trace("name:" + n);
        }
        if( n == name ) {
            return item;
        }
    }
    return null;
}

findItems = { root, name ->
    log.info("findItems(" + root + ", " + name + ")");
    def results = [];
    for( EntityId item : findAllItems(root) ) {
        if( log.isTraceEnabled() ) {
            log.trace("checking:" + item);
        }
        //String n = item.name;
        String n = item[ObjectName]?.name;
        if( log.isTraceEnabled() ) {
            log.trace("name:" + n);
        }
        if( n == name ) {
            results.add(item);
        }
    }
    return results.sort();
}

updateItem = { item, ContainedIn container ->
    // If the item does not already have the right container component
    // then set it
    if( item[ContainedIn] == null || item[ContainedIn].container == null ) {
        log.info("Fixing missing container for:" + item + " to:" + container);
        item << container;
    }

    // Just in case, update the item properties from the object type
    // in case we've redefined these standard values
    if( item.type.get("mass") ) {
        item << item.type.get("mass");
    }
    if( item.type.get("shape") ) {
        item << item.type.get("shape");
    }
    if( item.type.get("volume") ) {
        item << item.type.get("volume");
        log.info("update volume to:" + item.type.get("volume"));
    }
    if( item.type.get("containerVolume") ) {
        item << item.type.get("containerVolume");
        log.info("update containerVolume to:" + item.type.get("containerVolume"));
    }
}

dumpItemInfo = { entity ->
    log.info("   container:" + entity[ContainedIn]);
    log.info("      volume:" + entity[ObjectVolume]);
    log.info("       shape:" + entity[ShapeInfo]);
    log.info(" contVvolume:" + entity[ContainerVolume]);
    log.info("      attach:" + entity[AttachedTo]);
}

createItem = { name, typeName, ContainedIn container ->
    log.info("createItem(" + name + ", " + typeName + ", " + container + ")");

    def result = findItem(container.root, name);
    if( result != null ) {
        log.info("Found id:" + result);
        dumpItemInfo(result);

        updateItem(result, container);

        return result;
    }

    def type = type(typeName);
    log.info("creating item entity type:" + type + "  info:" + type.info);

    result = type.newInstance(
                ObjectName.create(name),
                type.getTypeVar("mass", new Mass(0.01)),
                type.getTypeVar("shape"),
                type.getTypeVar("volume"),
                new CreatedBy(container.root),
                container
            );

    log.info("Created:" + result + " name:" + result.name + " in:" + container);
    dumpItemInfo(result);

    return result;
}

createContainer = { name, typeName, ContainedIn container ->
    log.info("createContainer(" + name + ", " + typeName + ", " + container + ")");

    def results = findItems(container.root, name);
    if( results ) {
        if( results.size() > 1 ) {
            // Bad things happened and we've given the player more than one container
            log.warn("Found " + results.size() + " " + name + " containers:" + results);

            // For now we will leave the extra one because I feel like it will be
            // useful for testing the inventory UI.  Also, probably the cleanup should
            // be a separate scripted command maybe.
        }
        // Because they were sorted, the first is always the best one
        def result = results[0];
        log.info("existing id:" + result);
        updateItem(result, container);
        if( result[ContainerVolume] == null ) {
            throw new IllegalArgumentException("Item is not a container:" + name + ", " + typeName);
        }
        return result;
    }

    // Before creating the item, make sure that we can find its
    // container volume... else it's not a container
    def type = type(typeName);
    if( type.getTypeVar("containerVolume") == null ) {
        throw new IllegalArgumentException("Type is not a container:" + typeName);
    }

    // Else we can create the item as normal and than make it a container
    def item = createItem(name, typeName, container);

    item << type.getTypeVar("containerVolume");

    return item;
}


// When a player entity is created, we want to add some
// standard kit.
on( playerEntityCreated ) { event ->
    log.info("A player was created:" + event);

    def backpack = createContainer("Backpack", "Backpack", new ContainedIn(event.player, 0, 0));
    log.info("Backpack:" + backpack);
    if( backpack == null ) {
        throw new EventAbortedException("Player has no backpack.");
    }

    // Only one when they are created so they don't get multiple
    // copies if they join after placing but not finishing a dungeon.
    createItem("Dungeon Generator", "DungeonGenerator", new ContainedIn(event.player, backpack, 18, 0));

}

on( playerEntityJoining ) { event ->
    log.info("A player is joining:" + event);

    def backpack = createContainer("Backpack", "Backpack", new ContainedIn(event.player, 0, 0));
    log.info("Backpack:" + backpack);
    if( backpack == null ) {
        throw new EventAbortedException("Player has no backpack.");
    }

    // Attach it no matter what because that's what the old code did.
    // When users can swap things around we probably need to be smarter
    def backpackScale = backpack[ShapeInfo].scale;
    def rot = new Quatd().fromAngles(0, 0, 0);
    backpack << AttachedTo.create(event.player, "back", new Vec3d(-4, -5, 0).multLocal(backpackScale), rot);

    //def pouch = createContainer("Pouch", "Pouch", new ContainedIn(event.player, 1, 0));
    //if( pouch != null ) {
    //    def pouchScale = pouch[ShapeInfo].scale;
    //    rot = new Quatd().fromAngles(0, 0, 0);
    //    def offset = pouch[ShapeInfo].loadCells().size.toVec3d().mult(-0.5);
    //    //offset.z = 0;
    //    pouch << AttachedTo.create(event.player, "hip.right.01", offset.mult(pouchScale), rot);
    //}
    //
    //pouch = createContainer("Pouch2", "Pouch", new ContainedIn(event.player, 1, 0));
    //if( pouch != null ) {
    //    pouchScale = pouch[ShapeInfo].scale;
    //    rot = new Quatd().fromAngles(0, 0, 0);
    //    def offset = pouch[ShapeInfo].loadCells().size.toVec3d().mult(-0.5);
    //    //offset.z = 0;
    //    pouch << AttachedTo.create(event.player, "hip.right.02", offset.mult(pouchScale), rot);
    //}

    //// Make sure the player has some standard items
    def wand = createItem("Build Wand", "BuildWand", new ContainedIn(event.player, backpack, 0, 0));
    //createItem("Build Wand 2", "BuildWand2", new ContainedIn(event.player, backpack, 0, 4));
    createItem("Sword", "Sword", new ContainedIn(event.player, backpack, 2, 0));
    createItem("Axe", "Axe", new ContainedIn(event.player, backpack, 6, 0));
    createItem("Book", "Book", new ContainedIn(event.player, backpack, 10, 0));
    createItem("Object Tool", "ObjectTool", new ContainedIn(event.player, backpack, 16, 0));
    createItem("Test Wand", "TestWand", new ContainedIn(event.player, backpack, 17, 0));
    //createItem("Gourd Gun", "GourdGun", new ContainedIn(event.player, backpack, 16, 4));
    createItem("Pick-axe", "PickAxe", new ContainedIn(event.player, backpack, 10, 7));

    // Check and fix all contained "real" items to make sure that they don't have spawn positions
    // ...this is to catch items that would try to be managed by the physics engine and managed
    // by the inventory/attachment system at the same time.  The world startup will fix this but
    // just in case it happens during game, the player can at least log out and in to fix it
    // without a server restart.
    def filter = Filters.fieldEquals(ContainedIn.class, "root", event.player);
    findEntities(filter, ContainedIn.class, SpawnPosition.class, ShapeInfo.class, Mass.class).each { item ->
        fixAmbiguousPhysicality(item);
    }

    //createItem("Extractor Wand", "Extractor", new ContainedIn(event.player, backpack, 20, 7));

    // Only on create
    //createItem("Dungeon Generator", "DungeonGenerator", new ContainedIn(event.player, backpack, 18, 0));

    //createItem("Drop Testr", "DropTest", new ContainedIn(event.player, backpack, 18, 4));

    // If we aren't holding anything then default to holding the
    // build wand
    if( event.player[Holding] == null ) {
        log.info("Default holding build wand:" + wand);
        // This has an added benefit in that it will be executed on the next
        // update because GameSession.runAction() is normally called from a network
        // thread.  This is good because in the event handler our full environment might
        // not be 100% setup yet.  But it's important to remember that something checking
        // for 'holding' right after this will likely still see null.
        event.session.runAction(wand, "Equip");
    }
}

// Note: in single player games, we won't necessarily see this
// event.
on( playerEntityLeft ) { event ->
    log.info("A player has left:" + event);
}


